package com.digiwin.athena.agiledataecho.dto.mongodb.metric;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import java.io.Serializable;
import java.util.*;

/**
 * @Author: SunHong
 * @Date: 2024/11/11 9:28
 * @Description:
 */

@Data
public class MetricCheckResDTO implements Serializable {

    private List<Tabs> tabs;
    @Data
    public static class Tabs {

        private String tabTile;

        private int tabId;

        private boolean hasTreeList;

        private List<AreaInfo> areas;

        private List<Map<String,Object>> selectList;
    }

        @Data
        public static class AreaInfo {

            private int type;

            private String title;

            private String display;

            private boolean isCollapse;

            private List<ValueInfo> values;

            private Map<String,Object> selectValue;

            /**
             * 组装语义解释(来源应用)
             * @param appName 应用
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleAppName(String appName,Map<String,String> multilingualMap) {
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("appName"));
                areaInfo.setDisplay("line");
                areaInfo.setCollapse(false);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo applicationName = new AreaInfo.ValueInfo();
                applicationName.setCopy(true);
                applicationName.setValueType("text");
                applicationName.setValue(appName);
                values.add(applicationName);
                areaInfo.setValues(values);
                return  areaInfo;
            }

            /**
             * 组装语义解释(来源应用)
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleAppCode(String appCode,Map<String,String> multilingualMap) {
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("appCode"));
                areaInfo.setDisplay("line");
                areaInfo.setCollapse(false);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo applicationCode = new AreaInfo.ValueInfo();
                applicationCode.setCopy(true);
                applicationCode.setValueType("text");
                applicationCode.setValue(appCode);
                values.add(applicationCode);
                areaInfo.setValues(values);
                return  areaInfo;
            }

            public static AreaInfo assembleSameWord(Map<String,Object> debugInfoMap,Map<String,String> multilingualMap) {
                String sameWord;
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("sameWord"));
                areaInfo.setDisplay("row");
                areaInfo.setCollapse(true);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo logic = new AreaInfo.ValueInfo();
                logic.setCopy(true);
                logic.setValueType("text");
                values.add(logic);
                areaInfo.setValues(values);
                if(MapUtils.isEmpty(debugInfoMap)){
                    return areaInfo;
                }
                //获取同义词信息
                List<Map<String,Object>> sameWordList = (List<Map<String, Object>>) debugInfoMap.get("synonymMatching");
                if(CollectionUtils.isEmpty(sameWordList)){
                    return areaInfo;
                }
                StringBuilder builder = new StringBuilder();
                sameWordList.stream().forEach(item-> {
                    String matchWord = MapUtils.getString(item,"matchWord");
                    String standardWord = MapUtils.getString(item,"standardWord");
                    String entityType = MapUtils.getString(item,"entityType");
                    String matchTip = MapUtils.getString(multilingualMap,"matchTip");
                    String entityTip = MapUtils.getString(multilingualMap,"entityTip");
                    String description = MapUtils.getString(multilingualMap,"description");
                    if(StringUtils.isNotEmpty(matchWord)){
                        builder.append("【");
                        builder.append(matchWord);
                        builder.append("】");
                        builder.append("-");
                        builder.append(matchTip);
                        builder.append("->");
                        builder.append("【");
                        builder.append(standardWord);
                        builder.append("】");
                        if(StringUtils.isNotEmpty(entityType)) {
                            builder.append("(");
                            builder.append(entityTip);
                            builder.append(":");
                            builder.append(entityType);
                            builder.append(")");
                        }
                    } else {
                        builder.append("【");
                        builder.append(standardWord);
                        builder.append("】");
                        builder.append("(");
                        builder.append(description);
                        builder.append(")");

                    }
                    builder.append("\n");
                });
                sameWord = builder.toString();
                sameWord = StringUtils.removeEnd(sameWord, "\n");

                logic.setValue(sameWord);

                return  areaInfo;
            }

            /**
             * 组装语义解释(取值逻辑) - 代码格式
             * @param semantic ade 返回取值逻辑
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleValueLogic(Map<String,Object> semantic,Map<String,String> multilingualMap) {
                String semanticSqlDescription = "";
                if(!semantic.isEmpty()){
                    semanticSqlDescription = (String) semantic.get("semanticSqlDescription");
                }
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("valueLogic"));
                areaInfo.setDisplay("row");
                areaInfo.setCollapse(true);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo logic = new AreaInfo.ValueInfo();
                logic.setCopy(true);
                logic.setValueType("text");
                logic.setValue(semanticSqlDescription);
                values.add(logic);
                areaInfo.setValues(values);
                return  areaInfo;
            }

            /**
             * 组装语义解释 sql 模块
             * @param semantic ade 返回sql
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleSql(Map<String,Object> semantic,Map<String,String> multilingualMap) {
                String semanticSql = "";
                if(!semantic.isEmpty()){
                    semanticSql = (String) semantic.get("semanticSql");
                }
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("sql"));
                areaInfo.setDisplay("row");
                areaInfo.setCollapse(true);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo sql = new AreaInfo.ValueInfo();
                sql.setCopy(true);
                sql.setValueType("text");
                sql.setValue(semanticSql);
                values.add(sql);
                areaInfo.setValues(values);
                return  areaInfo;
            }

            /**
             * 组装运算流程模块
             * @param selectValue 运算value
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleProcessFlow(Map<String,Object> selectValue,Map<String,String> multilingualMap) {
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("valueLogic"));
                areaInfo.setDisplay("flow");
                areaInfo.setCollapse(true);
                areaInfo.setSelectValue(selectValue);
                return areaInfo;
            }

            /**
             * 组装取值SQL 模块
             * @param selectValue 运算value
             * @param multilingualMap 多语言字段集合
             * @return AreaInfo
             */
            public static AreaInfo assembleGetSQL(Map<String,Object> selectValue,Map<String,String> multilingualMap) {

                Map<String, Object> newMap = new HashMap<>();
                // 重新组装集合,只获取存在query集合的map 并且根据query list 生成数组 重新生成map
                for (Map.Entry<String, Object> entry : selectValue.entrySet()) {
                    String key = entry.getKey();
                    Object value = entry.getValue();

                    if (value instanceof List) {
                        List<Map<String, Object>> originalList = (List<Map<String, Object>>) value;
                        List<Map<String, Object>> newList = new ArrayList<>();

                        for (Map<String, Object> innerMap : originalList) {
                            if (innerMap.containsKey("query")) {
                                List<String> queryList = (List<String>) innerMap.get("query");
                                for (String query : queryList) {
                                    Object actionName = innerMap.get("actionName");
                                    if(Objects.isNull(actionName) || "null".equals(String.valueOf(actionName))){
                                        actionName = "";
                                    }
                                    Object actionId = innerMap.get("actionId");
                                    if(Objects.isNull(actionId)|| "null".equals(String.valueOf(actionId))){
                                        actionId = "";
                                    }
                                    Map<String, Object> newInnerMap = new HashMap<>();
                                    newInnerMap.put("title", actionName + " (" + actionId +")");
                                    newInnerMap.put("value", query);
                                    newInnerMap.put("copy", true);
                                    newList.add(newInnerMap);
                                }
                            }
                        }
                        if (!newList.isEmpty()) {
                            newMap.put(key, newList);
                        }
                    }
                }

                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(1);
                areaInfo.setTitle("");
                areaInfo.setDisplay("row");
                areaInfo.setCollapse(true);
                areaInfo.setSelectValue(newMap);
                return areaInfo;
            }

            public static AreaInfo assembleIdea(Map<String, Object> semanticData, Map<String, String> multilingualMap) {
                String ideaStr = "";
                if(!semanticData.isEmpty()){
                    ideaStr = (String) semanticData.get("explain4Gpt");
                }
                AreaInfo areaInfo = new AreaInfo();
                areaInfo.setType(0);
                areaInfo.setTitle(multilingualMap.get("idea"));
                areaInfo.setDisplay("row");
                areaInfo.setCollapse(true);

                List<AreaInfo.ValueInfo> values = new ArrayList<>();
                AreaInfo.ValueInfo idea = new AreaInfo.ValueInfo();
                idea.setCopy(true);
                idea.setValueType("text");
                idea.setValue(ideaStr);
                values.add(idea);
                areaInfo.setValues(values);
                return areaInfo;
            }

            @Data
            public static class ValueInfo {

                private String value;

                private String valueType;

                private boolean isCopy;

            }
    }



    /**
     * 来源应用/取值逻辑/取值逻辑SQl
     * @return 组装后集合
     */
    public static Tabs assembleSchemaExplain(Map<String, Object> request,Map<String,String> multilingualMap) {

        String appCode = (String) request.get("appCode");
        String appName = (String) request.get("appName");
        String metricId = (String) request.get("metricId");
        String schema = multilingualMap.get("schema");
        Object semantic = request.get("semantic");
        Object debugInfo = request.get("debugInfo");
        // metric debug 语义解释数据
        Map<String,Object> semanticData = Objects.nonNull(request.get("semantic")) ?
                (Map<String,Object>) semantic : new HashMap<>();
        Map<String,Object> debugInfoData = Objects.nonNull(request.get("debugInfo")) ?
                (Map<String,Object>) debugInfo : new HashMap<>();
        Tabs schemaExplain = new Tabs();
        schemaExplain.setTabId(1);
        schemaExplain.setTabTile(schema);
        // 应用名称
        AreaInfo name = AreaInfo.assembleAppName(appName,multilingualMap);
        // 应用编码
        AreaInfo code = AreaInfo.assembleAppCode(appCode,multilingualMap);
        // 同义词
        AreaInfo sameWord = AreaInfo.assembleSameWord(debugInfoData,multilingualMap);
        // 解题思路
        AreaInfo idea = AreaInfo.assembleIdea(semanticData,multilingualMap);
        // 要素识别
        AreaInfo valueLogic = AreaInfo.assembleValueLogic(semanticData,multilingualMap);
        // 取值逻辑sql
        AreaInfo sql = AreaInfo.assembleSql(semanticData,multilingualMap);

        List<AreaInfo> areaInfos = new ArrayList<>();
        areaInfos.add(name);
        areaInfos.add(code);
        areaInfos.add(sameWord);
        areaInfos.add(idea);
        areaInfos.add(valueLogic);
        areaInfos.add(sql);
        schemaExplain.setAreas(areaInfos);
        return schemaExplain;
    }

    /**
     * 取值逻辑
     * @return 组装后集合
     */
    public static Tabs assembleProcessFlow(List<Map<String,Object>> selectList,Map<String,Object> selectValue,Map<String,String> multilingualMap) {
        Tabs processFlow = new Tabs();
        processFlow.setTabId(2);
        processFlow.setTabTile(multilingualMap.get("processFlow"));
        processFlow.setHasTreeList(true);
        processFlow.setSelectList(selectList);
        AreaInfo processFlowData = AreaInfo.assembleProcessFlow(selectValue,multilingualMap);
        List<AreaInfo> areaInfos = new ArrayList<>();
        if(!processFlowData.selectValue.isEmpty()){
            areaInfos.add(processFlowData);
        }
        processFlow.setAreas(areaInfos);
        return processFlow;

    }

    /**
     * 取值SQL模块
     * @param selectList 值
     * @param selectValue 取数
     * @param multilingualMap 多语言字段集合
     * @return 前端tab
     */
    public static Tabs assembleGetSQL(List<Map<String,Object>> selectList,Map<String,Object> selectValue,Map<String,String> multilingualMap) {
        Tabs sql = new Tabs();
        sql.setTabId(3);
        sql.setTabTile(multilingualMap.get("getSql"));
        sql.setHasTreeList(true);
        sql.setSelectList(selectList);
        AreaInfo sqlData = AreaInfo.assembleGetSQL(selectValue,multilingualMap);
        List<AreaInfo> areaInfos = new ArrayList<>();
        if(!sqlData.selectValue.isEmpty()){
            areaInfos.add(sqlData);
        }
        sql.setAreas(areaInfos);
        return sql;
    }


    /**
     * 替换公共元素key
     * @param selectList 入参
     * @return 公共元素
     */
    public static List<Map<String, Object>> replaceMetricNamesAndIds(List<Map<String, Object>> selectList) {
        List<Map<String, Object>> newSelectList = new ArrayList<>();
        if (CollectionUtils.isEmpty(selectList)) {
            return newSelectList;
        }

        for (Map<String, Object> item : selectList) {
            Map<String, Object> newItem = new HashMap<>();

            // 先将原始的metricName等其他键值对复制过来
            for (Map.Entry<String, Object> entry : item.entrySet()) {
                if (!"metricId".equals(entry.getKey()) &&!"uuid".equals(entry.getKey())) {
                    newItem.put(entry.getKey(), entry.getValue());
                }
            }

            // 进行metricId替换为title和uuid替换为key的操作
            if (item.containsKey("metricName")) {
                newItem.put("title", String.valueOf(item.get("metricName")));
            }
            if (item.containsKey("uuid")) {
                newItem.put("key", item.get("uuid"));
            }

            newSelectList.add(newItem);

            // 处理子元素
            if (item.containsKey("children")) {
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> childrenList = (List<Map<String, Object>>) item.get("children");
                if (childrenList!= null) {
                    List<Map<String, Object>> newChildrenList = replaceMetricNamesAndIds(childrenList);
                    newItem.put("children", newChildrenList);
                }
            }
        }

        return newSelectList;
    }

}



