package com.digiwin.athena.agiledataecho.service.imp;


import cn.hutool.core.util.ObjUtil;
import com.digiwin.athena.agiledataecho.dto.EchoExperienceRoleConfigDataQueryDTO;
import com.digiwin.athena.agiledataecho.dto.EchoOpeningRemarksConfigResultDTO;
import com.digiwin.athena.agiledataecho.dto.EchoPageConfigProductBulletinQueryDTO;
import com.digiwin.athena.agiledataecho.dto.EchoPageConfigProductBulletinResultDTO;
import com.digiwin.athena.agiledataecho.mongodb.domain.*;
import com.digiwin.athena.agiledataecho.proxy.iam.UserService;
import com.digiwin.athena.agiledataecho.proxy.knowledgemaps.KnowledgeMapsService;
import com.digiwin.athena.agiledataecho.proxy.knowledgemaps.model.KMCommandIntentionsDTO;
import com.digiwin.athena.agiledataecho.service.EchoPageConfigService;
import com.digiwin.athena.agiledataecho.util.SystemLanguageUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

/**
 * GMC接口实现
 *
 * @author yanfeng
 * @date 2024/7/23
 */
@Slf4j
@Service
public class EchoPageConfigServiceImpl implements EchoPageConfigService {

    @Autowired
    @Qualifier("mongoTemplate")
    private MongoTemplate mongoTemplate;

    @Autowired
    private KnowledgeMapsService knowledgeMapsService;

    @Autowired
    private UserService userService;

    @Override
    public List<EchoPageConfigProductBulletinResultDTO> queryProductBulletinList(AuthoredUser user, EchoPageConfigProductBulletinQueryDTO queryDTO) {
        Query query = new Query();
        if (StringUtils.isNotBlank(queryDTO.getSource())) {
            query.addCriteria(Criteria.where("source").regex(queryDTO.getSource(), "i"));
        }

        List<EchoProductBulletinConfigData> echoProductBulletinConfigData = mongoTemplate.find(query, EchoProductBulletinConfigData.class);
        if (CollectionUtils.isEmpty(echoProductBulletinConfigData)) {
            log.warn("未查询到产品公告列表");
            return Collections.emptyList();
        }
        List<EchoPageConfigProductBulletinResultDTO> list = new ArrayList<>();
        echoProductBulletinConfigData.forEach(data -> {
            EchoPageConfigProductBulletinResultDTO gmcProductBulletinResultDTO = new EchoPageConfigProductBulletinResultDTO();
            gmcProductBulletinResultDTO.setId(data.getId());
            gmcProductBulletinResultDTO.setTitle(data.getTitle());
            gmcProductBulletinResultDTO.setContent(data.getContent());
            gmcProductBulletinResultDTO.setCoverUrl(data.getCoverUrl());
            gmcProductBulletinResultDTO.setDocumentList(data.getDocumentList());
            list.add(gmcProductBulletinResultDTO);
        });
        String locale = queryUserLocale(user);
        String traditional;
        if (Locale.TAIWAN.toString().equals(locale)) {
            traditional = SystemLanguageUtils.toTraditional(JsonUtils.objectToString(list));
        } else {
            traditional = SystemLanguageUtils.toSimple(JsonUtils.objectToString(list));
        }

        List<EchoPageConfigProductBulletinResultDTO> echoPageConfigProductBulletinResultDTOS = JsonUtils.jsonToObject(traditional, new TypeReference<List<EchoPageConfigProductBulletinResultDTO>>() {
        });
        //平台翻译接口会把数据简体情况下翻译为资料，所以有此行代码产生,只用前端的
        echoPageConfigProductBulletinResultDTOS.forEach(data -> data.setContent(data.getContent().replace("{assistantName}", queryDTO.getAssistantName())));

        return echoPageConfigProductBulletinResultDTOS;
    }

    @Override
    public EchoProductBulletinConfigData saveProductBulletinConfig(AuthoredUser user, EchoProductBulletinConfigData saveDTO) {
        try {
            if (StringUtils.isNotBlank(saveDTO.getId())) {
                Query query = new Query(Criteria.where("_id").is(new ObjectId(saveDTO.getId())));
                mongoTemplate.updateFirst(query, Update.fromDocument(Document.parse(JsonUtils.objectToString(saveDTO))), EchoProductBulletinConfigData.class);
                return mongoTemplate.findById(saveDTO.getId(), EchoProductBulletinConfigData.class);
            } else {
                return mongoTemplate.insert(saveDTO);
            }
        } catch (Exception e) {
            log.error("保存或更新产品公告配置失败: {}", e.getMessage(), e);
            return null;
        }
    }

    @Override
    public EchoOpeningRemarksConfigData saveOpeningRemarksConfig(AuthoredUser user, EchoOpeningRemarksConfigData saveDTO) {
        try {
            if (StringUtils.isNotBlank(saveDTO.getId())) {
                Query query = new Query(Criteria.where("_id").is(new ObjectId(saveDTO.getId())));
                mongoTemplate.updateFirst(query, Update.fromDocument(Document.parse(JsonUtils.objectToString(saveDTO))), EchoOpeningRemarksConfigData.class);
                return mongoTemplate.findById(saveDTO.getId(), EchoOpeningRemarksConfigData.class);
            } else {
                return mongoTemplate.insert(saveDTO);
            }
        } catch (Exception e) {
            log.error("保存或更新开场白配置失败: {}", e.getMessage(), e);
            return null;
        }
    }

    @Override
    public EchoOpeningRemarksConfigResultDTO queryOpeningRemarksConfig(AuthoredUser user, EchoPageConfigProductBulletinQueryDTO queryDTO) {
        Query query = new Query();
        if (StringUtils.isNotBlank(queryDTO.getSource())) {
            query.addCriteria(Criteria.where("source").regex(queryDTO.getSource(), "i"));
        }
        EchoOpeningRemarksConfigData configData = mongoTemplate.findOne(query, EchoOpeningRemarksConfigData.class);
        if (null == configData) {
            return null;
        }
        String token = user.getToken();
        String tenantId = user.getTenantId();
        String userId = user.getUserId();
        // 获取用户语言
        String locale = queryUserLocale(user);
        EchoOpeningRemarksConfigResultDTO resultDTO = new EchoOpeningRemarksConfigResultDTO();
        resultDTO.setTitle(configData.getTitle());
        resultDTO.setContent(configData.getContent());
        resultDTO.setCoverUrl(configData.getCoverUrl());

        List<KMCommandIntentionsDTO> kmCommandIntentions =
                knowledgeMapsService.queryCommandIntentions(token, userId, tenantId, locale);
        if (CollectionUtils.isNotEmpty(kmCommandIntentions)) {
            kmCommandIntentions.removeIf(intention -> CollectionUtils.isEmpty(intention.getIntentions()));
            List<KMCommandIntentionsDTO> firstKmIntentions = new ArrayList<>();
            if (CollectionUtils.isNotEmpty(kmCommandIntentions)) {
                String appCode = kmCommandIntentions.get(0).getApplicationCode();
                for (KMCommandIntentionsDTO intention : kmCommandIntentions) {
                    if (StringUtils.equals(intention.getApplicationCode(), appCode)) {
                        firstKmIntentions.add(intention);
                    }
                }
            }
            // 获取前4个元素
            int endIndex = Math.min(4, firstKmIntentions.size());
            firstKmIntentions = firstKmIntentions.subList(0, endIndex);
            if (!CollectionUtils.isEmpty(firstKmIntentions)) {
                List<EchoOpeningRemarksConfigResultDTO.Document> documentList = new ArrayList<>();
                for (KMCommandIntentionsDTO model : firstKmIntentions) {
                    if (CollectionUtils.isEmpty(model.getIntentions())) {
                        continue;
                    }
                    EchoOpeningRemarksConfigResultDTO.Document document = new EchoOpeningRemarksConfigResultDTO.Document();

                    document.setIndex(model.getIntentionGroupName());
                    if(CollectionUtils.isNotEmpty(model.getIntentions())){
                        document.setIntention(model.getIntentions().get(0).getIntention());
                    }
                    documentList.add(document);
                }
                resultDTO.setDocumentList(documentList);
            }
        }
        String traditional;
        if (Locale.TAIWAN.toString().equals(locale)) {
            traditional = SystemLanguageUtils.toTraditional(JsonUtils.objectToString(resultDTO));
        } else {
            traditional = SystemLanguageUtils.toSimple(JsonUtils.objectToString(resultDTO));
        }
        EchoOpeningRemarksConfigResultDTO echoOpeningRemarksConfigResultDTO = JsonUtils.jsonToObject(traditional, new TypeReference<EchoOpeningRemarksConfigResultDTO>() {
        });
        //平台翻译接口会把数据简体情况下翻译为资料，所以有此代码产生,只用前端的
        echoOpeningRemarksConfigResultDTO.setTitle(echoOpeningRemarksConfigResultDTO.getTitle().replace("{assistantName}", queryDTO.getAssistantName()));
        echoOpeningRemarksConfigResultDTO.setContent(echoOpeningRemarksConfigResultDTO.getContent().replace("{assistantName}", queryDTO.getAssistantName()));
        return echoOpeningRemarksConfigResultDTO;
    }

    @Override
    public List<EchoExperienceRoleConfigData> queryExperienceRoleConfigList(EchoExperienceRoleConfigDataQueryDTO queryDTO) {
        Query query = new Query();
        query.addCriteria(Criteria.where("deleted").is(false));
        List<EchoExperienceRoleConfigData> echoExperienceRoleConfigData = mongoTemplate.find(query, EchoExperienceRoleConfigData.class);
        echoExperienceRoleConfigData.forEach(data -> data.setRoleName(queryDTO.getDistrict(), queryDTO.getLocale()));
        return echoExperienceRoleConfigData;
    }

    @Override
    public EchoLiteUploadLimitationsConfigData queryLiteUploadLimitationsConfig(AuthoredUser user) {
        Query query = new Query();
        query.addCriteria(Criteria.where("deleted").is(false));
        EchoLiteUploadLimitationsConfigData one = mongoTemplate.findOne(query, EchoLiteUploadLimitationsConfigData.class);
        if (one != null) {
            one.setHtmlString(queryUserLocale(user));
        }
        return one;
    }

    @Override
    public List<KMCommandIntentionsDTO> queryCommandIntentions(AuthoredUser user) {
        // 获取用户语言
        String locale = queryUserLocale(user);
        return knowledgeMapsService.queryCommandIntentions(user.getToken(), user.getUserId(), user.getTenantId(), locale);
    }

    @Override
    public EchoFuzzyMatchingConfigData queryFuzzyMatchingConfig(AuthoredUser user) {
        EchoFuzzyMatchingConfigData one = mongoTemplate.findOne(new Query(), EchoFuzzyMatchingConfigData.class);
        if (ObjUtil.isEmpty(one)) {
            EchoFuzzyMatchingConfigData data = new EchoFuzzyMatchingConfigData();
            data.setFuzzyMatching(true);
            data.setPrecision(0.9);
            return data;
        }
        return one;
    }

    private String queryUserLocale(AuthoredUser user) {
        String token = user.getToken();
        String userId = user.getUserId();
        String tenantId = user.getTenantId();
        // 获取用户语言
        return userService.getUserLangNameByUserId(userId, tenantId, token);
    }

}