package com.digiwin.athena.show.service;

import cn.hutool.core.bean.BeanUtil;
import com.alibaba.fastjson.JSON;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.show.domain.BuildContext;
import com.digiwin.athena.show.domain.agileDataDTO.AgileReportECharts;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertAgileDataUtils;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertGetAgileData;
import com.digiwin.athena.show.domain.showDefine.ThemeMapReport;
import com.digiwin.athena.show.infrastructure.meta.ECHOConstants;
import com.digiwin.athena.show.metadata.MetadataField;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import digiwin.chartsdk.beans.sdk.chart.*;
import digiwin.chartsdk.enums.FieldTypeEnum;
import digiwin.chartsdk.utils.ChartUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * echarts展示
 */
@Service
@Slf4j
public class ChartBuildBase {

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private AgileDataCompensateDataService agileDataCompensateDataService;

    @Autowired
    private AgileReportRuleService agileReportRuleService;


    public Option buildChart(MetadataField metadataField, BuildContext buildContext){
        //构建入参
        ChartBase chartBase = this.buildChartParams(buildContext);
        //绑定规则
        bindRules(chartBase,buildContext.getThemeMapReport());
        //图形构建
        Option option = ChartUtil.getOption(chartBase);
        //设置标题
        this.setTitle(buildContext.getThemeMapReport(),option);
        //个性化设置
        transShow(metadataField,chartBase,option,buildContext);
        return option;
    }

    /**
     * 绑定规则
     * @param chartBase
     */
    public void bindRules(ChartBase chartBase,ThemeMapReport themeMapReport){
        agileReportRuleService.buildChartDataRule(chartBase,themeMapReport);
    }

    /**
     * 构建chartBI入参
     * @param buildContext
     * @return
     */
    public ChartBase buildChartParams(BuildContext buildContext){
        ChartBase chartBase = new ChartBase();
        ThemeMapReport themeMapReport = buildContext.getThemeMapReport();

        AgileReportECharts agileReportECharts = themeMapReport.getECharts();
        if(agileReportECharts == null || CollectionUtils.isEmpty((themeMapReport.getPageData()))){
            return null;
        }
        List<MetadataField> valueMetadataFields = Lists.newArrayList();
        List<ChartBaseSeries> chartBaseSeriesList = Lists.newArrayList();
        chartBase.setChartBaseSeries(chartBaseSeriesList);
        //绑定图形数据
        for(AgileReportECharts.EChartsData eChartsData : agileReportECharts.getData()) {
            ChartBaseSeries chartBaseSeries = new ChartBaseSeries();
            //ponits节点数据
            List<Point> points = Lists.newArrayList();
            List<MetadataField> pointFields = this.getChartMetadataField(themeMapReport.getDataSource(), eChartsData.getPoints(), buildContext);
            pointFields.stream().forEach(pointField -> {
                Point point = new Point();
                point.setName(pointField.getName());
                point.setDataType(FieldTypeEnum.getFieldTypeEnumByCode(pointField.getDataType()));
                point.setTitle(pointField.getDescription());
                points.add(point);
            });
            //values节点数据
            List<Value> values = Lists.newArrayList();
            List<MetadataField> valueFields = this.getChartMetadataField(themeMapReport.getDataSource(), eChartsData.getValues(), buildContext);
            valueMetadataFields.addAll(valueFields);
            valueFields.stream().forEach(valueField -> {
                Value value = new Value();
                value.setName(valueField.getName());
                String dataType = valueField.getDataType();
                if (StringUtils.equals("numeric", dataType)) {
                    dataType = "number";
                }
                value.setDataType(FieldTypeEnum.getFieldTypeEnumByCode(dataType));
                value.setTitle(valueField.getDescription());
                value.setType(this.getDoubleChartType(eChartsData,valueField.getName(),agileReportECharts));
                value.setDecimal(valueField.getDecimal());
                value.setPercent(valueField.getPercent());
                //打上解释标记
                if(StringUtils.isNotEmpty(valueField.getExplanation())){
                    StringBuilder descriptionBuilder = new StringBuilder();
                    descriptionBuilder.append(messageUtils.getMessage("agileData.description"));
                    descriptionBuilder.append(valueField.getExplanation());
                    value.setDescription(descriptionBuilder.toString());
                }
                if(StringUtils.isNotEmpty(valueField.getBusinessType())){
                    Map<String,Object> extendInfo = Maps.newHashMap();
                    extendInfo.put("businessType",valueField.getBusinessType());
                    value.setExtendInfo(extendInfo);
                }
                if(StringUtils.isNotEmpty(valueField.getDecimalRule())){
                    value.setDecimalRule(BeanUtil.beanToMap(ConvertAgileDataUtils.getDecimalRule(valueField.getDecimalRule())));
                }
                if(StringUtils.isNotEmpty(valueField.getUnit())){
                    value.setUnit(BeanUtil.beanToMap(ConvertAgileDataUtils.getUnitRule(valueField.getUnit())));
                }
                values.add(value);
            });

            chartBaseSeries.setPoints(points);
            chartBaseSeries.setValues(values);
            chartBaseSeries.setType(agileReportECharts.getChartType());
            chartBaseSeriesList.add(chartBaseSeries);
        }

        //业务数据
        List<Map<String,Object>> datas = themeMapReport.getPageData();
        //去除无数据的字段
        List<String> noValueFields = Lists.newArrayList();
        Set<String> hasValueList = new HashSet<>();
        for (Iterator<Map<String,Object>> iterator = datas.iterator(); iterator.hasNext();){
            Map<String,Object> dataMap = iterator.next();
            List<MetadataField> hasValues = valueMetadataFields.stream().filter(x -> dataMap.containsKey(x.getName())).collect(Collectors.toList());
            if (hasValues != null){
                hasValues.stream().forEach(value -> {
                    //必须所有数据为空，才可判定为无数据字段
                    if(MapUtils.getObject(dataMap,value.getName()) == null && !noValueFields.contains(value.getName())
                        && !hasValueList.contains(value.getName())){
                        noValueFields.add(value.getName());
                    } else {
                        hasValueList.add(value.getName());
                        noValueFields.remove(value.getName());
                    }
                });
            }
        }
        if(CollectionUtils.isNotEmpty(noValueFields)){
            datas.forEach(data -> noValueFields.forEach(data::remove));
        }
        if(CollectionUtils.isEmpty(chartBaseSeriesList)){
            chartBase.setDatas(datas);
            return chartBase;
        }
        List<Map<String, Object>> compensationData = new ArrayList<>();
        if (this.needCompensationData(agileReportECharts, datas, chartBaseSeriesList)) {
            String xKey = chartBaseSeriesList.get(0).getPoints().get(0).getName();
            boolean anyDateTypeNull = datas.stream().anyMatch(data -> Objects.isNull(data.get(xKey))  || StringUtils.isEmpty(String.valueOf(data.get(xKey))));
            if (anyDateTypeNull) {
                log.warn("buildChartParams dateType key is null datas:{}", JSON.toJSONString(datas));
            } else {
                compensationData = agileDataCompensateDataService.compensateChartLineAgileData(chartBaseSeriesList, buildContext,datas, themeMapReport, xKey);
            }
        }
        chartBase.setDatas(compensationData.isEmpty() ? datas : compensationData);
        themeMapReport.setPageData(chartBase.getDatas());
        return chartBase;
    }

    private boolean needCompensationData(AgileReportECharts agileReportECharts,List<Map<String,Object>> datas,List<ChartBaseSeries> chartBaseSeriesList){
        if(CollectionUtils.isEmpty(datas)){
            return false;
        }
        if (ECHOConstants.LINE.equals(agileReportECharts.getChartType())) {
            return true;
        }
        //多轴图，按x轴类型是否为如期类型进行判定
        if ("mixed".equals(agileReportECharts.getChartType())) {
            List<Point> points = chartBaseSeriesList.get(0).getPoints();
            if(CollectionUtils.isNotEmpty(points)){
                Point fieldPoint = points.stream().filter(point -> FieldTypeEnum.CHART_TYPE_ENUM_DATE.equals(point.getDataType()) || FieldTypeEnum.CHART_TYPE_ENUM_DATETIME.equals(point.getDataType())).findFirst().orElse(null);
                if(fieldPoint != null){
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 获取双轴图字段呈现类型
     * @param eChartsData
     * @param field
     * @return
     */
    private String getDoubleChartType(AgileReportECharts.EChartsData eChartsData,String field,AgileReportECharts agileReportECharts){
        String chartType = "";
        Optional<AgileReportECharts.EChartsDataDefine> eChartsDataDefineOperationDTO = eChartsData.getValues().stream().filter(eChartsDataDefine -> StringUtils.equals(field,eChartsDataDefine.getApplyToField())).findFirst();
        if(eChartsDataDefineOperationDTO.isPresent()){
            chartType = eChartsDataDefineOperationDTO.get().getChart();
        }
        if(StringUtils.isEmpty(chartType)){
            chartType = agileReportECharts.getChartType();
        }
        return chartType;
    }

    /**
     * 设置标题
     * @param themeMapReport
     * @param option
     */
    protected void setTitle(ThemeMapReport themeMapReport, Option option){
//        Map<String, Object> titleMap = Maps.newHashMap();
//        titleMap.put("text", themeMapReport.getECharts().getTitle());
//        option.setTitle(titleMap);
    }

    protected void transShow(MetadataField metadataField, ChartBase chartBase, Option option,BuildContext buildContext){
        //重构，能否在构建前实现格式化（结合SDK）
        ThemeMapReport themeMapReport =  buildContext.getThemeMapReport();
        AgileReportECharts agileReportECharts = themeMapReport.getECharts();
        if(agileReportECharts == null || CollectionUtils.isEmpty((themeMapReport.getPageData()))){
            return;
        }

        String entry = buildContext.getExecuteContext().getEntry();
        if(!StringUtils.isEmpty(entry) && StringUtils.equals("0",entry)){
            List<Map<String,Object>> xAxisList = (List<Map<String, Object>>) option.getxAxis();
            if(CollectionUtils.isNotEmpty(xAxisList)) {
                for (Map<String, Object> xAxis : xAxisList) {
                    Map<String, Object> axisLabelMap = Maps.newHashMap();
                    axisLabelMap.put("interval", 0);
                    axisLabelMap.put("rotate", 60);
                    axisLabelMap.put("width", 60);
                    axisLabelMap.put("overflow", "truncate");
                    xAxis.put("axisLabel", axisLabelMap);
                }
            }
            Map<String, Object> grid = Maps.newHashMap();
            grid.put("height","63%");
            grid.put("right","13%");
            option.setGrid(grid);
        }
        //宽度样式
        Map<String,Object> gridMap = Maps.newHashMap();
        if(option.getGrid() != null){
            gridMap = (Map<String, Object>) option.getGrid();
        }
        option.setGrid(gridMap);
    }

    /**
     * 查找报表数据
     * @param target
     * @param eChartsDatas
     * @param buildContext
     * @return
     */
    protected List<MetadataField> getChartMetadataField(String target,List<AgileReportECharts.EChartsDataDefine> eChartsDatas,BuildContext buildContext){
        List<MetadataField> metadataFields = Lists.newArrayList();
        for(AgileReportECharts.EChartsDataDefine eChartsData : eChartsDatas){
            Map<String,MetadataField> fieldMap = ConvertGetAgileData.getAgileData(target,buildContext.getThemeMapReport().getApiMetadata());
            metadataFields.add(fieldMap.get(eChartsData.getApplyToField()));
        }
        return metadataFields;
    }

}
