package com.digiwin.athena.aim.api;

import com.digiwin.athena.aim.api.dto.AppExpireChangeDTO;
import com.digiwin.athena.aim.app.config.RabbitConfig;
import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.domain.message.service.AppExpireChangeService;
import com.digiwin.athena.aim.infrastructure.thememap.dto.TmAppExpireChangeComponentDTO;
import com.digiwin.athena.aim.infrastructure.thememap.service.ThemeMapService;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.amqp.core.ExchangeTypes;
import org.springframework.amqp.rabbit.annotation.Exchange;
import org.springframework.amqp.rabbit.annotation.Queue;
import org.springframework.amqp.rabbit.annotation.QueueBinding;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.support.AmqpHeaders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.messaging.handler.annotation.Header;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Map;

/**
 * AppExpireChangeListener Description
 *
 * @author majianfu
 * @date 2022/8/23
 * @since
 */
@Slf4j
@Component
public class AppExpireChangeListener {
    // 过期60天后清除数据
    private static final String APP_CLEAN = "clean";

    private static final String COMPONENT_TYPE_PROJECT = "Project";

    private static final String COMPONENT_TYPE_Task = "Task";

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private AppExpireChangeService appExpireChangeService;

    @Autowired
    private EnvProperties envProperties;

    @Resource
    protected MongoTemplate msgMongoTemplate;

    private static final String MONGODB_COLLECTION_NAME = "AIM_ApiExpireChange_Log";

    /**
     * 处理应用过期状态变更MQ事件（用刚过期、续租、已过60天缓冲期）
     *
     * @param data
     * @param channel
     * @param tag
     */
    @RabbitListener(bindings = @QueueBinding(value = @Queue(value = RabbitConfig.AIM_APP_EXPIRE_CHANGE_QUEUE, durable = "true", autoDelete = "false")
            , exchange = @Exchange(value = RabbitConfig.APP_EXPIRE_CHANGE_EXCHANGE, type = ExchangeTypes.FANOUT)))
    public void handleAppExpireChange(String data, Channel channel, @Header(AmqpHeaders.DELIVERY_TAG) long tag) throws IOException {
        log.info("handleAppExpireChange-ATMC_appExpireChangeQueue-{}", data);
        Integer retryCount = 0;
        boolean successed = false;
        try {
            if (StringUtils.isNotBlank(data)) {
                AppExpireChangeDTO appExpireChangeDTO = JsonUtils.jsonToObject(data, AppExpireChangeDTO.class);
                if (checkParam(appExpireChangeDTO)) {
                    log.info("handleAppExpireChange-{}", appExpireChangeDTO.toString());
                    handleAppExpireChange(appExpireChangeDTO);
                }
            }
            // data数据合法、data不合法（tenantId、eventType、appCode等不合法）都认为正常消费了该消息
            successed = true;
        } catch (Exception ex) {
            log.error("handleAppExpireChange-{}, consumer failed, error: {}", data, ex);
            successed = false;
            retryCount = getRetryCount(data, ex);
        }

        // 消费成功或者达到重试次数上限，则发出ack
        if (successed || retryCount >= envProperties.getAppExpireChangeMqRetryCount()) {
            channel.basicAck(tag, false);
        } else {
            //消息重回队列
            channel.basicReject(tag, true);
        }
    }

    private Integer getRetryCount(String data, Exception exception) {
        Integer retryCount = 1;
        try {
            Object exceptionMessage = exception.toString();
            try {
                exceptionMessage = JsonUtils.jsonToObject(JsonUtils.objectToString(exception), Map.class);
            } catch (Exception e1) {
            }

            int messageId = data.hashCode();
            Query query = Query.query(Criteria.where("exception-message-id").is(messageId));
            query.fields().include("exception-retry-count");
            Map log = msgMongoTemplate.findOne(query, Map.class, MONGODB_COLLECTION_NAME);
            if (log != null) {
                retryCount = (Integer) log.get("exception-retry-count");
                retryCount = retryCount + 1;

                query = new Query(Criteria.where("_id").is(log.get("_id")));
                Update update = new Update()
                        .set("exception-message", exceptionMessage)
                        .set("exception-retry-count", retryCount)
                        .set("exception-modify-time", LocalDateTime.now());
                msgMongoTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
            } else {
                log = JsonUtils.jsonToObject(data, Map.class);
                log.put("exception-message", exceptionMessage);
                log.put("exception-message-id", messageId);
                log.put("exception-retry-count", retryCount);
                log.put("exception-create-time", LocalDateTime.now());
                log.put("exception-modify-time", LocalDateTime.now());
                msgMongoTemplate.insert(log, MONGODB_COLLECTION_NAME);
            }
        } catch (Exception ex) {
            log.error("handleAppExpireChange-{} save error log failed, error: {}", data, ex);
        }

        return retryCount;
    }

    private void handleAppExpireChange(AppExpireChangeDTO appExpireChangeDTO) {
        // 根据tenantId + appCode查询所有的项目信息
        TmAppExpireChangeComponentDTO appExpireChangeProjectComponent = this.getAppExpireChangeProjectTaskComponent(appExpireChangeDTO);
        if (null != appExpireChangeProjectComponent) {
            log.info("handleAppExpireChange-{}", appExpireChangeProjectComponent.toString());
            // 只处理clean类型
            switch (appExpireChangeDTO.getEventType()) {
                case APP_CLEAN:
                    appExpireChangeService.handleAppClean(appExpireChangeDTO.getTenantId(), appExpireChangeDTO.getAppCode()
                            , appExpireChangeProjectComponent.getProject(), appExpireChangeProjectComponent.getTask());
                    break;
                default:
                    log.info("handleAppExpireChange-{}: Unsupported eventType", appExpireChangeDTO.toString());
            }
        } else {
            log.info("handleAppExpireChange-{}-{}", appExpireChangeDTO.toString(), "未查询到项目列表");
        }
    }

    private TmAppExpireChangeComponentDTO getAppExpireChangeProjectTaskComponent(AppExpireChangeDTO appExpireChangeDTO) {
        TmAppExpireChangeComponentDTO tmAppExpireChangeComponentReq = new TmAppExpireChangeComponentDTO();
        tmAppExpireChangeComponentReq.setTenantId(appExpireChangeDTO.getTenantId());
        tmAppExpireChangeComponentReq.setEventType(appExpireChangeDTO.getEventType());
        tmAppExpireChangeComponentReq.setAppCode(appExpireChangeDTO.getAppCode());
        tmAppExpireChangeComponentReq.setComponent(Arrays.asList(COMPONENT_TYPE_PROJECT, COMPONENT_TYPE_Task));

        return themeMapService.getAppExpireChangeComponent(tmAppExpireChangeComponentReq);
    }

    private boolean checkParam(AppExpireChangeDTO appExpireChangeDTO) {
        return null != appExpireChangeDTO
                && StringUtils.isNotBlank(appExpireChangeDTO.getTenantId())
                && StringUtils.isNotBlank(appExpireChangeDTO.getAppCode())
                && StringUtils.isNotBlank(appExpireChangeDTO.getEventType());
    }
}
