package com.digiwin.athena.aim.api;

import com.digiwin.athena.aim.api.dto.MessagePagingQryCommand;
import com.digiwin.athena.aim.api.dto.ReadMessageByBatchGidCommand;
import com.digiwin.athena.aim.api.dto.ReadMessageByGidCommand;
import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.MessageChannelEnum;
import com.digiwin.athena.aim.domain.message.model.*;
import com.digiwin.athena.aim.domain.message.service.MessageServiceV2;
import com.digiwin.athena.aim.domain.message.subscriber.WecomMessageSubscriber;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;

import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * MessageControllerV2 Description
 *
 * @author majianfu
 * @date 2021/4/25
 * @since
 */
@Slf4j
@RestController
@RequestMapping(value = "/api/aim/v2")
public class MessageControllerV2 {
    @Autowired
    private MessageServiceV2 messageServiceV2;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private StringRedisTemplate stringRedisTemplate;


    /**
     * 删除消息渠道
     */
    @GetMapping("/message/removeMessageChannel")
    public ResponseEntity<?> removeMessageChannel(String tenantId) {
        if (StringUtils.isNotBlank(tenantId)) {
            stringRedisTemplate.delete(WecomMessageSubscriber.CACHE_KEY+tenantId);
        }
        return ResponseEntityWrapper.wrapperOk();
    }


    /**
     * 人员限制最大值
     */
    public static final int MESSAGE_PERSONNEL_COUNT_MAX = 1000;

    /**
     * 批量发送消息
     *
     * @param messageBatchUserList
     * @return
     */
    @PostMapping(value = {"/batch/user/message", "/open/batch/user/message"})
    public ResponseEntity<?> newMessageForBatchUsers(@Validated @RequestBody List<MessageBatchUserDTO> messageBatchUserList) {
        if (CollectionUtils.isNotEmpty(messageBatchUserList)) {
            messageServiceV2.newMessageForBatchUsers(messageBatchUserList);
        }
        return ResponseEntityWrapper.wrapperOk();
    }

    /**
     * 1、分页查询消息
     */
    @PostMapping("/message/summary")
    public ResponseEntity<?> getMsgSummary(@RequestBody MessagePagingQryCommand command) {
        return messageServiceV2.pagingQueryMessage(command);
    }

    /**
     * 清除已读信息
     */
    @GetMapping("/message/summary/remove")
    public ResponseEntity<?> summaryRemove(@RequestParam(value = "channelType", required = false) String channelType) {
        return messageServiceV2.summaryRemove(channelType);
    }

    /**
     * 2、标记某个集合中的消息已读
     */
    @PostMapping("/message/read")
    public ResponseEntity<?> readMessageByGid(@RequestBody ReadMessageByBatchGidCommand command) {
        return messageServiceV2.readMessageByGid(command.getGidList());
    }

    /**
     * 一键已读当前用户的工作提醒
     */
    @GetMapping("/message/readAll")
    public ResponseEntity<?> readMessageAll(@RequestParam(value = "channelType", required = false) String channelType) {
        return messageServiceV2.readMessageAll(channelType);
    }

    /**
     * 2、标记某个集合中的消息已读
     */
    @PostMapping("/message/read/detail")
    public ResponseEntity<?> readMessageDetailByGid(@RequestBody ReadMessageByGidCommand command) {
        return messageServiceV2.readMessageDetailByGid(command.getGid());
    }

    /**
     * 应用到期提醒 批量发送消息
     *
     * @param expireMessageReq 消息
     * @return
     */
    @PostMapping(value = {"/expire/message", "/open/expire/message"})
    public ResponseEntity<?> insertAppExpirationMessage(@Validated @RequestBody ExpireMessageReq expireMessageReq) {
        if (expireMessageReq.getUserIdList().stream().anyMatch(StringUtils::isBlank)) {
            return ResponseEntityWrapper.wrapperFail(500, "userIdList cannot be empty");
        }
        messageServiceV2.insertAppExpirationMessage(expireMessageReq);
        return ResponseEntityWrapper.wrapperOk(true);
    }

    /**
     * 应用到期提醒 查询消息列表
     *
     * @return
     */
    @GetMapping(value = {"/expire/list", "/open/expire/list"})
    public ResponseEntity<?> queryAppExpirationMessageList(@RequestParam(required = false) Integer platformFlag) {
        String uri = envProperties.getSpringProperties().getUri();
        String database = envProperties.getSpringProperties().getDatabase();
        log.info("MessageControllerV2 queryAppExpirationMessageList. uri:{}, database:{}", uri, database);
        return ResponseEntityWrapper.wrapperOk(messageServiceV2.queryAppExpirationMessage(platformFlag));
    }

    /**
     * 应用到期提醒 应用续约，即删除该应用的到期提醒
     *
     * @param appExpireMessageDO 请求体
     * @return
     */
    @PostMapping(value = {"/expire/renewal", "/open/expire/renewal"})
    public ResponseEntity<?> deleteAppExpirationMessage(@Validated @RequestBody AppExpireMessageDO appExpireMessageDO) {
        messageServiceV2.deleteAppExpirationMessage(appExpireMessageDO);
        return ResponseEntityWrapper.wrapperOk(true);
    }

    /**
     * 各渠道发送消息统一接口，该接口已经废弃，不可调用，后续需要删除，发送消息能力由小AI统一消息替换
     *
     * @see MessageCenterController#send(MessageCenterSendReq)
     * @param sendCommonChannelMsgReq 请求对象
     * @return 返回对象
     */
    @PostMapping(value = {"/common/sendMessage", "/open/common/sendMessage"})
    @Deprecated
    public ResponseEntity<?> sendCommonChannelMsg(@Validated @RequestBody SendCommonChannelMsgReq sendCommonChannelMsgReq) {
        JSONObject jsonObject = new JSONObject();
        // 站内信
        if (CollectionUtils.isNotEmpty(sendCommonChannelMsgReq.getRemind())) {
            for (MessageBatchUserDTO messageBatchUserDTO : sendCommonChannelMsgReq.getRemind()) {
                if (messageBatchUserDTO.getUserIdList().size() > MESSAGE_PERSONNEL_COUNT_MAX) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeEnum.FAIL.getErrCode(), ErrorCodeEnum.FAIL.getErrMsg());
                }
            }
            messageServiceV2.newMessageForBatchUsers(sendCommonChannelMsgReq.getRemind());
        }
        // 邮件
        List<SendCommonChannelMsgReq.MsgInfoReq> emailMsgList = sendCommonChannelMsgReq.getEmail();
        if (CollectionUtils.isNotEmpty(emailMsgList)) {
            for (SendCommonChannelMsgReq.MsgInfoReq msgInfoReq : emailMsgList) {
                if ((CollectionUtils.isNotEmpty(msgInfoReq.getUserIdList()) && msgInfoReq.getUserIdList().size() > MESSAGE_PERSONNEL_COUNT_MAX)
                        || (CollectionUtils.isNotEmpty(msgInfoReq.getCcUserIdList()) && msgInfoReq.getCcUserIdList().size() > MESSAGE_PERSONNEL_COUNT_MAX)
                        || (CollectionUtils.isNotEmpty(msgInfoReq.getBccUserIdList()) && msgInfoReq.getBccUserIdList().size() > MESSAGE_PERSONNEL_COUNT_MAX)) {
                    return ResponseEntityWrapper.wrapperFail(ErrorCodeEnum.FAIL.getErrCode(), ErrorCodeEnum.FAIL.getErrMsg());
                }
            }
            JSONObject emailJSONbject = messageServiceV2.sendCommonChannelMessage(MessageChannelEnum.EMAIL.getFlag(), emailMsgList);
            jsonObject.put("email", emailJSONbject);
        }
        // 短信
        List<SendCommonChannelMsgReq.MsgInfoReq> smsMsgList = sendCommonChannelMsgReq.getSms();
        if (CollectionUtils.isNotEmpty(smsMsgList)) {
            if (validNumberMax(smsMsgList)) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeEnum.FAIL.getErrCode(), ErrorCodeEnum.FAIL.getErrMsg());
            }
            JSONObject smsJSONbject = messageServiceV2.sendCommonChannelMessage(MessageChannelEnum.SMS.getFlag(), smsMsgList);
            jsonObject.put("sms", smsJSONbject);
        }
        // 企微
        List<SendCommonChannelMsgReq.MsgInfoReq> weComMsgList = sendCommonChannelMsgReq.getWecom();
        if (CollectionUtils.isNotEmpty(weComMsgList)) {
            if (validNumberMax(weComMsgList)) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeEnum.FAIL.getErrCode(), ErrorCodeEnum.FAIL.getErrMsg());
            }
            JSONObject wecomJSONbject = messageServiceV2.sendCommonChannelMessage(MessageChannelEnum.WECOM.getFlag(), weComMsgList);
            jsonObject.put("wecom", wecomJSONbject);
        }
        // 钉钉
        List<SendCommonChannelMsgReq.MsgInfoReq> dingTalkMsgList = sendCommonChannelMsgReq.getDingTalk();
        if (CollectionUtils.isNotEmpty(dingTalkMsgList)) {
            if (validNumberMax(dingTalkMsgList)) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeEnum.FAIL.getErrCode(), ErrorCodeEnum.FAIL.getErrMsg());
            }
            JSONObject dingTalkJSONbject = messageServiceV2.sendCommonChannelMessage(MessageChannelEnum.DINGTALK.getFlag(), dingTalkMsgList);
            jsonObject.put("dingTalk", dingTalkJSONbject);
        }
        return ResponseEntityWrapper.wrapperOk(jsonObject);
    }

    /**
     * 校验入参人员最大值
     *
     * @param smsMsgList 参数
     * @return 返回
     */
    private boolean validNumberMax(List<SendCommonChannelMsgReq.MsgInfoReq> smsMsgList) {
        for (SendCommonChannelMsgReq.MsgInfoReq msgInfoReq : smsMsgList) {
            if (CollectionUtils.isNotEmpty(msgInfoReq.getUserIdList()) && msgInfoReq.getUserIdList().size() > MESSAGE_PERSONNEL_COUNT_MAX) {
                return true;
            }
        }
        return false;
    }

    /**
     * 发送消息/邮件
     *
     * @param msgAndEmailBatchDTOList
     * @return
     */
    @PostMapping(value = {"/sendMsgAndEmail", "/open/sendMsgAndEmail"})
    public ResponseEntity<?> sendMsgAndEmail(@Validated @RequestBody List<MsgAndEmailBatchDTO> msgAndEmailBatchDTOList) {
        if (CollectionUtils.isNotEmpty(msgAndEmailBatchDTOList)) {
            messageServiceV2.sendMsgAndEmail(msgAndEmailBatchDTOList);
        }
        return ResponseEntityWrapper.wrapperOk();
    }
}
