package com.digiwin.athena.aim.app.config;

import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.classic.spi.StackTraceElementProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.encoder.Encoder;
import com.digiwin.dap.middleware.lmc.http.client.HttpConfig;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import com.digiwin.dap.middleware.lmc.util.LoggingEventSizeUtil;
import com.digiwin.dap.middleware.lmc.util.ThreadPoolUtil;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.json.JSONArray;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * 用日志中心的，不用jugg-agile-biz-digiwin这个jar中的
 * 		<dependency>
 * 			<groupId>com.digiwin.dap.middleware</groupId>
 * 			<artifactId>lmc-sdk-logback</artifactId>
 * 			<version>2.4.1.1</version>
 * 		</dependency>
 *
 * @author chenzhuang
 */
public class DwLogbackAppender<E> extends UnsynchronizedAppenderBase<E> {

    private SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

    private static final String CUSTOM_CONTENT_KEY_SEPARATOR = ",";

    private static List<String> CUSTOM_CONTENT_KEY_LIST = new ArrayList<>();

    /**
     * 默认间隔时间:2s
     */
    private static final int DEFAULT_PERIOD_MILL_SECONDS = 2 ;
    /**
     * 默认批量条数:50
     */
    private static final int DEFAULT_BATCH_SIZE = 50;
    /**
     * 默认单条日志最大大小:10KB
     */
    private static final Long DEFAULT_MAX_LOG_SIZE = 10240L;

    private static final Integer DEFAULT_THREAD_NUM = 5;

    private ScheduledExecutorService executor;

    private int consumeCount = 0;

    private Queue<Map<String, Object>> workQueue = new LinkedList<>();

    private CloseableHttpClient client;

    protected Encoder<E> encoder;
    private String userAgent = "logback";

    private Integer intervals;

    private Integer batchSize;

    private Long maxSingleLogSize;

    private String app;
    private String endpoint;

    private String customContentKey;


    @Override
    public void start() {
        if(null == this.intervals){
            this.intervals = DEFAULT_PERIOD_MILL_SECONDS;
        }
        if(null == this.batchSize){
            this.batchSize = DEFAULT_BATCH_SIZE;
        }
        if(null == this.maxSingleLogSize){
            this.maxSingleLogSize = DEFAULT_MAX_LOG_SIZE;
        }
        if(null != this.customContentKey && customContentKey.length() != 0){
            CUSTOM_CONTENT_KEY_LIST = Arrays.asList(this.customContentKey.split(CUSTOM_CONTENT_KEY_SEPARATOR));
        }
        // 请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConfig.CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(HttpConfig.CONNECT_REQUEST_TIMEOUT)
                .setSocketTimeout(HttpConfig.SOCKET_TIMEOUT).build();

        // 连接池管理
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setMaxTotal(HttpConfig.MAX_TOTAL_CONNECTIONS);
        connectionManager.setDefaultMaxPerRoute(HttpConfig.MAXIMUM_CONNECTION_PER_ROUTE);
        connectionManager.setValidateAfterInactivity(HttpConfig.CONNECTION_VALIDATE_AFTER_INACTIVITY_MS);
        this.client = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager).build();

        // 配置定时
        executor = Executors.newSingleThreadScheduledExecutor();
        executor.scheduleAtFixedRate(() -> {
            synchronized (DwLogbackAppender.this) {
                if (!workQueue.isEmpty()) {
                    List<Map<String, Object>> logList = new LinkedList<>();
                    for (Map<String, Object> element : workQueue) {
                        logList.add(element);
                    }
                    asyncPersistence(logList);
                    consumeCount = 0;
                    workQueue.clear();
                }
            }
        }, 0, this.intervals, TimeUnit.SECONDS);

        super.start();
    }

    @Override
    public void stop() {
        if (!isStarted()) {
            return;
        }
        List<Map<String, Object>> logList = new LinkedList<>();
        for (Map<String, Object> element : workQueue) {
            logList.add(element);
        }
        asyncPersistence(logList);
        executor.shutdown();
        super.stop();
        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException exception) {
                addError(exception.getMessage());
            }
        }
    }

    @Override
    public void append(E eventObject) {
        synchronized (this) {
            while (this.workQueue.size() > this.batchSize) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            Map<String, Object> map = convertLoggingEventToMap(eventObject);
            if (LoggingEventSizeUtil.getSize(map) >= this.maxSingleLogSize * 1024) {
                List<Map<String,Object>> log = Arrays.asList(map);
                asyncPersistence(log);
                this.workQueue.clear();
                consumeCount = 0;
            }
            if (Objects.nonNull(map)) {
                this.workQueue.offer(map);
                notifyAll();
                consumeCount++;
            }
            if (consumeCount >= this.batchSize) {
                List<Map<String, Object>> logList = new LinkedList<>();
                for (Map<String, Object> element : workQueue) {
                    logList.add(element);
                }
                asyncPersistence(logList);
                this.workQueue.clear();
                consumeCount = 0;
            }
        }
    }

    private void asyncPersistence(List<Map<String,Object>> logList) {
        ThreadPoolUtil.executor(
                () -> {
                    if (null == logList || logList.isEmpty()) {
                        return;
                    }
                    CloseableHttpResponse hResponse = null;
                    HttpPost postMethod = new HttpPost(LMCResourceUri.getSaveDevLogBatchUrl(this.endpoint));
                    try {
                        JSONArray jsonArray = new JSONArray(logList);
                        String formatValue = jsonArray.toString();
                        postMethod.setEntity(new StringEntity(formatValue, ContentType.APPLICATION_JSON));
                        hResponse = this.client.execute(postMethod);
                        HttpEntity repEntity = hResponse.getEntity();
                        int statusCode = hResponse.getStatusLine().getStatusCode();
                        if (statusCode != 200) {
                            postMethod.abort();
                        }
                    } catch (Exception e) {
                        addError(e.getMessage());
                    } finally {
                        if (hResponse != null) {
                            try {
                                hResponse.close();
                            } catch (IOException e) {
                                addError(e.getMessage());
                            }
                        }
                    }
                }
        );
    }

    private Map<String, Object> convertLoggingEventToMap(E eventObject) {
        if (!(eventObject instanceof LoggingEvent)) {
            return null;
        }
        LoggingEvent event = (LoggingEvent) eventObject;
        Map<String, Object> logMap = new HashMap();

        Map<String, String> mdcPropertyMap = event.getMDCPropertyMap();
        String traceId;
        if (!mdcPropertyMap.isEmpty()) {
            String ptxId = mdcPropertyMap.get("PtxId");
            //String pspanId = mdcPropertyMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
            Map<String, String> customContent = new HashMap<>();
            CUSTOM_CONTENT_KEY_LIST.forEach(key -> {
                if (Objects.nonNull(mdcPropertyMap.get(key))) {
                    customContent.put(key, mdcPropertyMap.get(key));
                }
            });
            logMap.put("customContent", customContent);
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date(event.getTimeStamp())));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLogbackAppender");

        StackTraceElement[] caller = event.getCallerData();
        if (caller != null && caller.length > 0) {
            logMap.putIfAbsent("location", caller[0].toString());
        }

        String message = event.getFormattedMessage();
        logMap.putIfAbsent("message", message);

        IThrowableProxy iThrowableProxy = event.getThrowableProxy();
        if (iThrowableProxy != null) {
            StringBuilder builder = new StringBuilder();

            String throwableProxyClassName = iThrowableProxy.getClassName();
            String throwableProxyMessage = iThrowableProxy.getMessage();

            String throwable = (throwableProxyMessage != null) ?
                    (throwableProxyClassName + ": " + throwableProxyMessage) : throwableProxyClassName;

            builder.append(throwable);
            for (StackTraceElementProxy step : event.getThrowableProxy().getStackTraceElementProxyArray()) {
                builder.append(CoreConstants.LINE_SEPARATOR);
                String string = step.toString();
                builder.append(CoreConstants.TAB).append(string);
                ThrowableProxyUtil.subjoinPackagingData(builder, step);
            }
            logMap.putIfAbsent("throwable", builder.toString());
        }

        if (this.encoder != null) {
            logMap.putIfAbsent("log", new String(this.encoder.encode(eventObject)));
        }
        return logMap;
    }



    public String getApp() {
        return app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public Integer getIntervals() {
        return intervals;
    }

    public void setIntervals(Integer intervals) {
        this.intervals = intervals;
    }

    public Integer getBatchSize() {
        return batchSize;
    }

    public void setBatchSize(Integer batchSize) {
        this.batchSize = batchSize;
    }

    public Long getMaxSingleLogSize() {
        return maxSingleLogSize;
    }

    public void setMaxSingleLogSize(Long maxSingleLogSize) {
        this.maxSingleLogSize = maxSingleLogSize;
    }

    public String getCustomContentKey() {
        return customContentKey;
    }

    public void setCustomContentKey(String customContentKey) {
        this.customContentKey = customContentKey;
    }

    public Encoder<E> getEncoder() {
        return encoder;
    }

    public void setEncoder(Encoder<E> encoder) {
        this.encoder = encoder;
    }
}
