package com.digiwin.athena.aim.app.config;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import com.digiwin.athena.aim.util.WebToolUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;



import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.MediaType;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.util.ContentCachingRequestWrapper;
import org.springframework.web.util.ContentCachingResponseWrapper;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;


/**
 * 请求参数和响应日志打印
 */
@Configuration
@Slf4j
public class LogParamConfig {

    private static final Set<String> EXCLUDE_URL_SET = Sets.newHashSet(Lists.newArrayList(
            "/api/env",
            "/api/aim/v1/misc/downFile",
            "/aim/actuator/prometheus"
    ));

    private static final int LOG_SIZE=2000;

    @Bean
    public OncePerRequestFilter contentCachingRequestFilter() {
        return new OncePerRequestFilter() {
            @Override
            protected void doFilterInternal(final HttpServletRequest request, final HttpServletResponse response, final FilterChain filterChain) throws ServletException, IOException {
                boolean isLog = !EXCLUDE_URL_SET.contains(request.getServletPath());
                long startTime = 0L;
                if (isLog) {
                    // 记录请求开始以及log信息
                    recordRequestStart(request, response);
                    startTime = System.currentTimeMillis();
                }
                ContentCachingRequestWrapper wrappedRequest = new ContentCachingRequestWrapper(request);
                ContentCachingResponseWrapper wrappedResponse = new ContentCachingResponseWrapper(response);
                filterChain.doFilter(wrappedRequest, wrappedResponse);
                // 记录请求结束
                if (isLog) {
                    recordRequestEnd(wrappedRequest, wrappedResponse, System.currentTimeMillis() - startTime);
                }
                wrappedResponse.copyBodyToResponse();
            }
        };
    }


    /**
     * 记录网络请求开始
     *
     * @param request 请求
     */
    @SuppressWarnings("all")
    private void recordRequestStart(HttpServletRequest request, HttpServletResponse response) {
        try {
            MDC.put("filter1", getUUID());
            String userId;
            String tenantId;
            if (AppAuthContextHolder.getContext() != null && AppAuthContextHolder.getContext().getAuthoredUser() != null) {
                userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
                tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
                MDC.put("userId", userId);
                MDC.put("tenantId", tenantId);
            }
            Map<String, String> extraInfoMap = new HashMap<>();
            extraInfoMap.put("Accessor IP", WebToolUtils.getRemortIP(request));
            extraInfoMap.put("Host IP", request.getRemoteHost());
            extraInfoMap.put("UrlPath", request.getRequestURL().toString());
            extraInfoMap.put("QueryString", request.getQueryString());
            extraInfoMap.put("Referer", request.getHeader("Referer"));
            extraInfoMap.put("Token", request.getHeader("Token"));
            extraInfoMap.put("traceId", request.getHeader("traceId"));
            //客户端请求的唯一标识（前端会在请求头中增加这个uuid）
            extraInfoMap.put("requestedId", request.getHeader("X-Requested-With"));
            log.info(request.getServletPath() + " 接口请求开始，\r\n" + mapToString(extraInfoMap));
        } catch (Exception e) {
            log.error("记录请求开始异常", e);
        }

    }

    /**
     * 记录网络请求结束
     *
     * @param requestWrapper 氢气
     * @param resultWrapper  结果
     * @param cost           耗时
     */
    @SuppressWarnings("all")
    private void recordRequestEnd(ContentCachingRequestWrapper requestWrapper, ContentCachingResponseWrapper resultWrapper, long cost) {
        try {
            Map<String, String> logExtraInfo = new HashMap<>();
            //获取方法参数
            byte[] requestBytes = requestWrapper.getContentAsByteArray();
            if (requestBytes.length>LOG_SIZE) {
                logExtraInfo.put("request", new String(slice(requestBytes, 0, LOG_SIZE)));
            }
            else {
                logExtraInfo.put("request", new String(requestBytes));
            }
            String contentType = resultWrapper.getContentType();
            if (StringUtils.isNotBlank(contentType) && contentType.contains(MediaType.APPLICATION_OCTET_STREAM_VALUE)) {
                logExtraInfo.put("response", "响应文件流");
            } else {
                byte[] responseBytes = resultWrapper.getContentAsByteArray();
                if (responseBytes.length>LOG_SIZE) {
                    logExtraInfo.put("response", new String(slice(responseBytes, 0, LOG_SIZE)));
                }
                else {
                    logExtraInfo.put("response", new String(responseBytes));
                }
            }
            logExtraInfo.put("cost", cost + "");
            if (cost>1000) {
                log.warn("慢请求：{}，{}", cost,requestWrapper.getServletPath());
            }
            log.info(requestWrapper.getServletPath() + " 接口请求结束，\r\n" + mapToString(logExtraInfo));

        } catch (Exception e) {
            log.error("记录请求结束异常", e);
        } finally {
            try {
               MDC.clear();
            } catch (Exception e) {
                log.error("logMdcRemoveEx", e);
            }

        }
    }


    private static String mapToString(Map<String, String> extraInfo) {
        StringBuilder sb = new StringBuilder();
        if (MapUtils.isEmpty(extraInfo)) {
            return StringUtils.EMPTY;
        }
        if(sb.length() > 0) {
            sb.append("\r\n");
        }
        Iterator var3 = extraInfo.keySet().iterator();
        while (var3.hasNext()) {
            String s = (String) var3.next();
            sb.append(s).append(":").append(extraInfo.get(s)).append("\r\n");
        }
        sb.setLength(Math.max(sb.length() - 1, 0));
        return sb.toString();
    }


    private static byte[] slice(byte[] original, int start, int length) {
        byte[] slice = new byte[length];
        System.arraycopy(original, start, slice, 0, length);
        return slice;
    }


    private static String getUUID() {
        UUID uuid = UUID.randomUUID();
        char[] dest = new char[32];
        char[] src = uuid.toString().toCharArray();
        System.arraycopy(src, 0, dest, 0, 8);
        System.arraycopy(src, 9, dest, 8, 4);
        System.arraycopy(src, 14, dest, 12, 4);
        System.arraycopy(src, 19, dest, 16, 4);
        System.arraycopy(src, 24, dest, 20, 12);
        return new String(dest);
    }




}
