package com.digiwin.athena.aim.app.env;

import com.digiwin.athena.aim.infrastructure.mobile.MobileProperties;
import com.digiwin.athena.aim.infrastructure.mobile.SpringProperties;
import com.digiwin.athena.aim.infrastructure.mqtt.MqttProperties;
import com.digiwin.athena.appcore.AppCoreProperties;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.Data;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

/**
 * 配置信息
 *
 * @author chenxsa
 */
@Component
@Data
@Configuration
@JsonIgnoreProperties("$$beanFactory")
public class EnvProperties {
    private static final Logger logger = LoggerFactory.getLogger(EnvProperties.class);

    @Autowired
    private AppCoreProperties appCoreProperties;

    @Autowired
    private MobileProperties mobileProperties;

    @Autowired
    private SpringProperties springProperties;

    @Autowired
    private MqttProperties mqttConfiguration;

    @Value("${app.version:}")
    private String version;

    @Value("${app.build.time}")
    private String buildTime;

    @Value("${spring.profiles.active:dev}")
    private String active;

    @Value("${server.port}")
    private String port;

    @Value("${knowledgegraph.uri:}")
    private String knowledgegraphUri;

    private boolean espMockData;

    private boolean espGatheringESPData;

    /**
     * 应用过期MQ消费失败重试次数
     */
    private int appExpireChangeMqRetryCount = 10;

    /**
     * dap中间件
     */
    @Value("${iam.uri:}")
    private String iamUri;

    /**
     * dap中间件
     */
    @Value("${emc.uri:}")
    private String emcUri;


    /**
     * dap中间件
     */
    @Value("${eoc.uri:}")
    private String eocUri;

    /**
     * atmc
     */
    @Value("${atmc.uri:}")
    private String atmcUri;

    /**
     * semc
     */
    @Value("${semc.uri:}")
    private String semcUri;

    /**
     * athena地址
     */
    @Value("${athenaDd.uri:}")
    private String athenaDdUri;

    /**
     * athena 移动端地址
     */
    @Value("${athenaMobile.uri:}")
    private String athenaMobileUri;

    /**
     * PC域名
     */
    @Value("${pcAddress.uri:}")
    private String pcAddressUri;


    @Value("${dingding.athenaAppToken:}")
    private String appToken;

    /**
     * 发送钉钉消息事件ID
     */
    @Value("${dingding.noticeEventId:}")
    private String noticeEventId;

    /**
     * 娜娜发送asa企微消息代开发应用模板ID
     */
    @Value("${asa.wecomNoticeSuiteId:}")
    private String nanaSuiteId;

    /**
     * 娜娜发送asa企微消息模版
     */
    @Value("${asa.wecomNoticeEventId:}")
    private String wecomNoticeEventId;

    /**
     * 娜娜发送asa企微消息模版
     */
    @Value("${asa.dingNoticeEventId:}")
    private String asaNoticeEventId;

    /**
     * 娜娜PC域名
     */
    @Value("${ania-web.uri:}")
    private String aniaWebAddressUri;

    /**
     * 娜娜PC域名
     */
    @Value("${ania.uri:}")
    private String aniaAddressUri;

    @Value("${asa.lineLiffUrl}")
    private String asaLineUrl;

    @Value("${asa.lineNoticeEventId}")
    private String lineNoticeEventId;

    /**
     * dap翻译
     */
    @Value("${trans.uri:}")
    private String transUri;

    /**
     * 默认语系
     */
    @Value("${language.default:}")
    private String defaultLang;


    /**
     * 当前进程运行的主机名
     */
    private String host;
    /**
     * 当前进程所在的IP地址
     */
    private String ipAddress;
    /**
     * 空闲内存
     */
    private long freeMemory;
    /**
     * 内存总量
     */
    private long totalMemory;
    /**
     * java虚拟机允许开启的最大的内存
     */
    private long maxMemory;
    /**
     * 操作系统名称
     */
    private String osName;
    /**
     * 进程号
     */
    private long pid;
    /**
     * 程序启动时间
     */
    private LocalDateTime startTime;
    /**
     * 类所在路径
     */
    private String classPath;
    private String projectPath;
    /**
     * 程序运行时间，单位毫秒
     */
    private long runtime;
    /**
     * 线程总量
     */
    private int threadCount;

    public EnvProperties() {
    }

    /**
     * 把byte转换成M
     *
     * @param bytes
     * @return
     */
    static long byteToM(long bytes) {
        long kb = (bytes / 1024 / 1024);
        return kb;
    }

    /**
     * 获取进程号，适用于windows与linux
     *
     * @return
     */
    static long tryGetPid() {
        try {
            String name = ManagementFactory.getRuntimeMXBean().getName();
            String pid = name.split("@")[0];
            return Long.parseLong(pid);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    public void refresh() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        Runtime runtime = Runtime.getRuntime();
        //空闲内存
        long freeMemory = runtime.freeMemory();
        this.setFreeMemory(byteToM(freeMemory));
        //内存总量
        long totalMemory = runtime.totalMemory();
        this.setTotalMemory(byteToM(totalMemory));
        //最大允许使用的内存
        long maxMemory = runtime.maxMemory();
        this.setMaxMemory(byteToM(maxMemory));
        //操作系统
        this.setOsName(System.getProperty("os.name"));
        InetAddress localHost;
        try {
            localHost = InetAddress.getLocalHost();
            String hostName = localHost.getHostName();
            this.setHost(hostName);
            if (ipAddress == null) {
                ipAddress = localHost.getHostAddress();
            }
        } catch (UnknownHostException e) {
            logger.error("UnknownHostException",e);
            this.setHost("未知");
        }
        //ip
        this.setIpAddress(ipAddress);

        //程序启动时间
        long startTime = runtimeMXBean.getStartTime();
        LocalDateTime startDate = Instant.ofEpochMilli(startTime).atZone(ZoneOffset.ofHours(8)).toLocalDateTime();
        this.setStartTime(startDate);
        //类所在路径
        this.setRuntime(runtimeMXBean.getUptime());
        //线程总数
        this.setThreadCount(ManagementFactory.getThreadMXBean().getThreadCount());
        this.setPid(tryGetPid());
    }

    public String getVersion() {
        return version;
    }

    public void setVersion(String version) {
        this.version = version;
    }

    public String getBuildTime() {
        return buildTime;
    }

    public void setBuildTime(String buildTime) {
        this.buildTime = buildTime;
    }

    public String getActive() {
        return active;
    }

    public void setActive(String active) {
        this.active = active;
    }

    public String getPort() {
        return port;
    }

    public void setPort(String port) {
        this.port = port;
    }

    public String getIamUri() {
        return iamUri;
    }

    public void setIamUri(String iamUri) {
        this.iamUri = iamUri;
    }

    public String getNoticeEventId() {
        return noticeEventId;
    }

    public void setNoticeEventId(String noticeEventId) {
        this.noticeEventId = noticeEventId;
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = ipAddress;
    }

    public long getFreeMemory() {
        return freeMemory;
    }

    public void setFreeMemory(long freeMemory) {
        this.freeMemory = freeMemory;
    }

    public long getTotalMemory() {
        return totalMemory;
    }

    public void setTotalMemory(long totalMemory) {
        this.totalMemory = totalMemory;
    }

    public long getMaxMemory() {
        return maxMemory;
    }

    public void setMaxMemory(long maxMemory) {
        this.maxMemory = maxMemory;
    }

    public String getOsName() {
        return osName;
    }

    public void setOsName(String osName) {
        this.osName = osName;
    }

    public long getPid() {
        return pid;
    }

    public void setPid(long pid) {
        this.pid = pid;
    }

    public LocalDateTime getStartTime() {
        return startTime;
    }

    public void setStartTime(LocalDateTime startTime) {
        this.startTime = startTime;
    }

    public String getClassPath() {
        return classPath;
    }

    public void setClassPath(String classPath) {
        this.classPath = classPath;
    }

    public String getProjectPath() {
        return projectPath;
    }

    public void setProjectPath(String projectPath) {
        this.projectPath = projectPath;
    }

    public long getRuntime() {
        return runtime;
    }

    public void setRuntime(long runtime) {
        this.runtime = runtime;
    }

    public int getThreadCount() {
        return threadCount;
    }

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    public String getEmcUri() {
        return emcUri;
    }

    public void setEmcUri(String emcUri) {
        this.emcUri = emcUri;
    }

    public String getAtmcUri() {
        return atmcUri;
    }

    public void setAtmcUri(String atmcUri) {
        this.atmcUri = atmcUri;
    }


    public String getAthenaDdUri() {
        return athenaDdUri;
    }

    public void setAthenaDdUri(String athenaDdUri) {
        this.athenaDdUri = athenaDdUri;
    }
}
