package com.digiwin.athena.aim.domain.message.service.impl;

import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.digiwin.athena.aim.api.dto.DapEmpDisabledDTO;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.model.MsgStateEnum;
import com.digiwin.athena.aim.domain.message.repository.MessageMapperV2;
import com.digiwin.athena.aim.domain.message.service.ExtCustomMsgService;
import com.digiwin.athena.aim.domain.message.service.MessageSendService;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.iam.dto.IamUserDTO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;

import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;

/**
 * @program: aim
 * @description: 外部定制发送消息服务
 * @author: Hang Liu
 * @create: 2023-06-25 11:20
 */
@Service
@Slf4j
public class ExtCustomMsgServiceImpl implements ExtCustomMsgService {

    private static final String EMP_DISABLED = "empDisabled";

    @Resource
    private MessageSendService messageSendService;

    @Resource
    private IamService iamService;

    @Autowired
    private MessageMapperV2 messageMapperV2;

    @Autowired
    MessageUtils messageUtils;

    @Override
    public void dapEmpDisabled(DapEmpDisabledDTO dapEmpDisabledDTO) {

        log.info("[dapEmpDisabled] sendMessageRemindToClient-payload ：{}", JsonUtils.objectToString(dapEmpDisabledDTO));
        // 查询租户管理员信息
        List<IamUserDTO> userList = iamService.getUserListByRole("superadmin");
        Set<String> userIds = userList.stream().filter(user -> !user.isDeleted() || !user.isDisabled())
            .map(IamUserDTO::getId).collect(Collectors.toSet());
        List<MessageDO> messageDOList = new ArrayList<>();
        userIds.forEach(userId -> {
            // 构建消息
            MessageDO messageDO = buildMessagePayload(dapEmpDisabledDTO, userId);
            messageDOList.add(messageDO);
            // 发送消息
            log.info("[dapEmpDisabled] sendMessageRemindToClient-payload ：tenantId={},userId={},message={}",
                    dapEmpDisabledDTO.getTenantId(),
                    userId,
                    JsonUtils.objectToString(messageDO));
            messageSendService.sendToClient(dapEmpDisabledDTO.getTenantId(), userId, messageDO);
        });
        // 记录消息
        log.info("[dapEmpDisabled] insert message to mongo ：{}", JsonUtils.objectToString(messageDOList));
        if (CollectionUtils.isNotEmpty(messageDOList)){
            messageMapperV2.batchInsert(messageDOList);
        }

    }

    private MessageDO buildMessagePayload(DapEmpDisabledDTO dapEmpDisabledDTO, String userId) {

        MessageDO messageDO = new MessageDO();
        messageDO.setState(MsgStateEnum.UNREAD.getState());
        messageDO.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
        messageDO.setUserId(userId);
        messageDO.setTenantId(dapEmpDisabledDTO.getTenantId());
        messageDO.setSendDate(LocalDateTime.now());
        messageDO.setType(EMP_DISABLED);
        String langName = iamService.getUserMetadataAllTenant(userId, 1,0);
        messageDO.setLangName(langName);
        JSONObject content = new JSONObject();
        content.put("title", MessageFormat.format(messageUtils.getMessageByLangName("emp.disabled.msg.notice", langName), dapEmpDisabledDTO.getUserName(), dapEmpDisabledDTO.getReplacerUserName()));
        content.put("msg","");
        messageDO.setContent(content);
        return messageDO;
    }
}
