package com.digiwin.athena.aim.domain.message.service.impl;

import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.MessageCenterConstant;
import com.digiwin.athena.aim.domain.message.model.*;
import com.digiwin.athena.aim.domain.message.repository.MessageCenterMapper;
import com.digiwin.athena.aim.domain.message.repository.MessageMapperV2;
import com.digiwin.athena.aim.domain.message.service.MessageCenterService;
import com.digiwin.athena.aim.infrastructure.concurrent.pool.DynamicThreadPool;
import com.digiwin.athena.aim.infrastructure.emc.EmcService;
import com.digiwin.athena.aim.infrastructure.eoc.EocService;
import com.digiwin.athena.aim.infrastructure.eoc.dto.EmpDTO;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.iam.dto.PersonalizedDto;
import com.digiwin.athena.aim.util.ScriptUtil;
import com.digiwin.athena.aim.util.ValidateUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.data.domain.PageImpl;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


@Slf4j
@Service
public class MessageCenterServiceImpl implements MessageCenterService {


    @Autowired
    private EmcService emcService;

    @Autowired
    private EocService eocService;

    @Autowired
    private IamService iamService;

    @Autowired
    private MessageMapperV2 messageMapperV2;

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private MessageCenterMapper messageCenterMapper;

    @Autowired
    DynamicThreadPool taskExecutor;


    @Override
    public PageImpl<MessageCenterRecordDO> record(MessageCenterRecordReq messageCenterSendReq) throws IllegalAccessException {

        ValidateUtils.checkPageParam(messageCenterSendReq);
        return messageCenterMapper.pageMessageCenterRecord(messageCenterSendReq);
    }


    @Override
    public void send(MessageCenterSendReq messageCenterSendReq) {

        log.info("[send] send msg start, req = {}", JsonUtils.objectToString(messageCenterSendReq));
        // 查询场景配置
        List<MessageCenterConfigDTO> msgConfigList = emcService.getMsgConfigList(messageCenterSendReq.getTenantId(), messageCenterSendReq.getAppCode(), messageCenterSendReq.getSceneId());
        log.info("[send] 查询场景配置 result = {}", JsonUtils.objectToString(msgConfigList));

        if (CollectionUtils.isEmpty(msgConfigList)) {
            return;
        }
        // 处理发送场景
        for (MessageCenterConfigDTO messageCenterConfig : msgConfigList) {
            // 条件判断
            Boolean isSend = judgmentConditional(messageCenterSendReq.getData(), messageCenterConfig.getType().getTriggerCondition());
            log.info("[send] 条件判断 result = {}", isSend);
            if (isSend) {
                // 发送消息
                sendMsg(messageCenterSendReq.getTenantId(), messageCenterSendReq.getAppCode(), messageCenterSendReq.getData(), messageCenterConfig);
            }
        }
    }

    /**
     * 消息发送
     *
     * @param tenantId
     * @param appCode
     * @param data
     * @param messageCenterConfig
     */
    private void sendMsg(String tenantId, String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        log.info("[send] sendMsg start tenantId = {}, appCode = {}, data = {}, messageCenterConfig = {} "
                , tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig));

        if (CollectionUtils.isEmpty(messageCenterConfig.getSettings())) {
            return;
        }

        //  去除后续所有可能因为data 造成的空指针问题
        if (MapUtils.isEmpty(data)) {
            data = new HashMap<>();
        }

        // web消息发送
        sendWebMsg(tenantId, appCode, data, messageCenterConfig);

        // app消息发送
        sendAppMsg(tenantId, appCode, data, messageCenterConfig);

        // 邮箱发送
        sendEmailMsg(appCode, data, messageCenterConfig);

        // 短信发送
        sendSmsMsg(appCode, data, messageCenterConfig);

        // 钉钉发送
        sendDingTalkMsg(tenantId, appCode, data, messageCenterConfig);

        // 企业微信发送
        sendWeComMsg(appCode, data, messageCenterConfig);

        log.info("[send] sendMsg end tenantId = {}, appCode = {}, data = {}, messageCenterConfig = {} "
                , tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig));
    }

    public void sendWeComMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        //获取企业微信渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.WE_COM.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }

        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendWeComMsg start data = {},messageCenterSendConfig = {}",
                        JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.WE_COM, data, messageCenterSendConfig);
                log.info("[send] sendWeComMsg  analysisPushConfig receivers = {}", JsonUtils.objectToString(receivers));
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts)) {
                    return;
                }
                // 消息发送
                sendWeComMsg(appCode, data, messageCenterConfig, contacts);
            }
        });
    }

    public void sendDingTalkMsg(String tenantId, String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        //获取钉钉渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.DING_TALK.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }

        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendDingTalkMsg start data = {},messageCenterSendConfig = {}",
                        JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.DING_TALK, data, messageCenterSendConfig);
                log.info("[send] sendDingTalkMsg  analysisPushConfig receivers = {}", JsonUtils.objectToString(receivers));
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts)) {
                    return;
                }
                // 消息发送
                sendDingTalkMsg(tenantId, appCode, data, messageCenterConfig, contacts);
            }
        });


    }

    public void sendSmsMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        // 获取sms渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.SMS.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }

        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendSmsMsg start data = {},messageCenterSendConfig = {}",
                        JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.SMS, data, messageCenterSendConfig);
                log.info("[send] sendSmsMsg  analysisPushConfig receivers = {}", JsonUtils.objectToString(receivers));
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts)) {
                    return;
                }
                sendSmsMsg(appCode, data, messageCenterConfig, contacts);
            }
        });

    }

    public void sendEmailMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        //获取email渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.EMAIL.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }

        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendEmailMsg start data = {},messageCenterSendConfig = {}",
                        JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.EMAIL, data, messageCenterSendConfig);
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                log.info("[send] sendEmailMsg  analysisPushConfig receivers = {}", JsonUtils.objectToString(receivers));
                // 获取抄送人
                List<EmpDTO> ccContacts = receivers.get(MessageCenterConstant.ContactsType.CC_CONTACTS);
                // 获取密送人
                List<EmpDTO> bccContacts = receivers.get(MessageCenterConstant.ContactsType.BCC_CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts) && CollectionUtils.isEmpty(ccContacts) && CollectionUtils.isEmpty(bccContacts)) {
                    return;
                }
                // 发送消息
                sendEmailMsg(appCode, data, messageCenterConfig, contacts, ccContacts, bccContacts);
            }
        });
    }

    public void sendAppMsg(String tenantId, String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        //获取app渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.APP.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }
        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendAppMsg start tenantId = {},appCode = {},data = {},messageCenterSendConfig = {}",
                        tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.APP, data, messageCenterSendConfig);
                log.info("[send] sendAppMsg  analysisPushConfig  tenantId = {},appCode = {},receivers = {}", tenantId, appCode, JsonUtils.objectToString(receivers));
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts)) {
                    return;
                }
                // 获取模板
                List<MessageCenterTemplateDTO> templates = analysisTemplate(data, messageCenterSendConfig);
                log.info("[send] sendAppMsg analysisTemplate  tenantId = {},appCode = {},data = {},templates = {}",
                        tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(templates));
                // 数据插入
                saveMessage(tenantId, appCode, MessageCenterConstant.MessageChannel.APP, contacts, templates, data);
                // 记录消息记录
                saveMessageRecord(MessageCenterConstant.MessageChannel.APP, tenantId, appCode, messageCenterConfig.getType(), messageCenterSendConfig, contacts, templates);
            }
        });

    }


    public void sendWebMsg(String tenantId, String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig) {

        //获取web渠道配置
        List<MessageCenterSendConfigDTO> messageCenterSendConfigList = messageCenterConfig.getSettings().stream().filter(messageCenterSendConfigDTO -> MessageCenterConstant.MessageChannel.WEB.equals(messageCenterSendConfigDTO.getType())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(messageCenterSendConfigList)) {
            return;
        }
        taskExecutor.submit(() -> {
            for (MessageCenterSendConfigDTO messageCenterSendConfig : messageCenterSendConfigList) {
                log.info("[send] sendWebMsg start tenantId = {},appCode = {},data = {},messageCenterSendConfig = {}",
                        tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterSendConfig));
                // 解析渠道人员
                Map<String, List<EmpDTO>> receivers = analysisPushConfig(MessageCenterConstant.MessageChannel.WEB, data, messageCenterSendConfig);
                log.info("[send] sendWebMsg analysisPushConfig  tenantId = {},appCode = {},receivers = {}", tenantId, appCode, JsonUtils.objectToString(receivers));
                // 获取联系人
                List<EmpDTO> contacts = receivers.get(MessageCenterConstant.ContactsType.CONTACTS);
                // 没有联系人，不发送
                if (CollectionUtils.isEmpty(contacts)) {
                    return;
                }
                // 获取模板
                List<MessageCenterTemplateDTO> templates = analysisTemplate(data, messageCenterSendConfig);
                log.info("[send] sendWebMsg analysisTemplate  tenantId = {},appCode = {},data = {},templates = {}",
                        tenantId, appCode, JsonUtils.objectToString(data), JsonUtils.objectToString(templates));
                // 消息提醒数据插入
                saveMessage(tenantId, appCode, MessageCenterConstant.MessageChannel.WEB, contacts, templates, data);
                // 记录消息记录
                saveMessageRecord(MessageCenterConstant.MessageChannel.WEB, tenantId, appCode, messageCenterConfig.getType(), messageCenterSendConfig, contacts, templates);
            }
        });
    }

    private void saveMessageRecord(String channelType, String tenantId, String appCode, MessageCenterSceneDTO messageCenterScene, MessageCenterSendConfigDTO messageCenterSendConfig, List<EmpDTO> contacts, List<MessageCenterTemplateDTO> templates) {

        log.info("[send] saveMessageHistory start  tenantId = {},appCode = {},channel = {},templates = {},contacts = {}",
                tenantId, appCode, channelType, JsonUtils.objectToString(templates), JsonUtils.objectToString(contacts));
        MessageCenterRecordDO messageCenterRecordDO = new MessageCenterRecordDO();
        messageCenterRecordDO.setTenantId(tenantId);
        messageCenterRecordDO.setAppCode(appCode);
        messageCenterRecordDO.setChannelType(channelType);
        messageCenterRecordDO.setStatus(1L);
        messageCenterRecordDO.setMsgType(messageCenterSendConfig.getMsgType());
        if (null != messageCenterScene && null != messageCenterScene.getTriggerTiming()) {
            messageCenterRecordDO.setSceneId(messageCenterScene.getAthenaMessageId());
            messageCenterRecordDO.setSceneName(messageCenterScene.getName());
            messageCenterRecordDO.setEventId(messageCenterScene.getTriggerTiming().getEventId());
            messageCenterRecordDO.setEventName(messageCenterScene.getTriggerTiming().getEventName());
            Map<String, Map<String, String>> lang = new HashMap<>();
            if (MapUtils.isNotEmpty(messageCenterScene.getLang())) {
                lang.put("sceneName", messageCenterScene.getLang().get("name"));
            }
            if (MapUtils.isNotEmpty(messageCenterScene.getTriggerTiming().getLang())) {
                lang.put("eventName", messageCenterScene.getTriggerTiming().getLang().get("eventName"));
            }
            messageCenterRecordDO.setLang(lang);
        }
        messageCenterRecordDO.setReceivers(bulidString(contacts, channelType));

        Map<String, Object> message = new HashMap<>();

        if (CollectionUtils.isNotEmpty(templates)) {
            // 获取模板
            MessageCenterTemplateDTO template = getTemplate(LocaleContextHolder.getLocale().toString(), templates);
            message.put("title", template.getSubject());
            message.put("content", template.getContent());
            // 组装多语言
            Map<String, Map<String, String>> langMap = new HashMap<>();
            langMap.put("title", templates.stream().collect(Collectors.toMap(MessageCenterTemplateDTO::getLanguage, MessageCenterTemplateDTO::getSubject)));
            langMap.put("content", templates.stream().collect(Collectors.toMap(MessageCenterTemplateDTO::getLanguage, MessageCenterTemplateDTO::getContent)));
            message.put("lang", langMap);
            messageCenterRecordDO.setMessage(message);
        }
        messageCenterRecordDO.setCreateDate(LocalDateTime.now());
        messageCenterRecordDO.setModifyDate(LocalDateTime.now());
        log.info("[send] saveMessageHistory insert record  tenantId = {},appCode = {},channel = {},messageCenterRecordDO = {}",
                tenantId, appCode, channelType, JsonUtils.objectToString(messageCenterRecordDO));
        messageCenterMapper.insert(messageCenterRecordDO);
    }

    private void sendWeComMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig, List<EmpDTO> contacts) {

        // 处理联系人语系，处理  多语言为三类  zh_TW  zh_CN  en_US
        Map<String, List<EmpDTO>> contactsLang = contacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
        log.info("[send] sendWeComMsg start data = {},messageCenterSendConfig = {}, contactsLang = {}, contacts = {}",
                JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig), JsonUtils.objectToString(contactsLang), JsonUtils.objectToString(contacts));
        for (String lang : contactsLang.keySet()) {
            // 当前语系联系人
            List<EmpDTO> empDTOS = contactsLang.get(lang);
            // 组装联系人
            List<String> userList = bulidContactsList(empDTOS, MessageCenterConstant.MessageChannel.WE_COM);
            // 组装入参
            JSONObject req = new JSONObject();
            req.put("currentAppId", appCode);
            req.put("eventId", messageCenterConfig.getType().getId());
            List<String> types = new ArrayList<>();
            types.add(MessageCenterConstant.MessageChannel.WE_COM);
            req.put("types", types);
            Map<String, Object> reqData = new HashMap<>();
            reqData.put("userList", userList);
            String appId = getAppId();
            if (StringUtils.isBlank(appId)){
                log.info("[send] sendWeComMsg  appId is null ");
                continue;
            }
            reqData.put("appId", appId);
            reqData.put("messageData", data);
            Map<String, Object> message = new HashMap<>();
            message.put("data", reqData);
            req.put("message", message);
            log.info("[send] sendWeComMsg  lang = {},req = {}", lang, req);
            emcService.sendCommonChannelMsg(lang, req);
        }
    }

    private String getAppId() {
        ResponseEntity<JSONObject> wecomConfig = emcService.getWecomConfig();
        if (null != wecomConfig && org.apache.http.HttpStatus.SC_OK == wecomConfig.getStatusCodeValue() && wecomConfig.getBody() != null) {
            JSONArray data = wecomConfig.getBody().getJSONArray("data");
            if (data != null && data.size() > 0) {
                JSONObject jsonObject = data.getJSONObject(0);
                String appId = jsonObject.getString("appId");
                return appId;
            }
        }
        return "";
    }

    private void sendDingTalkMsg(String tenantId, String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig, List<EmpDTO> contacts) {

        // 处理联系人语系，处理  多语言为三类  zh_TW  zh_CN  en_US
        Map<String, List<EmpDTO>> contactsLang = contacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
        log.info("[send] sendDingTalkMsg start data = {},messageCenterSendConfig = {}, contactsLang = {}, contacts = {}",
                JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig), JsonUtils.objectToString(contactsLang), JsonUtils.objectToString(contacts));
        for (String lang : contactsLang.keySet()) {
            // 当前语系联系人
            List<EmpDTO> empDTOS = contactsLang.get(lang);
            // 组装联系人
            List<String> userList = bulidContactsList(empDTOS, MessageCenterConstant.MessageChannel.DING_TALK);
            // 组装入参
            JSONObject req = new JSONObject();
            req.put("currentAppId", appCode);
            req.put("eventId", messageCenterConfig.getType().getId());
            List<String> types = new ArrayList<>();
            types.add(MessageCenterConstant.MessageChannel.DING_TALK);
            req.put("types", types);
            Map<String, Object> reqData = new HashMap<>();
            reqData.put("userList", userList);
            reqData.put("appId", appCode);
            String corpid = iamService.getTenantCorpid();
            reqData.put("corpId", StringUtils.isNotBlank(corpid) ? corpid : tenantId);
            reqData.put("tenantId", tenantId);
            data.put("tenantId", StringUtils.isNotBlank(corpid) ? corpid : tenantId);
            reqData.put("messageData", data);
            Map<String, Object> message = new HashMap<>();
            message.put("data", reqData);
            req.put("message", message);
            log.info("[send] sendDingTalkMsg  lang = {},req = {}", lang, req);
            emcService.sendCommonChannelMsg(lang, req);
        }
    }


    private void sendSmsMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig, List<EmpDTO> contacts) {

        // 处理联系人语系，处理  多语言为三类  zh_TW  zh_CN  en_US
        Map<String, List<EmpDTO>> contactsLang = contacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
        log.info("[send] sendSmsMsg start data = {},messageCenterSendConfig = {}, contactsLang = {}, contacts = {}",
                JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig), JsonUtils.objectToString(contactsLang), JsonUtils.objectToString(contacts));
        for (String lang : contactsLang.keySet()) {
            // 当前语系联系人
            List<EmpDTO> empDTOS = contactsLang.get(lang);
            String contactsStr = bulidString(empDTOS, MessageCenterConstant.MessageChannel.SMS);
            JSONObject req = new JSONObject();
            req.put("currentAppId", appCode);
            req.put("eventId", messageCenterConfig.getType().getId());
            req.put("telephone", contactsStr);
            List<String> types = new ArrayList<>();
            types.add(MessageCenterConstant.MessageChannel.SMS);
            req.put("types", types);
            Map<String, Object> message = new HashMap<>();
            message.put("data", data);
            req.put("message", message);
            log.info("[send] sendSmsMsg  lang = {},req = {}", lang, req);
            emcService.sendCommonChannelMsg(lang, req);
        }
    }

    private void sendEmailMsg(String appCode, Map<String, Object> data, MessageCenterConfigDTO messageCenterConfig, List<EmpDTO> contacts, List<EmpDTO> ccContacts, List<EmpDTO> bccContacts) {

        // 处理联系人语系，处理  多语言为三类  zh_TW  zh_CN  en_US
//        Map<String, List<EmpDTO>> contactsLang = contacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
        log.info("[send] sendEmailMsg start data = {},messageCenterSendConfig = {}, contacts = {}, ccContacts = {}, bccContacts = {}",
                JsonUtils.objectToString(data), JsonUtils.objectToString(messageCenterConfig), JsonUtils.objectToString(contacts), JsonUtils.objectToString(ccContacts), JsonUtils.objectToString(bccContacts));
        // 联系人
        String contactsStr = bulidString(contacts, MessageCenterConstant.MessageChannel.EMAIL);

        // 抄送人
        String ccContactsStr = bulidString(ccContacts, MessageCenterConstant.MessageChannel.EMAIL);

        // 密送人
        String bccContactsStr = bulidString(bccContacts, MessageCenterConstant.MessageChannel.EMAIL);

//        // 是否存在抄送人
//        String ccContactsStr = "";
//        if (CollectionUtils.isNotEmpty(ccContacts)) {
//            Map<String, List<EmpDTO>> ccContactsLang = ccContacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
//            if (CollectionUtils.isNotEmpty(ccContactsLang.get(lang))) {
//                ccContactsStr = bulidString(ccContactsLang.get(lang), MessageCenterConstant.MessageChannel.EMAIL);
//            }
//        }
//        // 是否存在密送人
//        String bccContactsStr = "";
//        if (CollectionUtils.isNotEmpty(bccContacts)) {
//            Map<String, List<EmpDTO>> bccContactsLang = bccContacts.stream().collect(Collectors.groupingBy(EmpDTO::getLang));
//            if (CollectionUtils.isNotEmpty(bccContactsLang.get(lang))) {
//                bccContactsStr = bulidString(bccContactsLang.get(lang), MessageCenterConstant.MessageChannel.EMAIL);
//            }
//        }
        JSONObject req = new JSONObject();
        req.put("currentAppId", appCode);
        req.put("eventId", messageCenterConfig.getType().getId());
        req.put("contacts", contactsStr);
        req.put("ccContacts", ccContactsStr);
        req.put("bccContacts", bccContactsStr);
        List<String> types = new ArrayList<>();
        types.add(MessageCenterConstant.MessageChannel.EMAIL);
        req.put("types", types);
        Map<String, Object> message = new HashMap<>();
        message.put("data", data);
        req.put("message", message);
        // 邮件发送，按照当前系统默认语系发送
        String lang = envProperties.getDefaultLang();
        log.info("[send] sendEmailMsg  lang = {},req = {}", lang, req);
        emcService.sendCommonChannelMsg(lang, req);

    }

    private String bulidString(List<EmpDTO> empDTOS, String channel) {

        if (CollectionUtils.isEmpty(empDTOS)) {
            return "";
        }

        StringBuilder sb = new StringBuilder();
        if (MessageCenterConstant.MessageChannel.EMAIL.equals(channel)) {
            for (EmpDTO empDTO : empDTOS) {
                if (StringUtils.isNotBlank(empDTO.getEmail())) {
                    sb.append(empDTO.getEmail()).append(";");
                }
            }
        }
        if (MessageCenterConstant.MessageChannel.SMS.equals(channel)) {
            for (EmpDTO empDTO : empDTOS) {
                if (StringUtils.isNotBlank(empDTO.getTelephone())) {
                    sb.append(empDTO.getTelephone()).append(";");
                }
            }
        }

        if (MessageCenterConstant.MessageChannel.WEB.equals(channel) || MessageCenterConstant.MessageChannel.APP.equals(channel)) {
            for (EmpDTO empDTO : empDTOS) {
                if (StringUtils.isNotBlank(empDTO.getUserId())) {
                    sb.append(empDTO.getUserId()).append(";");
                }
            }
        }
        if (sb.length() == 0) {
            return "";
        }
        return sb.deleteCharAt(sb.length() - 1).toString();
    }


    private List<String> bulidContactsList(List<EmpDTO> empDTOS, String channel) {

        List<String> userList = new ArrayList<>();

        if (CollectionUtils.isEmpty(empDTOS)) {
            return userList;
        }

        if (MessageCenterConstant.MessageChannel.DING_TALK.equals(channel)) {
            empDTOS.forEach(empDTO -> {
                if (StringUtils.isNotBlank(empDTO.getUserId())) {
                    userList.add(empDTO.getUserId());
                }
            });
        }

        if (MessageCenterConstant.MessageChannel.WE_COM.equals(channel)) {
            empDTOS.forEach(empDTO -> {
                if (StringUtils.isNotBlank(empDTO.getUserId())) {
                    userList.add(empDTO.getUserId());
                }
            });
        }
        return userList;
    }

    private void saveMessage(String tenantId, String appCode, String channel, List<EmpDTO> contacts, List<MessageCenterTemplateDTO> templates, Map<String, Object> data) {

        log.info("[send] saveMessage start  tenantId = {},appCode = {},channel = {},data = {},templates = {},contacts = {}",
                tenantId, appCode, channel, JsonUtils.objectToString(data), JsonUtils.objectToString(templates), JsonUtils.objectToString(contacts));
        if (CollectionUtils.isEmpty(contacts) || CollectionUtils.isEmpty(templates)) {
            return;
        }
        List<MessageDO> messageList = new ArrayList<>();
        contacts.forEach(empDTO -> {
            if (StringUtils.isNotBlank(empDTO.getUserId())) {
                MessageDO messageDO = new MessageDO();
                messageDO.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
                messageDO.setTenantId(tenantId);
                messageDO.setAppCode(appCode);
                messageDO.setUserId(empDTO.getUserId());
                messageDO.setChannelType(channel);
                messageDO.setState(0);
                messageDO.setHasReadDetail(null != data.get("hasReadDetail") ? (Boolean) data.get("hasReadDetail") : false);
                messageDO.setLangName(empDTO.getLang());
                messageDO.setImportance(null != data.get("importance") ? Integer.parseInt(String.valueOf(data.get("importance"))) : 0);
                messageDO.setSource(null != data.get("source") ? String.valueOf(data.get("source")) : "athena");
                messageDO.setType(null != data.get("type") ? String.valueOf(data.get("type")) : "other");
                messageDO.setCategory(null != data.get("category") ? String.valueOf(data.get("category")) : "OTHER");
                messageDO.setSubType(String.valueOf(data.get("subType")));
                messageDO.setSubTypeCategory(null != data.get("subTypeCategory") ? String.valueOf(data.get("subTypeCategory")) : "OTHER");

                // 根据语系获取模板
                MessageCenterTemplateDTO template = getTemplate(empDTO.getLang(), templates);
                if (null != template) {
                    data.put("title", template.getSubject());
                    data.put("msg", template.getContent());
                    messageDO.setContent(JSONObject.fromObject(data));
                    messageDO.setCreateDate(LocalDateTime.now());
                    messageDO.setSendDate(LocalDateTime.now());
                    messageDO.setModifyDate(LocalDateTime.now());
                    messageList.add(messageDO);
                }
            }
        });
        if (CollectionUtils.isEmpty(messageList)) {
            return;
        }
        log.info("[send] saveMessage insert message  tenantId = {},appCode = {},channel = {},messageList = {}",
                tenantId, appCode, channel, JsonUtils.objectToString(messageList));
        messageMapperV2.batchInsert(messageList);

    }

    private MessageCenterTemplateDTO getTemplate(String lang, List<MessageCenterTemplateDTO> templates) {

        Optional<MessageCenterTemplateDTO> templateDTOOptional = templates.stream().filter(messageCenterTemplateDTO -> lang.equals(messageCenterTemplateDTO.getLanguage())).findFirst();

        if (templateDTOOptional.isPresent()) {
            return templateDTOOptional.get();
        }
        // 用户的语系找不到，则根据当前系统的语系再寻找一次，上面两种方式都找不到，则取第一个
        Optional<MessageCenterTemplateDTO> systemOptional = templates.stream().filter(messageCenterTemplateDTO -> envProperties.getDefaultLang().equals(messageCenterTemplateDTO.getLanguage())).findFirst();
        return systemOptional.orElseGet(() -> templates.get(0));
    }


    /**
     * 获取模板信息
     *
     * @param data
     * @param messageCenterSendConfig
     * @return
     */
    private List<MessageCenterTemplateDTO> analysisTemplate(Map<String, Object> data, MessageCenterSendConfigDTO messageCenterSendConfig) {

        List<MessageCenterTemplateDTO> templates = messageCenterSendConfig.getTemplates();

        if (CollectionUtils.isEmpty(templates)) {
            return new ArrayList<>();
        }
        // 替换常量
        return replaceTemplatesVariable(data, templates);

    }

    private List<MessageCenterTemplateDTO> replaceTemplatesVariable(Map<String, Object> data, List<MessageCenterTemplateDTO> templates) {

        for (MessageCenterTemplateDTO template : templates) {
            // 处理消息标题
            template.setSubject(templateVariableReplace(data, template.getSubject()));
            // 处理消息内容
            template.setContent(templateVariableReplace(data, template.getContent()));
        }
        return templates;
    }

    private String templateVariableReplace(Map<String, Object> data, String content) {

        if (MapUtils.isEmpty(data)) {
            return content;
        }
        // 使用正则匹配占位符 {key}
        Pattern pattern = Pattern.compile("\\{(\\w+)}");
        Matcher matcher = pattern.matcher(content);
        StringBuffer result = new StringBuffer();

        while (matcher.find()) {
            String key = matcher.group(1); // 获取占位符内部的 key
            Object replacement = data.getOrDefault(key, matcher.group(0)); // 如果找不到，就保留原始占位符
            // 处理 Date 类型
            if (replacement instanceof Date) {
                SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                replacement = dateFormat.format(replacement);
            }

            // 针对不同类型进行格式化
            if (replacement instanceof Date) {
                SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                replacement = dateFormat.format((Date) replacement);
            } else if (replacement instanceof LocalDateTime) {
                DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"); // LocalDateTime
                replacement = ((LocalDateTime) replacement).format(dateTimeFormatter);
            } else if (replacement instanceof LocalDate) {
                DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd"); // LocalDate
                replacement = ((LocalDate) replacement).format(dateFormatter);
            }
            matcher.appendReplacement(result, String.valueOf(replacement));
        }
        matcher.appendTail(result);
        return result.toString();
    }


    private Map<String, List<EmpDTO>> analysisPushConfig(String channel, Map<String, Object> data, MessageCenterSendConfigDTO messageCenterSendConfig) {

        List<MessageCenterPushConfig> pushConfig = messageCenterSendConfig.getPushConfig();
        if (CollectionUtils.isEmpty(pushConfig)) {
            return new HashMap<>();
        }
        Map<String, List<EmpDTO>> receivers = new HashMap<>();

        // 解析联系人
        List<EmpDTO> contacts = analysisContacts(MessageCenterConstant.ContactsType.CONTACTS, channel, data, pushConfig);
        receivers.put(MessageCenterConstant.ContactsType.CONTACTS, contacts);

        // 解析抄送人
        List<EmpDTO> ccContacts = analysisContacts(MessageCenterConstant.ContactsType.CC_CONTACTS, channel, data, pushConfig);
        receivers.put(MessageCenterConstant.ContactsType.CC_CONTACTS, ccContacts);

        // 解析密送人
        List<EmpDTO> bccContacts = analysisContacts(MessageCenterConstant.ContactsType.BCC_CONTACTS, channel, data, pushConfig);
        receivers.put(MessageCenterConstant.ContactsType.BCC_CONTACTS, bccContacts);

        return receivers;
    }

    private List<EmpDTO> analysisContacts(String contactsType, String channel, Map<String, Object> data, List<MessageCenterPushConfig> pushConfig) {

        Optional<MessageCenterPushConfig> messageCenterPushConfigOptional = pushConfig.stream().filter(messageCenterPushConfig -> contactsType.equals(messageCenterPushConfig.getType())).findFirst();
        if (!messageCenterPushConfigOptional.isPresent()) {
            return new ArrayList<>();
        }

        List<EmpDTO> empList = new ArrayList<>();
        Set<String> userIdList = new HashSet<>();
        List<MessageCenterPushReceiver> messageCenterPushReceivers = messageCenterPushConfigOptional.get().getReceivers();
        if (CollectionUtils.isNotEmpty(messageCenterPushReceivers)) {
            // 解析 personnel 类型
            Optional<MessageCenterPushReceiver> personnelPushReceiverOptional = messageCenterPushReceivers.stream().filter(messageCenterPushReceiver -> MessageCenterConstant.ReceiversType.PERSONNEL.equals(messageCenterPushReceiver.getClassification())).findFirst();
            if (personnelPushReceiverOptional.isPresent()) {
                List<MessageCenterPushReceiverPersonnel> receiverPersonnels = personnelPushReceiverOptional.get().getPersonnel();
                if (CollectionUtils.isNotEmpty(receiverPersonnels)) {
                    List<String> empIds = receiverPersonnels.stream().map(MessageCenterPushReceiverPersonnel::getEmpId).collect(Collectors.toList());
                    // 根据员工查询员工信息
                    List<EmpDTO> emps = eocService.getEmpByIds(null, empIds);
                    empList.addAll(emps);
                    List<String> userIds = emps.stream().map(EmpDTO::getUserId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                    userIdList.addAll(userIds);
                }

            }
            // 解析 duty 类型
            Optional<MessageCenterPushReceiver> dutyPushReceiverOptional = messageCenterPushReceivers.stream().filter(messageCenterPushReceiver -> MessageCenterConstant.ReceiversType.DUTY.equals(messageCenterPushReceiver.getClassification())).findFirst();
            if (dutyPushReceiverOptional.isPresent()) {
                List<MessageCenterPushReceiverDuty> receiverDutys = dutyPushReceiverOptional.get().getDuty();
                if (CollectionUtils.isNotEmpty(receiverDutys)) {
                    List<String> dutyIds = receiverDutys.stream().map(MessageCenterPushReceiverDuty::getDutyId).collect(Collectors.toList());
                    // 根据部门查询员工信息
                    List<EmpDTO> emps = eocService.getEmpByDutyIds(dutyIds);
                    if (CollectionUtils.isNotEmpty(emps)) {
                        empList.addAll(emps);
                        List<String> userIds = emps.stream().map(EmpDTO::getUserId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                        userIdList.addAll(userIds);
                    }
                }
            }
            // 解析 outsiders 类型
            Optional<MessageCenterPushReceiver> outsidersPushReceiverOptional = messageCenterPushReceivers.stream().filter(messageCenterPushReceiver -> MessageCenterConstant.ReceiversType.OUTSIDERS.equals(messageCenterPushReceiver.getClassification())).findFirst();
            if (outsidersPushReceiverOptional.isPresent()) {
                List<String> outsiders = outsidersPushReceiverOptional.get().getOutsiders();
                if (CollectionUtils.isNotEmpty(outsiders)) {
                    // 根据渠道构建虚拟员工数据
                    List<EmpDTO> emps = buildEmpsByChannel(outsiders, channel);
                    List<String> userIds = emps.stream().map(EmpDTO::getUserId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                    userIdList.addAll(userIds);
                    empList.addAll(emps);
                }
            }
            // 解析 variable 类型
            Optional<MessageCenterPushReceiver> variablePushReceiverOptional = messageCenterPushReceivers.stream().filter(messageCenterPushReceiver -> MessageCenterConstant.ReceiversType.VARIABLE.equals(messageCenterPushReceiver.getClassification())).findFirst();
            if (variablePushReceiverOptional.isPresent()) {
                List<MessageCenterPushReceiverVariable> receiverVariables = variablePushReceiverOptional.get().getVariable();
                if (CollectionUtils.isNotEmpty(receiverVariables)) {
                    Set<String> keySet = receiverVariables.stream().filter(messageCenterPushReceiverVariable -> StringUtils.isNotBlank(messageCenterPushReceiverVariable.getKey())).map(MessageCenterPushReceiverVariable::getKey).collect(Collectors.toSet());
                    // 获取所有variable对应的id ,此id可能是员工id，也可能是用户id
                    List<String> idList = analysisVariable(data, keySet);
                    // 根据员工id查询
                    List<EmpDTO> empByEmps = eocService.getEmpByIds(null, idList);
                    if (CollectionUtils.isNotEmpty(empByEmps)) {
                        empList.addAll(empByEmps);
                        List<String> userIds = empByEmps.stream().map(EmpDTO::getUserId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                        userIdList.addAll(userIds);
                    }
                    // 根据用户id查询
                    List<EmpDTO> empByUserList = eocService.getEmpByIds(idList, null);
                    if (CollectionUtils.isNotEmpty(empByUserList)) {
                        empList.addAll(empByUserList);
                        List<String> userIds = empByUserList.stream().map(EmpDTO::getUserId).filter(StringUtils::isNotBlank).collect(Collectors.toList());
                        userIdList.addAll(userIds);
                    }
                }
            }
        }

        // 状态启用的员工
        empList = empList.stream().filter(EmpDTO::getStatus).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(empList)) {
            return empList;
        }
        // 去除重复员工数据
        empList = empList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(EmpDTO::getId))), ArrayList::new));

        // 处理员工语系
        if (CollectionUtils.isNotEmpty(userIdList)) {
            List<PersonalizedDto> personalizedDtos = iamService.queryUserLangMetadataList(new ArrayList<>(userIdList));
            if (CollectionUtils.isNotEmpty(personalizedDtos)) {
                Map<String, String> langRel = personalizedDtos.stream().collect(Collectors.toMap(PersonalizedDto::getUserId, PersonalizedDto::getValue));
                empList.stream().forEach(empDTO -> {
                    if (StringUtils.isNotBlank(empDTO.getUserId())) {
                        empDTO.setLang(StringUtils.isNotBlank(langRel.get(empDTO.getUserId())) ? langRel.get(empDTO.getUserId()) : envProperties.getDefaultLang());
                    } else {
                        empDTO.setLang(envProperties.getDefaultLang());
                    }
                });
            }
        }
        return empList;
    }

    private List<String> analysisVariable(Map<String, Object> data, Set<String> keySet) {

        if (CollectionUtils.isEmpty(keySet) || MapUtils.isEmpty(data)) {
            return new ArrayList<>();
        }

        List<String> idList = new ArrayList<>();
        for (String key : keySet) {
            if (null != data.get(key)) {
                idList.add(String.valueOf(data.get(key)));
            }
        }
        return idList;
    }


    /**
     * // 根据渠道构建虚拟员工信息
     *
     * @param outsiders
     * @param channel
     * @return
     */
    private List<EmpDTO> buildEmpsByChannel(List<String> outsiders, String channel) {
        List<EmpDTO> emps = new ArrayList<>();
        for (String outsider : outsiders) {
            EmpDTO emp = new EmpDTO();
            emp.setId("outsider-" + outsider);
            emp.setStatus(true);
            emp.setLang(envProperties.getDefaultLang());
            if (MessageCenterConstant.MessageChannel.WEB.equals(channel) || MessageCenterConstant.MessageChannel.APP.equals(channel)) {
                emp.setUserId(outsider);
                emp.setName(outsider);
            }
            if (MessageCenterConstant.MessageChannel.SMS.equals(channel)) {
                emp.setTelephone(outsider);
                emp.setName(outsider);
            }
            if (MessageCenterConstant.MessageChannel.EMAIL.equals(channel)) {
                emp.setEmail(outsider);
                emp.setName(outsider);
            }
            if (MessageCenterConstant.MessageChannel.WE_COM.equals(channel)) {
                emp.setUserId(outsider);
                emp.setName(outsider);
            }
            if (MessageCenterConstant.MessageChannel.DING_TALK.equals(channel)) {
                emp.setUserId(outsider);
                emp.setName(outsider);
            }
            emps.add(emp);
        }
        return emps;
    }

    /**
     * 场景条件判断
     *
     * @param data
     * @param triggerCondition
     */
    private Boolean judgmentConditional(Map<String, Object> data, List<Map<String, Object>> triggerCondition) {
        if (CollectionUtils.isEmpty(triggerCondition)) {
            return true;
        }
        //  转换 条件
        List<MessageCenterConditionDTO> messageCenterConditions = transferCondition(triggerCondition);
        log.info("[send] transferCondition result = {}", JsonUtils.objectToString(messageCenterConditions));
        // 转换条件为表达式
        String expression = transferExpression(messageCenterConditions);
        log.info("[send] judgmentConditional expression = {}", expression);

        try {
            return (Boolean) ScriptUtil.getInstance().JsEvaluator(expression, "data", data);
        } catch (Exception e) {
            return false;
        }
    }


    private static String transferExpression(List<MessageCenterConditionDTO> messageCenterConditionS) {

        StringBuilder expression = new StringBuilder();
        for (MessageCenterConditionDTO condition : messageCenterConditionS) {
            String searchTablePath = "data.";
            // 并行分支的，不拼路径
            String bracket = condition.getBracket();
            String logic = condition.getLogic();
            String searchOperator = condition.getSearch_operator();
            String mode = condition.getMode();
            String searchField = condition.getSearch_field();
            List<Object> searchValue = condition.getSearch_value();
            if (StringUtils.isNotBlank(bracket)) {
                // 存在(括号
                if (bracket.equals("(")) {
                    expression.append(bracket);
                }
            }
            List<String> searchOperatorList = searchOperatorChange(searchOperator);
            if (CollectionUtils.isNotEmpty(searchOperatorList)) {
                if (searchOperatorList.size() == 1) {
                    generateExpressionByOperate(expression, searchOperatorList.get(0), searchTablePath,
                            searchField, searchValue.get(0), mode);
                } else if (searchOperatorList.size() == 2) {
                    generateExpressionByOperate(expression, searchOperatorList.get(0), searchTablePath,
                            searchField, searchValue.get(0), mode);
                    expression.append("&&").append(" ");
                    generateExpressionByOperate(expression, searchOperatorList.get(1), searchTablePath,
                            searchField, searchValue.get(1), mode);
                }
            }
            if (StringUtils.isNotBlank(bracket)) {
                // 存在)括号
                if (bracket.equals(")")) {
                    expression.append(bracket);
                }
            }
            if (StringUtils.isNotBlank(logic)) {
                logic = logicChange(logic);
                expression.append(logic).append(" ");
            }
        }
        return expression.toString();

    }

    private static List<String> searchOperatorChange(String searchOperator) {
        List<String> searchOperatorResultList = new ArrayList<>();
        switch (searchOperator) {
            case "equal": //等于
                searchOperatorResultList.add("==");
                break;
            case "greater": //大于
                searchOperatorResultList.add(">");
                break;
            case "less": //小于
                searchOperatorResultList.add("<");
                break;
            case "greater_equal": // 大于等于
                searchOperatorResultList.add(">=");
                break;
            case "less_equal": // 小于等于
                searchOperatorResultList.add("<=");
                break;
            case "like": //包含
                searchOperatorResultList.add("includes");
                break;
            case "not_like": //不包含
                searchOperatorResultList.add("not includes");
                break;
            case "not_equal": //不等于
                searchOperatorResultList.add("!=");
                break;
            case "between": // 介于
                searchOperatorResultList.add(">=");
                searchOperatorResultList.add("<=");
        }
        return searchOperatorResultList;
    }

    private static String logicChange(String logic) {
        String logicResult = "";
        if (logic.equals("AND")) {//与
            logicResult = "&&";
        } else if (logic.equals("OR")) {//或
            logicResult = "||";
        }
        return logicResult;
    }

    private static void generateExpressionByOperate(StringBuilder expression, String searchOperator, String searchTablePath,
                                                    String searchField, Object conditionValue, String type) {
        boolean numberFlag = false;
        if (conditionValue instanceof Integer || conditionValue instanceof Double
                || conditionValue instanceof Long || conditionValue instanceof BigDecimal
                || conditionValue instanceof Float || conditionValue instanceof Boolean) {
            numberFlag = true;
        } else if (StringUtils.isNotBlank(type) && "CUSTOM_CONSTANT".equals(type)) {
            numberFlag = true;
        }
        if (searchOperator.equals("includes")) {
            expression.append(searchTablePath).append(searchField).append(".indexOf('").append(conditionValue).append("') != -1").append(" ");
        } else if (searchOperator.equals("not includes")) {
            expression.append(searchTablePath).append(searchField).append(".indexOf('").append(conditionValue).append("') == -1").append(" ");
        } else {
            if (numberFlag) {
                expression.append(searchTablePath).append(searchField).append(" ").append(searchOperator).append(conditionValue).append(" ");
            } else {
                expression.append(searchTablePath).append(searchField).append(" ").append(searchOperator).append(" '").append(conditionValue).append("' ");
            }
        }
    }


    public static List<MessageCenterConditionDTO> transferCondition(List<Map<String, Object>> triggerCondition) {
        List<MessageCenterConditionDTO> searchInfo = new ArrayList<>();
        int order = 0;

        for (int i = 0; i < triggerCondition.size(); i++) {
            Map<String, Object> item = triggerCondition.get(i);

            // 获取 queryGroup 和 logic，假设它们在 Map 中有对应的键
            List<Map<String, Object>> queryGroup = (List<Map<String, Object>>) item.get("queryGroup");
            String logic = (String) item.get("logic");

            for (int j = 0; j < queryGroup.size(); j++) {
                Map<String, Object> groupItem = queryGroup.get(j);

                // search_value 统一转换为数组
                List<Object> searchValue = new ArrayList<>();
                Object searchValueObj = groupItem.get("searchValue");
                if (!(searchValueObj instanceof List)) {
                    searchValue.add(searchValueObj);
                } else {
                    searchValue = (List<Object>) searchValueObj;
                }

                MessageCenterConditionDTO searchParams = new MessageCenterConditionDTO();
                searchParams.setOrder(order++);

                // 通过 Map 获取各个字段值
                searchParams.setSearch_field((String) groupItem.get("searchField"));
                searchParams.setSearch_operator((String) groupItem.get("searchOperator"));
                searchParams.setSearch_value(searchValue);
                searchParams.setMode((String) groupItem.get("mode"));
                searchParams.setSearch_table_path((String) groupItem.get("search_table_path"));

                // 处理特定操作符
                if ("empty".equals(searchParams.getSearch_operator())) {
                    searchParams.setSearch_operator("equal");
                }
                if ("notEmpty".equals(searchParams.getSearch_operator())) {
                    searchParams.setSearch_operator("not_equal");
                }

                // 处理括号
                if (queryGroup.size() > 1) {
                    if (j == 0) {
                        searchParams.setBracket("(");
                    } else if (j == queryGroup.size() - 1) {
                        searchParams.setBracket(")");
                    }
                }

                // 处理逻辑关系
                if (j != 0) {
                    searchInfo.get(searchInfo.size() - 1).setLogic((String) groupItem.get("logic"));
                } else {
                    if (i != 0) {
                        searchInfo.get(searchInfo.size() - 1).setLogic(logic);
                    }
                }

                searchInfo.add(searchParams);
            }
        }

        return searchInfo;
    }


}
