package com.digiwin.athena.aim.domain.message.service.impl;

import com.digiwin.athena.aim.common.Constants;
import com.digiwin.athena.aim.domain.message.event.NoticeMobileAppEvent;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.model.MessageRemindDTO;
import com.digiwin.athena.aim.domain.message.model.MessageTemplateDTO;
import com.digiwin.athena.aim.domain.message.model.TemplateVariableDTO;
import com.digiwin.athena.aim.domain.message.model.TenantRouteChangeMessageDO;
import com.digiwin.athena.aim.domain.message.service.MessageSendService;
import com.digiwin.athena.aim.domain.message.service.MessageTemplateService;
import com.digiwin.athena.aim.infrastructure.mqtt.MqttGateway;
import com.digiwin.athena.aim.infrastructure.mqtt.MqttProperties;
import com.digiwin.athena.aim.infrastructure.trans.TranslateService;
import com.digiwin.athena.appcore.constant.LogConstant;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.google.common.eventbus.AsyncEventBus;

import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class MessageSendServiceImpl implements MessageSendService {

    private String[] locals = {"zh_CN", "zh_TW", "en_US"};

    @Autowired
    private MqttGateway mqttGateway;

    @Autowired
    private MqttProperties mqttConfiguration;

    @Autowired
    private MessageTemplateService messageTemplateService;

    @Resource
    private TranslateService translateService;

    @Resource
    private AsyncEventBus asyncEventBus;

    public static final String VARIABLE = "$%s$";

    @Override
    public void sendToClient(String tenantId, String userId, MessageDO payload) {
        log.info("【sendToClient-payload】：{}", payload);
        for (String local : locals) {
            String message = payload.getMessageByLanguage(local);
            String topic = mqttConfiguration.getServerName() + "/" + tenantId + "/" + userId + "/" + local;
            mqttGateway.sendToMqtt(topic, 1, message);
            LogDto logDto = new LogDto("【sendToClient】发送客户端消息处理，消息内容：" + JsonUtils.objectToString(payload), tenantId + LogConstant.TRACE_SEPARATOR + userId);
            log.info(logDto.toString());
            log.info("【sendToClient-sendToClient发送消息成功】topic：{}，message：{}", topic, message);
        }

    }

    @Override
    public void sendTenantRouteChange(String tenantId, String app, TenantRouteChangeMessageDO payload) {
        String message = JsonUtils.objectToString(payload);
        String topic = mqttConfiguration.getServerName() + "/" + tenantId + "/" + app;
        log.info("【sendToClient-TenantRouteChangeMessageDO】：{}, topic: {}", message, topic);
        mqttGateway.sendToMqtt(topic, 1, message);
        log.info("【sendToClient-TenantRouteChangeMessageDO发送消息成功】topic：{}", topic);
    }

    /**
     * @Author：SYQ
     * @Date：2022/4/11 14:47
     */
    @Override
    public void sendMessageRemindToClient(MessageRemindDTO messageRemindDTO) {
        log.info("【sendMessageRemindToClient-messageRemindDTO】：{}", messageRemindDTO);
        for (Map receiverMap : messageRemindDTO.getReceivers()) {
            for (String locale : locals) {
                String key = mqttConfiguration.getServerName() + "/" + receiverMap.get("tenantId") + "/" + receiverMap.get("userId") + "/" + locale;
                log.info("【sendMessageRemindToClient-topic-receiver】：{}，{}", key, receiverMap);
                mqttGateway.sendToMqtt(key, 1, JsonUtils.objectToString(messageRemindDTO));
                String tenantId = receiverMap.get("tenantId") == null ? "" : receiverMap.get("tenantId").toString();
                LogDto logDto = new LogDto("【sendToClient】客户端消息发送成功，receiverMap：" + receiverMap, tenantId);
                log.info(logDto.toString());
                log.info("【sendMessageRemindToClient发送消息成功：topic：{}，messageRemindDTO：{}】", key, JsonUtils.objectToString(messageRemindDTO));
            }
        }
    }

    /**
     * 批量发送消息
     *
     * @param messageList
     * @Author majfa
     */
    @Override
    public void sendMessageRemindToClient(List<MessageRemindDTO> messageList) {
        log.info("【sendMessageRemindToClient-messageRemindDTO】：{}", messageList);
        for (MessageRemindDTO messageRemindDTO : messageList) {
            this.sendMessageRemindToClient(messageRemindDTO);
        }
    }

    @Override
    public void sendToAppClient(String tenantId, String userId, MessageDO message) {

        message.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
        message.setUserId(StringUtils.isNotBlank(userId) ? userId : message.getUserId());
        message.setTenantId(StringUtils.isNotBlank(tenantId) ? tenantId : message.getTenantId());
        String msg = replaceVariable(message.getTemplate(), message.getLocale());
        if (StringUtils.isNotBlank(msg)) {
            JSONObject content = message.getContent();
            content.put("msg", msg);
        }
        // 异步通知给mobile app
        NoticeMobileAppEvent event = new NoticeMobileAppEvent(Collections.singletonList(message), MDC.getCopyOfContextMap());
        asyncEventBus.post(event);
    }

    /**
     * 如果发送的是模板消息，则需要替换模板中的变量
     *
     * @param template 模板
     * @param locale   语言
     * @return
     */
    private String replaceVariable(TemplateVariableDTO template, String locale) {
        String msg = StringUtils.EMPTY;
        if (template != null && StringUtils.isNotBlank(template.getCode()) && CollectionUtils.isNotEmpty(template.getVariableList())) {
            // 根据模板code查询模板
            List<MessageTemplateDTO> templateDTOList = messageTemplateService.queryMessageTemplateList(template.getCode());
            if (CollectionUtils.isNotEmpty(templateDTOList)) {
                msg = templateDTOList.get(0).getContent();
                for (TemplateVariableDTO.VariableDTO variableDTO : template.getVariableList()) {
                    msg = msg.replace(String.format(VARIABLE, variableDTO.getKey()), variableDTO.getValue());
                }
                // 消息转换为繁体
                if (StringUtils.isNotBlank(locale) && locale.equals(Constants.ZH_TW_LOCALE)) {
                    msg = translateService.translateText(msg, Constants.ZH_TW_LOCALE);
                }
            }
        }
        return msg;
    }
}
