package com.digiwin.athena.aim.domain.message.service.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.AsyncEventBus;

import com.digiwin.athena.aim.api.dto.ChangeMsgStateDTO;
import com.digiwin.athena.aim.api.dto.MsgSummaryDTO;
import com.digiwin.athena.aim.domain.message.event.DingDingMessageEvent;
import com.digiwin.athena.aim.domain.message.event.NoticeMobileAppEvent;
import com.digiwin.athena.aim.domain.message.event.WecomMessageEvent;
import com.digiwin.athena.aim.domain.message.model.MessageBatchUserDTO;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.model.MsgSubTypeCategoryEnum;
import com.digiwin.athena.aim.domain.message.service.MessageService;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.mongo.MongoMessageMapper;
import com.digiwin.athena.aim.infrastructure.semc.SemcService;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;

import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import static com.digiwin.athena.aim.common.Constants.MESSAGE_TYPE_ACTIVITY;
import static com.digiwin.athena.aim.common.MeesqgeConstant.SUMMARY_YES;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class MessageServiceImpl implements MessageService {
    @Autowired
    private MongoMessageMapper messageMapper;

    @Resource
    private SemcService semcService;

    @Resource
    private AsyncEventBus asyncEventBus;

    @Autowired
    private IamService iamService;

    @Autowired
    private MessageUtils messageUtils;

    @Override
    public ResponseEntity<?> newMessage(MessageDO messageDO) {
        log.info("【提交时消息提醒newMessage】：{}",messageDO);
        if (StringUtils.isBlank(messageDO.getGid())) {
            messageDO.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
        }

        MessageDO insertedMsg = messageMapper.insert(messageDO);
        if (BooleanUtils.isTrue(messageDO.getNoticeMobileApp())) {
            // 异步通知给mobile app
            NoticeMobileAppEvent event = new NoticeMobileAppEvent(Collections.singletonList(messageDO),MDC.getCopyOfContextMap());
            asyncEventBus.post(event);
        }
        //重新拼接msg
        try {
            rebuildMsg(messageDO);
        } catch (Exception e) {
            log.warn("aim重装msg内容异常：{}",e);
        }
        //异步发送钉钉消息
        List<MessageBatchUserDTO> messageBatchUserList=new ArrayList<>();
        MessageBatchUserDTO batchUserDTO=new MessageBatchUserDTO();
        List<String> userIds=new ArrayList<>();
        batchUserDTO.setTenantId(messageDO.getTenantId());
        userIds.add(messageDO.getUserId());
        batchUserDTO.setUserIdList(userIds);
        batchUserDTO.setMessage(messageDO);
        messageBatchUserList.add(batchUserDTO);
        DingDingMessageEvent dingDingEvent=new DingDingMessageEvent(messageBatchUserList, MDC.getCopyOfContextMap());
        asyncEventBus.post(dingDingEvent);

        // 判断是否发送企微
        WecomMessageEvent wecomMessageEvent = new WecomMessageEvent(messageBatchUserList, MDC.getCopyOfContextMap());
        asyncEventBus.post(wecomMessageEvent);
        return ResponseEntityWrapper.wrapperOk(insertedMsg);
    }

    private void rebuildMsg(MessageDO messageDO) {
        //查询用户的语言别
        String langName = iamService.getUserMetadataAllTenant(messageDO.getUserId(), 1 ,0);
        JSONObject content = messageDO.getContent();
        if (content!=null && !content.isEmpty() && MESSAGE_TYPE_ACTIVITY.equals(messageDO.getType())){
            JSONObject messageFiled = content.getJSONObject("messageFiled");
            //如果messageFiled字段存在则重新去拼接消息内容
            if (messageFiled!=null && !messageFiled.isEmpty()){
//                String summaryTag = messageFiled.getString("summaryTag");
                String summaryTag = MapUtils.getString(messageFiled,"summaryTag","");
//                String tmActivityName = messageFiled.getString("tmActivityName");
                String tmActivityName = MapUtils.getString(messageFiled,"tmActivityName","");
//                String endTime = messageFiled.getString("endTime");
                String endTime = MapUtils.getString(messageFiled,"endTime","");
//                String projectName = messageFiled.getString("projectName");
                String projectName = MapUtils.getString(messageFiled,"projectName","");
//                String summaryLayout = messageFiled.getString("summaryLayout");
                String summaryLayout = MapUtils.getString(messageFiled,"summaryLayout","");
                tmActivityName = StringUtils.isNotEmpty(tmActivityName)?MessageUtils.getMessageByLanguage(tmActivityName, langName):"";
                projectName = StringUtils.isNotEmpty(projectName)?MessageUtils.getMessageByLanguage(projectName, langName):"";
                summaryLayout = StringUtils.isNotEmpty(summaryLayout)?MessageUtils.getMessageByLanguage(summaryLayout, langName):"";

               String msg = MessageFormat.format(messageUtils.getMessageByLangName("message.content.msg", langName), endTime, "【" + tmActivityName+ "】" + " " + projectName);
                //拼接摘要
                if (SUMMARY_YES.equals(summaryTag)){
                    //摘要过长则截取摘要长度
                    if (StringUtils.isNotEmpty(summaryLayout) && summaryLayout.length()>60){
                        summaryLayout = summaryLayout.substring(0,60) + "...";
                    }
                    summaryLayout = summaryLayout.replaceAll("<br>"," ");
                    String msgSummary = MessageFormat.format(messageUtils.getMessageByLangName("message.content.summary", langName), summaryLayout);
                    msg = msg + msgSummary;
                }
                //重新给msg赋值
                messageDO.getContent().put("msg",msg);
                messageDO.getContent().put("emailMsg",msg);
            }
        }
    }

    @Override
    public ResponseEntity<?> getMsgUnreadCount(Integer importance, String channelType) {
        MessageDO condition = buildQryMessageDO();
        condition.setImportance(importance);
        condition.setChannelType(channelType);
        return ResponseEntityWrapper.wrapperOk(messageMapper.selectUnreadCount(condition));
    }

    @Override
    public ResponseEntity<?> getMsgChanged(Integer importance) {
        return ResponseEntityWrapper.wrapperOk();
    }

    @Override
    public ResponseEntity<?> getMsgSummary(Integer importance) {
        // 获取每个分组的未读消息数量
        MessageDO condition = buildQryMessageDO();
        condition.setImportance(importance);

        List<Map> unreadCountList = messageMapper.selectUnreadCountGroupBySubTypeCategory(condition);
        if (null == unreadCountList) {
            unreadCountList = Lists.newArrayList();
        }

        // 获取每个分组的最新一条数据
        List<MessageDO> latestMessageList = messageMapper.selectLatestGroupBySubTypeCategory(condition);

        List<MsgSummaryDTO> msgSummaryList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(latestMessageList)) {
            return ResponseEntityWrapper.wrapperOk(msgSummaryList);
        }

        for (MessageDO messageDO : latestMessageList) {
            Map<String, Object> msgUnreadMap = Maps.newHashMap();
            for (Map<String, Object> map : unreadCountList) {
                if (StringUtils.equals(messageDO.getSubTypeCategory(), String.valueOf(map.get("subTypeCategory")))) {
                    msgUnreadMap = map;
                    break;
                }
            }

            MsgSummaryDTO msgSummaryDTO = new MsgSummaryDTO();

            if (MapUtils.isNotEmpty(msgUnreadMap)) {
                int unreadCount = Integer.parseInt(String.valueOf(msgUnreadMap.get("unreadCount")));
                msgSummaryDTO.setUnreadCount(unreadCount);
            } else {
                msgSummaryDTO.setUnreadCount(0);
            }

            msgSummaryDTO.setType(messageDO.getType());
            msgSummaryDTO.setSubType(messageDO.getSubType());
            msgSummaryDTO.setSubTypeCategory(messageDO.getSubTypeCategory());
            msgSummaryDTO.setCategory(messageDO.getCategory());
            msgSummaryDTO.setMessage(messageDO);

            msgSummaryList.add(msgSummaryDTO);
        }

        Collections.sort(msgSummaryList, (obj1, obj2) -> {
            if (MsgSubTypeCategoryEnum.customValueOf(obj1.getSubTypeCategory()).getOrder() <= MsgSubTypeCategoryEnum.customValueOf(obj2.getSubTypeCategory()).getOrder()) {
                return -1;
            } else {
                return 1;
            }
        });

        return ResponseEntityWrapper.wrapperOk(msgSummaryList);
    }

    @Override
    public ResponseEntity<?> getMsgSummaryBySubTypeCategory(String subTypeCategory, Integer importance) {
        MessageDO condition = buildQryMessageDO();
        condition.setSubTypeCategory(subTypeCategory);
        condition.setImportance(importance);

        return ResponseEntityWrapper.wrapperOk(messageMapper.selectMsgSummaryBySubTypeCategory(condition));
    }

    @Override
    public ResponseEntity<?> getWorkNewsMsgSummary() {
        MessageDO condition = buildQryMessageDO();
        return ResponseEntityWrapper.wrapperOk(messageMapper.selectWorkNewsMsgSummary(condition));
    }

    @Override
    public ResponseEntity<?> changeStateToRead(ChangeMsgStateDTO changeMsgStateDTO) {
        MessageDO condition = buildQryMessageDO();
        condition.setImportance(changeMsgStateDTO.getImportance());
        condition.setSubTypeCategory(changeMsgStateDTO.getSubTypeCategory());

        return ResponseEntityWrapper.wrapperOk(messageMapper.changeStateToRead(condition));
    }

    @Override
    public ResponseEntity<?> deleteReadMsg(ChangeMsgStateDTO changeMsgStateDTO) {
        MessageDO condition = buildQryMessageDO();
        condition.setImportance(changeMsgStateDTO.getImportance());
        condition.setSubTypeCategory(changeMsgStateDTO.getSubTypeCategory());

        return ResponseEntityWrapper.wrapperOk(messageMapper.deleteReadMsg(condition));
    }

    private MessageDO buildQryMessageDO() {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();

        MessageDO condition = new MessageDO();
        condition.setUserId(user.getUserId());
        condition.setTenantId(user.getTenantId());
        return condition;
    }
}
