package com.digiwin.athena.aim.domain.message.service.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.AsyncEventBus;

import com.digiwin.athena.aim.api.dto.MessagePagingQryCommand;
import com.digiwin.athena.aim.common.Constants;
import com.digiwin.athena.aim.common.MeesqgeConstant;
import com.digiwin.athena.aim.common.MessageChannelEnum;
import com.digiwin.athena.aim.domain.message.event.DingDingMessageEvent;
import com.digiwin.athena.aim.domain.message.event.LineMessageEvent;
import com.digiwin.athena.aim.domain.message.event.NoticeMobileAppEvent;
import com.digiwin.athena.aim.domain.message.event.WecomMessageEvent;
import com.digiwin.athena.aim.domain.message.model.AppExpireMessageDO;
import com.digiwin.athena.aim.domain.message.model.ExpireMessageReq;
import com.digiwin.athena.aim.domain.message.model.MessageBatchUserDTO;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.model.MessageRemindDTO;
import com.digiwin.athena.aim.domain.message.model.MessageTemplateDTO;
import com.digiwin.athena.aim.domain.message.model.MsgAndEmailBatchDTO;
import com.digiwin.athena.aim.domain.message.model.MsgSendTypeEnum;
import com.digiwin.athena.aim.domain.message.model.SendCommonChannelMsgReq;
import com.digiwin.athena.aim.domain.message.model.TemplateVariableDTO;
import com.digiwin.athena.aim.domain.message.model.UserMsgAndEmailDTO;
import com.digiwin.athena.aim.domain.message.repository.MessageMapperV2;
import com.digiwin.athena.aim.domain.message.service.MessageSendService;
import com.digiwin.athena.aim.domain.message.service.MessageServiceV2;
import com.digiwin.athena.aim.domain.message.service.MessageTemplateService;
import com.digiwin.athena.aim.infrastructure.emc.EmcService;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.iam.dto.IamUserDTO;
import com.digiwin.athena.aim.infrastructure.iam.dto.PersonalizedDto;
import com.digiwin.athena.aim.infrastructure.semc.SemcService;
import com.digiwin.athena.aim.infrastructure.semc.dto.GetSsoUrlReq;
import com.digiwin.athena.aim.infrastructure.semc.dto.GetSsoUrlResp;
import com.digiwin.athena.aim.infrastructure.semc.dto.QueryBlackWhiteListResp;
import com.digiwin.athena.aim.infrastructure.semc.dto.QueryTemplateListResp;
import com.digiwin.athena.aim.infrastructure.semc.dto.QueryTemplateRelResp;
import com.digiwin.athena.aim.infrastructure.trans.TranslateService;
import com.digiwin.athena.aim.util.ValidateUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;

import net.sf.json.JSONObject;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * MessageServiceV2Impl Description
 *
 * @author majianfu
 * @date 2021/4/25
 * @since
 */
@Slf4j
@Service
public class MessageServiceV2Impl implements MessageServiceV2 {
    @Autowired
    private MessageMapperV2 messageMapperV2;

    @Resource
    private AsyncEventBus asyncEventBus;

    @Resource
    private TranslateService translateService;

    @Resource
    private SemcService semcService;

    @Resource
    private IamService iamService;

    @Resource
    private EmcService emcService;

    @Autowired
    private MessageTemplateService messageTemplateService;

    @Autowired
    private MessageSendService messageSendService;

    public static final String VARIABLE = "$%s$";

    @Autowired
    private MessageUtils messageUtils;

    /**
     * 分页查询消息
     *
     * @param command
     * @return
     */
    @Override
    public ResponseEntity<?> pagingQueryMessage(MessagePagingQryCommand command) {
        ValidateUtils.checkPageParam(command);
        MessageDO condition = buildQryMessageDO();
        condition.setType(command.getType());
        condition.setSubType(command.getSubType());
        condition.setSubTypeCategory(command.getSubTypeCategory());
        condition.setCategory(command.getCategory());
        condition.setImportance(command.getImportance());
        condition.setSource(command.getSource());
        condition.setState(command.getState());
        condition.setUnterminate(command.getUnterminate());
        condition.setChannelType(command.getChannelType());
        condition.setStartTimeStr(command.getStartTimeStr());
        condition.setEndTimeStr(command.getEndTimeStr());
        condition.setKey(command.getKey());
        return ResponseEntityWrapper.wrapperOk(messageMapperV2.pagingQuery(condition, command));
    }

    @Override
    public ResponseEntity<?> summaryRemove(String channelType) {
        // 查询租户下用户的消息 更改deleted字段 最多一千条
        return ResponseEntityWrapper.wrapperOk(messageMapperV2.updateDeleted(channelType, AppAuthContextHolder.getContext().getAuthoredUser()));
    }


    @Override
    public ResponseEntity<?> readMessageByGid(List<String> gidList) {
        if (CollectionUtils.isEmpty(gidList)) {
            return ResponseEntityWrapper.wrapperOk(0);
        }
        List<String> tmpGidList = gidList.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(tmpGidList)) {
            return ResponseEntityWrapper.wrapperOk(0);
        }

        return ResponseEntityWrapper.wrapperOk(messageMapperV2.readMessageByGid(null, gidList, AppAuthContextHolder.getContext().getAuthoredUser()));
    }

    /**
     * 一键已读当用户的工作提醒
     *
     * @return 返回
     */
    @Override
    public ResponseEntity<?> readMessageAll(String channelType) {
        return ResponseEntityWrapper.wrapperOk(messageMapperV2.readMessageByGid(channelType,null, AppAuthContextHolder.getContext().getAuthoredUser()));
    }

    @Override
    public ResponseEntity<?> readMessageDetailByGid(String gid) {
        if (StringUtils.isBlank(gid)) {
            return ResponseEntityWrapper.wrapperOk();
        }

        return ResponseEntityWrapper.wrapperOk(messageMapperV2.readMessageDetailByGid(gid, AppAuthContextHolder.getContext().getAuthoredUser()));
    }

    @Override
    public void newMessageForBatchUsers(List<MessageBatchUserDTO> messageBatchUserList) {
        List<MessageDO> newMessageList = new ArrayList<>();
        List<MessageDO> mobileMsgList = new ArrayList<>();
        List<MessageRemindDTO> onlineMsgList = new ArrayList<>();

        for (MessageBatchUserDTO messageBatchUser : messageBatchUserList) {
            MessageDO message = messageBatchUser.getMessage();
            // 如果是模板消息，则替换模板中的变量
            String msg = replaceVariable(message.getTemplate(), message.getLocale());
            for (String userId : messageBatchUser.getUserIdList()) {
                MessageDO newMessage = new MessageDO();
                BeanUtils.copyProperties(message, newMessage);
                newMessage.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
                newMessage.setUserId(userId);
                newMessage.setTenantId(messageBatchUser.getTenantId());
                // 模板消息
                if (StringUtils.isNotBlank(msg)) {
                    JSONObject content = newMessage.getContent();
                    content.put("msg", msg);
                }
                newMessageList.add(newMessage);
                // 推送消息给APP
                if (BooleanUtils.isTrue(message.getNoticeMobileApp())) {
                    mobileMsgList.add(newMessage);
                }
            }

            // 推送消息给在线用户
            if (BooleanUtils.isTrue(message.getNoticeOnlineUser())) {
                MessageRemindDTO remindDTO = new MessageRemindDTO();
                remindDTO.setType("task"); // 写死task是临时为了便于前端展示，前端需要效果，时间比较赶，后面改成 message.getType()
                remindDTO.setSource(message.getSource());
                JSONObject content = message.getContent();
                remindDTO.setTitle(StringUtils.isNotBlank(content.getString("title")) ? content.getString("title") : "");
                remindDTO.setMsg(StringUtils.isNotBlank(content.getString("msg")) ? content.getString("msg") : "");
                List<Map> userList = messageBatchUser.getUserIdList().stream().map(x -> {
                    Map<String, String> map = Maps.newHashMap();
                    map.put("userId", x);
                    map.put("tenantId", messageBatchUser.getTenantId());
                    return map;
                }).collect(Collectors.toList());
                remindDTO.setReceivers(userList);
                onlineMsgList.add(remindDTO);
            }
        }
        this.messageMapperV2.batchInsert(newMessageList);
        log.info("【提交时消息提醒needNotifyMsgList】:{}", mobileMsgList.size());
        if (CollectionUtils.isNotEmpty(mobileMsgList)) {
            // 异步通知给mobile app
            NoticeMobileAppEvent event = new NoticeMobileAppEvent(mobileMsgList,MDC.getCopyOfContextMap());
            asyncEventBus.post(event);
        }

        // 通知在线用户
        if (CollectionUtils.isNotEmpty(onlineMsgList)) {
            messageSendService.sendMessageRemindToClient(onlineMsgList);
        }

        //异步发送钉钉消息
        DingDingMessageEvent dingDingEvent = new DingDingMessageEvent(messageBatchUserList, MDC.getCopyOfContextMap());
        asyncEventBus.post(dingDingEvent);

        // 异步发送企微
        WecomMessageEvent wecomMessageEvent = new WecomMessageEvent(messageBatchUserList, MDC.getCopyOfContextMap());
        asyncEventBus.post(wecomMessageEvent);

        // 异步发送line
        if (("ASA".equalsIgnoreCase(messageBatchUserList.get(0).getMessage().getSource()) && Constants.MESSAGE_TYPE_TEXT.equalsIgnoreCase(messageBatchUserList.get(0).getMessage().getType())) || "AGILE_DATA".equalsIgnoreCase(messageBatchUserList.get(0).getMessage().getSource())) {
            // 娜娜消息任务(type=text,source=ASA) 或 敏捷数据消息（source=AGILE_DATA）
            LineMessageEvent lineMessageEvent = new LineMessageEvent(messageBatchUserList, MDC.getCopyOfContextMap());
            asyncEventBus.post(lineMessageEvent);
        }
    }

    /**
     * 如果发送的是模板消息，则需要替换模板中的变量
     *
     * @param template 模板
     * @param locale   语言
     * @return
     */
    private String replaceVariable(TemplateVariableDTO template, String locale) {
        String msg = StringUtils.EMPTY;
        if (template != null && StringUtils.isNotBlank(template.getCode()) && CollectionUtils.isNotEmpty(template.getVariableList())) {
            // 根据模板code查询模板
            List<MessageTemplateDTO> templateDTOList = messageTemplateService.queryMessageTemplateList(template.getCode());
            if (CollectionUtils.isNotEmpty(templateDTOList)) {
                msg = templateDTOList.get(0).getContent();
                for (TemplateVariableDTO.VariableDTO variableDTO : template.getVariableList()) {
                    msg = msg.replace(String.format(VARIABLE, variableDTO.getKey()), variableDTO.getValue());
                }
                // 消息转换为繁体
                if (StringUtils.isNotBlank(locale) && locale.equals(Constants.ZH_TW_LOCALE)) {
                    msg = translateService.translateText(msg, Constants.ZH_TW_LOCALE);
                }
            }
        }
        return msg;
    }

    private MessageDO buildQryMessageDO() {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        MessageDO condition = new MessageDO();
        condition.setUserId(user.getUserId());
        condition.setTenantId(user.getTenantId());
        return condition;
    }

    /**
     * 应用到期提醒 批量发送消息
     *
     * @param expireMessageReq 消息
     */
    @Override
    public void insertAppExpirationMessage(ExpireMessageReq expireMessageReq) {
        ExpireMessageReq.MessageDTO message = expireMessageReq.getMessage();
        // 如果是模板消息，则替换模板中的变量
        String msg = replaceVariable(message.getTemplate(), message.getLocale());
        List<ExpireMessageReq.MessageDTO> messageDOList = Lists.newArrayList();
        for (String userId : expireMessageReq.getUserIdList()) {
            ExpireMessageReq.MessageDTO newMessage = new ExpireMessageReq.MessageDTO();
            BeanUtils.copyProperties(message, newMessage);
            newMessage.setGid(String.valueOf(SnowflakeIdWorker.getInstance().newId()));
            newMessage.setUserId(userId);
            newMessage.setTenantId(expireMessageReq.getTenantId());
            // 模板消息
            if (StringUtils.isNotBlank(msg)) {
                JSONObject content = new JSONObject();
                content.put("msg", msg);
                newMessage.setContent(content);
            }
            messageDOList.add(newMessage);
        }
        this.messageMapperV2.insertAppExpirationMessage(messageDOList);
    }

    /**
     * 应用到期提醒 查询消息列表
     *
     * @param platformFlag 调用来源标识
     * @return
     */
    @Override
    public List<MessageDO> queryAppExpirationMessage(Integer platformFlag) {
        MessageDO condition = buildQryMessageDO();
        List<MessageDO> messageDOList = messageMapperV2.queryAppExpirationMessage(condition);
        if (platformFlag != null && platformFlag.equals(Constants.INVOKE_PLATFORM_FLAG)) {
            String locale = LocaleContextHolder.getLocale().toString();
            messageDOList.forEach(x -> {
                JSONObject content = x.getContent();
                if (content != null && StringUtils.isNotBlank(content.getString("msg"))) {
                    String msg = translateService.translateText(content.getString("msg"), locale);
                    content.put("msg", msg);
                }
            });
        }
        return messageDOList;
    }

    /**
     * 应用到期提醒 应用续约，即删除该应用的到期提醒
     *
     * @param appExpireMessageDO 请求体
     * @return
     */
    @Override
    public void deleteAppExpirationMessage(AppExpireMessageDO appExpireMessageDO) {
        this.messageMapperV2.deleteAppExpirationMessage(appExpireMessageDO);
    }

    /**
     * 各渠道发送消息
     *
     * @param channelFlag    email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReqList 请求消息体
     */
    @Override
    public JSONObject sendCommonChannelMessage(String channelFlag, List<SendCommonChannelMsgReq.MsgInfoReq> msgInfoReqList) {
        // 发送邮件
        JSONObject result = new JSONObject();
        if (MessageChannelEnum.EMAIL.getFlag().equals(channelFlag)) {
            result = sendCommonEmailMessage(channelFlag, msgInfoReqList);
        }
        // 发送短信
        else if (MessageChannelEnum.SMS.getFlag().equals(channelFlag)) {
            result = sendCommonSmsMessage(channelFlag, msgInfoReqList);
        }
        // 发送企微
        else if (MessageChannelEnum.WECOM.getFlag().equals(channelFlag)) {
            result = sendCommonWeComMessage(channelFlag, msgInfoReqList);
        }
        // 发送钉钉
        else if (MessageChannelEnum.DINGTALK.getFlag().equals(channelFlag)) {
            result = sendCommonDingTalkMessage(channelFlag, msgInfoReqList);
        }
        return result;
    }

    /**
     * 发送邮件
     *
     * @param channelFlag    email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReqList 请求消息体
     */
    private JSONObject sendCommonEmailMessage(String channelFlag, List<SendCommonChannelMsgReq.MsgInfoReq> msgInfoReqList) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("errorCount", 0);
        jsonObject.put("errorMessage", "");
        for (int i = 0; i < msgInfoReqList.size(); i++) {
            SendCommonChannelMsgReq.MsgInfoReq msgInfoReq = msgInfoReqList.get(i);
            try {
                // 事件和文本必须二传一
                String msg = validRequiredParam(msgInfoReq, channelFlag);
                if (StringUtils.isNotBlank(msg)) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", msg);
                    break;
                }
                // 租户id不能为空
                if (StringUtils.isBlank(msgInfoReq.getTenantId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send email message param of tenantId is missing");
                    break;
                }
                // 接收人必传
                if (CollectionUtils.isEmpty(msgInfoReq.getUserIdList()) && CollectionUtils.isEmpty(msgInfoReq.getCcUserIdList())
                        && CollectionUtils.isEmpty(msgInfoReq.getBccUserIdList())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send email message param of userIdList is empty");
                    break;
                }

                // 构造邮件发送参数
                JSONObject msgJson = new JSONObject();
                msgJson.put("types", Collections.singletonList(channelFlag));
                if (CollectionUtils.isNotEmpty(msgInfoReq.getFileIdList())) {
                    msgJson.put("dmcAttachments", msgInfoReq.getFileIdList());
                    msgJson.put("dmcBucket", Constants.INTELLIGENTENTRY_APP_ID);
                }
                // 查询黑白名单用户
                List<QueryBlackWhiteListResp> blackWhiteList = semcService.queryBlackWhiteUserList(msgInfoReq.getTenantId(), null, null);
                // 如果传了事件id，则查询智能入口后台配置 1、租户是否启用了该事件 2、是否配置了接收人
                Map<String, Map<String, List<String>>> emailMap;
                if (msgInfoReq.getEvent() != null && StringUtils.isNotBlank(msgInfoReq.getEvent().getEventId())) {
                    // 校验事件是否存在且是否启用
                    List<QueryTemplateListResp> templateList = semcService.queryTemplateList(channelFlag, msgInfoReq.getEvent().getEventId(), msgInfoReq.getEvent().getRemark(), msgInfoReq.getTenantId());
                    if (CollectionUtils.isEmpty(templateList)) {
                        log.error("MessageControllerV2 | send common message eventId is not exist:{}, channelFlag:{}", msgInfoReq, channelFlag);
                        jsonObject.put("errorCount", msgInfoReqList.size() - i);
                        jsonObject.put("errorMessage", "send common message eventId is not exist:" + msgInfoReq.getEvent().getEventId() + ", channelFlag:" + channelFlag);
                        break;
                    }
                    if (templateList.get(0).getValidStatus().equals(Constants.VALID_STATUS_UNUSABLE)) {
                        log.error("MessageControllerV2 | send common message eventId is disabled:{}, channelFlag:{}", msgInfoReq, channelFlag);
                        jsonObject.put("errorCount", msgInfoReqList.size() - i);
                        jsonObject.put("errorMessage", "send common message eventId is disabled:" + msgInfoReq.getEvent().getEventId() + ", channelFlag:" + channelFlag);
                        break;
                    }

                    msgJson.put("eventId", msgInfoReq.getEvent().getEventId());
                    JSONObject emailMsg = new JSONObject();
                    emailMsg.put("data", msgInfoReq.getEvent().getVariableData());
                    msgJson.put("message", emailMsg);
                    // 构建邮件收件人
                    emailMap = buildMessageReceiver(channelFlag, msgInfoReq, blackWhiteList, templateList.get(0).getId());
                } else { // 如果没传事件id，则以默认事件发送
                    msgJson.put("eventId", Constants.SEMC_MESSAGE_DEFAULT_EVENT_ID);
                    JSONObject defaultJson = new JSONObject();
                    defaultJson.put("defaultVarKey", msgInfoReq.getText());
                    JSONObject emailMsg = new JSONObject();
                    emailMsg.put("data", defaultJson);
                    msgJson.put("message", emailMsg);
                    // 构建邮箱收件人
                    emailMap = buildEmail(channelFlag, msgInfoReq, blackWhiteList);
                }
                // 按语系循环发送，因为DAP接口不支持一组发送
                sendGroupEmail(msgJson, emailMap);
            } catch (Exception e) {
                jsonObject.put("errorCount", msgInfoReqList.size() - i);
                jsonObject.put("errorMessage", e.getMessage());
                break;
            }
        }
        return jsonObject;
    }

    /**
     * 按语系循环发送，因为DAP接口不支持一组发送
     *
     * @param msgJson  消息体
     * @param emailMap 收件人
     * @return 结果
     */
    private void sendGroupEmail(JSONObject msgJson, Map<String, Map<String, List<String>>> emailMap) {
        // 按多语言循环发送
        for (Map.Entry<String, Map<String, List<String>>> stringMapEntry : emailMap.entrySet()) {
            String language = stringMapEntry.getKey();
            Map<String, List<String>> emailUserList = stringMapEntry.getValue();

            // 收件人
            if (CollectionUtils.isNotEmpty(emailUserList.get("email"))) {
                List<String> email = emailUserList.get("email").stream().distinct().collect(Collectors.toList());
                msgJson.put("contacts", StringUtils.join(email, ";"));
            } else {
                msgJson.put("contacts", ";");
            }
            // 抄送人
            if (CollectionUtils.isNotEmpty(emailUserList.get("ccEmail"))) {
                List<String> email = emailUserList.get("ccEmail").stream().distinct().collect(Collectors.toList());
                msgJson.put("ccContacts", StringUtils.join(email, ";"));
            } else {
                msgJson.put("ccContacts", ";");
            }
            // 密送人
            if (CollectionUtils.isNotEmpty(emailUserList.get("bccEmail"))) {
                List<String> email = emailUserList.get("bccEmail").stream().distinct().collect(Collectors.toList());
                msgJson.put("bccContacts", StringUtils.join(email, ";"));
            } else {
                msgJson.put("bccContacts", ";");
            }
            emcService.sendCommonChannelMsg(language, msgJson);
        }
    }

    /**
     * 发送短信
     *
     * @param channelFlag    email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReqList 请求消息体
     */
    private JSONObject sendCommonSmsMessage(String channelFlag, List<SendCommonChannelMsgReq.MsgInfoReq> msgInfoReqList) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("errorCount", 0);
        jsonObject.put("errorMessage", "");
        for (int i = 0; i < msgInfoReqList.size(); i++) {
            SendCommonChannelMsgReq.MsgInfoReq msgInfoReq = msgInfoReqList.get(i);
            try {
                // 事件必须传
                if (msgInfoReq == null || StringUtils.isBlank(msgInfoReq.getEvent().getEventId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send sms message param of eventId is missing");
                    break;
                }
                // 租户id不能为空
                if (StringUtils.isBlank(msgInfoReq.getTenantId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send sms message param of tenantId is missing");
                    break;
                }
                // 接收人必传
                if (CollectionUtils.isEmpty(msgInfoReq.getUserIdList())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send sms message param of userIdList is empty");
                    break;
                }
                // 校验事件是否存在且是否启用
                String msg = validEvent(channelFlag, msgInfoReq);
                if (StringUtils.isNotBlank(msg)) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", msg);
                    break;
                }

                // 查询黑白名单用户
                List<QueryBlackWhiteListResp> blackWhiteList = semcService.queryBlackWhiteUserList(msgInfoReq.getTenantId(), null, null);
                // 查询智能入口后台配置的接收人
                Map<String, Map<String, List<String>>> emailMap = buildMessageReceiver(channelFlag, msgInfoReq, blackWhiteList, null);

                // 构造发送参数
                JSONObject msgJson = new JSONObject();
                JSONObject dataJsonObject = new JSONObject();
                dataJsonObject.put("data", msgInfoReq.getEvent().getVariableData());
                msgJson.put("message", dataJsonObject);
                msgJson.put("types", Collections.singletonList(channelFlag));
                msgJson.put("eventId", msgInfoReq.getEvent().getEventId());

                // 按语系循环发送，因为DAP接口不支持一组发送
                for (Map.Entry<String, Map<String, List<String>>> stringMapEntry : emailMap.entrySet()) {
                    String language = stringMapEntry.getKey();
                    Map<String, List<String>> telephoneMap = stringMapEntry.getValue();
                    if (CollectionUtils.isNotEmpty(telephoneMap.get("telephone"))) {
                        List<String> telephoneList = telephoneMap.get("telephone").stream().distinct().collect(Collectors.toList());
                        msgJson.put("telephone", StringUtils.join(telephoneList, ";"));
                    } else {
                        msgJson.put("telephone", ";");
                    }
                    emcService.sendCommonChannelMsg(language, msgJson);
                }
            } catch (Exception e) {
                jsonObject.put("errorCount", msgInfoReqList.size() - i);
                jsonObject.put("errorMessage", e.getMessage());
                break;
            }
        }
        return jsonObject;
    }

    /**
     * 发送企微
     *
     * @param channelFlag    email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReqList 请求消息体
     */
    private JSONObject sendCommonWeComMessage(String channelFlag, List<SendCommonChannelMsgReq.MsgInfoReq> msgInfoReqList) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("errorCount", 0);
        jsonObject.put("errorMessage", "");
        for (int i = 0; i < msgInfoReqList.size(); i++) {
            SendCommonChannelMsgReq.MsgInfoReq msgInfoReq = msgInfoReqList.get(i);
            try {
                // 事件id和文本必须二传一
                String msg = validRequiredParam(msgInfoReq, channelFlag);
                if (StringUtils.isNotBlank(msg)) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", msg);
                    break;
                }
                // 租户id不能为空
                if (StringUtils.isBlank(msgInfoReq.getTenantId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send wecom message param of tenantId is missing");
                    break;
                }
                // 接收人必传
                if (CollectionUtils.isEmpty(msgInfoReq.getUserIdList())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send wecom message param of userIdList is empty");
                    break;
                }

                // 查询黑白名单用户
                List<QueryBlackWhiteListResp> blackWhiteList = semcService.queryBlackWhiteUserList(msgInfoReq.getTenantId(), null, null);
                // 构造发送参数
                JSONObject msgJson = new JSONObject();
                msgJson.put("types", Collections.singletonList("weCom"));
                // 如果传了事件id，则查询智能入口后台配置 1、租户是否启用了该事件 2、是否配置了接收人
                if (msgInfoReq.getEvent() != null && StringUtils.isNotBlank(msgInfoReq.getEvent().getEventId())) {
                    msg = validEvent(channelFlag, msgInfoReq);
                    if (StringUtils.isNotBlank(msg)) {
                        jsonObject.put("errorCount", msgInfoReqList.size() - i);
                        jsonObject.put("errorMessage", msg);
                        break;
                    }
                    // 构建邮箱收件人
                    Map<String, Map<String, List<String>>> userMap = buildMessageReceiver(channelFlag, msgInfoReq, blackWhiteList, null);
                    msgJson.put("eventId", msgInfoReq.getEvent().getEventId());
                    JSONObject dataJson = new JSONObject();
                    dataJson.put("messageData", msgInfoReq.getEvent().getVariableData());

                    // 按语系循环发送，因为DAP接口不支持一组发送
                    for (Map.Entry<String, Map<String, List<String>>> stringMapEntry : userMap.entrySet()) {
                        String language = stringMapEntry.getKey();
                        Map<String, List<String>> map = stringMapEntry.getValue();
                        if (CollectionUtils.isNotEmpty(map.get("userList"))) {
                            List<String> userIdList = map.get("userList").stream().distinct().collect(Collectors.toList());
                            dataJson.put("userList", userIdList);
                        } else {
                            dataJson.put("userList", Lists.newArrayList());
                        }
                        JSONObject dataJsonObject = new JSONObject();
                        dataJsonObject.put("data", dataJson);
                        msgJson.put("message", dataJsonObject);
                        emcService.sendCommonChannelMsg(language, msgJson);
                    }
                } else { // 如果没传事件id，则以默认事件发送
                    msgJson.put("eventId", Constants.SEMC_MESSAGE_DEFAULT_EVENT_ID);

                    JSONObject defaultJson = new JSONObject();
                    JSONObject varData = new JSONObject();
                    varData.put("defaultVarKey", msgInfoReq.getText());
                    defaultJson.put("messageData", varData);

                    // 构建邮箱收件人
                    Map<String, Map<String, List<String>>> userMap = buildUserIdOrTelephone(channelFlag, msgInfoReq, blackWhiteList);
                    // 按语系循环发送，因为DAP接口不支持一组发送
                    for (Map.Entry<String, Map<String, List<String>>> stringMapEntry : userMap.entrySet()) {
                        String language = stringMapEntry.getKey();
                        Map<String, List<String>> map = stringMapEntry.getValue();
                        if (CollectionUtils.isNotEmpty(map.get("userList"))) {
                            List<String> userIdList = map.get("userList").stream().distinct().collect(Collectors.toList());
                            defaultJson.put("userList", userIdList);
                        } else {
                            defaultJson.put("userList", Lists.newArrayList());
                        }
                        JSONObject dataJsonObject = new JSONObject();
                        dataJsonObject.put("data", defaultJson);
                        msgJson.put("message", dataJsonObject);
                        emcService.sendCommonChannelMsg(language, msgJson);
                    }
                }
            } catch (Exception e) {
                jsonObject.put("errorCount", msgInfoReqList.size() - i);
                jsonObject.put("errorMessage", e.getMessage());
                break;
            }
        }
        return jsonObject;
    }

    /**
     * 发送钉钉
     *
     * @param channelFlag    email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReqList 请求消息体
     */
    private JSONObject sendCommonDingTalkMessage(String channelFlag, List<SendCommonChannelMsgReq.MsgInfoReq> msgInfoReqList) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("errorCount", 0);
        jsonObject.put("errorMessage", "");
        for (int i = 0; i < msgInfoReqList.size(); i++) {
            SendCommonChannelMsgReq.MsgInfoReq msgInfoReq = msgInfoReqList.get(i);
            try {
                // 事件id必须传
                if (msgInfoReq.getEvent() == null || StringUtils.isBlank(msgInfoReq.getEvent().getEventId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send dingTalk message param of eventId is missing");
                    break;
                }
                // 租户id不能为空
                if (StringUtils.isBlank(msgInfoReq.getTenantId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send dingTalk message param of tenantId is missing");
                    break;
                }
                // appId不能为空
                if (StringUtils.isBlank(msgInfoReq.getAppId())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send dingTalk message param of appId is missing");
                    break;
                }
                // 接收人必传
                if (CollectionUtils.isEmpty(msgInfoReq.getUserIdList())) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", "send dingTalk message param of userIdList is empty");
                    break;
                }
                // 校验事件是否存在且是否启用
                String msg = validEvent(channelFlag, msgInfoReq);
                if (StringUtils.isNotBlank(msg)) {
                    jsonObject.put("errorCount", msgInfoReqList.size() - i);
                    jsonObject.put("errorMessage", msg);
                    break;
                }
                // 查询黑白名单用户
                List<QueryBlackWhiteListResp> blackWhiteList = semcService.queryBlackWhiteUserList(msgInfoReq.getTenantId(), null, null);

                // 查询智能入口后台配置的接收人
                Map<String, Map<String, List<String>>> userMap = buildMessageReceiver(channelFlag, msgInfoReq, blackWhiteList, null);
                // 构造发送参数
                JSONObject msgJson = new JSONObject();
                msgJson.put("types", Collections.singletonList(channelFlag));
                msgJson.put("eventId", msgInfoReq.getEvent().getEventId());
                JSONObject dataJson = new JSONObject();
                dataJson.put("appId", msgInfoReq.getAppId());
                dataJson.put("corpId", msgInfoReq.getTenantId());
                dataJson.put("tenantId", msgInfoReq.getTenantId());
                dataJson.put("messageData", msgInfoReq.getEvent().getVariableData());

                // 按语系循环发送，因为DAP接口不支持一组发送
                for (Map.Entry<String, Map<String, List<String>>> stringMapEntry : userMap.entrySet()) {
                    String language = stringMapEntry.getKey();
                    Map<String, List<String>> map = stringMapEntry.getValue();
                    if (CollectionUtils.isNotEmpty(map.get("userList"))) {
                        List<String> userIdList = map.get("userList").stream().distinct().collect(Collectors.toList());
                        dataJson.put("userList", userIdList);
                    } else {
                        dataJson.put("userList", Lists.newArrayList());
                    }
                    JSONObject dataJsonObject = new JSONObject();
                    dataJsonObject.put("data", dataJson);
                    msgJson.put("message", dataJsonObject);
                    emcService.sendCommonChannelMsg(language, msgJson);
                }
            } catch (Exception e) {
                jsonObject.put("errorCount", msgInfoReqList.size() - i);
                jsonObject.put("errorMessage", e.getMessage());
                break;
            }
        }
        return jsonObject;
    }

    /**
     * 必填参数校验
     *
     * @param msgInfoReq  请求体
     * @param channelFlag email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @return 返回
     */
    private String validRequiredParam(SendCommonChannelMsgReq.MsgInfoReq msgInfoReq, String channelFlag) {
        if ((msgInfoReq.getEvent() == null || StringUtils.isBlank(msgInfoReq.getEvent().getEventId())) && StringUtils.isBlank(msgInfoReq.getText())) {
            log.error("MessageControllerV2 | send common message param is missing:{}, channelFlag:{}", msgInfoReq, channelFlag);
            return "send common message param is missing : [event|text]";
        }
        return "";
    }

    /**
     * 校验事件是否存在且是否启用
     *
     * @param channelFlag email：邮箱 sms：短信 wecom：企微 dingTalk：钉钉
     * @param msgInfoReq  请求体
     * @return 返回
     */
    private String validEvent(String channelFlag, SendCommonChannelMsgReq.MsgInfoReq msgInfoReq) {
        List<QueryTemplateListResp> templateList = semcService.queryTemplateList(channelFlag, msgInfoReq.getEvent().getEventId(), msgInfoReq.getEvent().getRemark(), msgInfoReq.getTenantId());
        if (CollectionUtils.isEmpty(templateList)) {
            log.error("MessageControllerV2 | send common message eventId is not exist:{}, channelFlag:{}", msgInfoReq, channelFlag);
            return "send common message eventId is not exist:" + msgInfoReq.getEvent().getEventId() + ", channelFlag:" + channelFlag;
        }
        if (templateList.get(0).getValidStatus().equals(Constants.VALID_STATUS_UNUSABLE)) {
            log.error("MessageControllerV2 | send common message eventId is disabled:{}, channelFlag:{}", msgInfoReq, channelFlag);
            return "send common message eventId is disabled:" + msgInfoReq.getEvent().getEventId() + ", channelFlag:" + channelFlag;
        }
        return "";
    }

    /**
     * 构建接收人
     *
     * @param channelFlag    渠道标识
     * @param msgInfoReq     请求对象
     * @param blackWhiteList 黑白名单
     * @return 返回
     */
    private Map<String, Map<String, List<String>>> buildMessageReceiver(String channelFlag, SendCommonChannelMsgReq.MsgInfoReq msgInfoReq, List<QueryBlackWhiteListResp> blackWhiteList, String templateId) {
        Map<String, Map<String, List<String>>> receiverMap = Maps.newHashMap();
        // 查询管理后台配置是否配置了收件人
        QueryTemplateRelResp queryTemplateRelResp = semcService.queryRelPerson(msgInfoReq.getTenantId(), msgInfoReq.getEvent().getEventId(), channelFlag, templateId, null);
        // 邮箱
        if (MessageChannelEnum.EMAIL.getFlag().equals(channelFlag)) {
            // 如果管理后台没有配置，则查询传入的收件人邮箱
            if (CollectionUtils.isEmpty(queryTemplateRelResp.getContactList())
                    && CollectionUtils.isEmpty(queryTemplateRelResp.getCcContactList())
                    && CollectionUtils.isEmpty(queryTemplateRelResp.getBccContactList())) {
                receiverMap = buildEmail(channelFlag, msgInfoReq, blackWhiteList);
            } else {
                queryTemplateRelResp.setContactList(CollectionUtils.isNotEmpty(queryTemplateRelResp.getContactList()) ? queryTemplateRelResp.getContactList() : Lists.newArrayList());
                queryTemplateRelResp.setCcContactList(CollectionUtils.isNotEmpty(queryTemplateRelResp.getCcContactList()) ? queryTemplateRelResp.getCcContactList() : Lists.newArrayList());
                queryTemplateRelResp.setBccContactList(CollectionUtils.isNotEmpty(queryTemplateRelResp.getBccContactList()) ? queryTemplateRelResp.getBccContactList() : Lists.newArrayList());
                // 查询黑白名单邮箱，过滤黑名单，添加白名单
                if (CollectionUtils.isNotEmpty(blackWhiteList)) {
                    List<String> blackEmailList = Lists.newArrayList();
                    List<IamUserDTO> whiteList = Lists.newArrayList();
                    buildBlackWhiteParamList(channelFlag, blackWhiteList, blackEmailList, whiteList);

                    queryTemplateRelResp.getContactList().removeIf(x -> blackEmailList.contains(x.getValue()));
                    whiteList.forEach(x -> {
                        QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                        personDTO.setValue(x.getEmail());
                        personDTO.setLanguage(x.getLanguage());
                        queryTemplateRelResp.getContactList().add(personDTO);
                    });
                    queryTemplateRelResp.getCcContactList().removeIf(x -> blackEmailList.contains(x.getValue()));
                    queryTemplateRelResp.getBccContactList().removeIf(x -> blackEmailList.contains(x.getValue()));
                }

                // 组装数据
                List<QueryTemplateRelResp.RelPersonDTO> userList = Lists.newArrayList();
                userList.addAll(queryTemplateRelResp.getContactList());
                userList.addAll(queryTemplateRelResp.getCcContactList());
                userList.addAll(queryTemplateRelResp.getBccContactList());
                List<String> emailRespList = queryTemplateRelResp.getContactList().stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).collect(Collectors.toList());
                List<String> ccEmailRespList = queryTemplateRelResp.getCcContactList().stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).collect(Collectors.toList());
                List<String> bccEmailRespList = queryTemplateRelResp.getBccContactList().stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).collect(Collectors.toList());

                // 按语系分组
                List<String> emailList = Lists.newArrayList();
                List<String> ccEmailList = Lists.newArrayList();
                List<String> bccEmailList = Lists.newArrayList();
                Map<String, List<QueryTemplateRelResp.RelPersonDTO>> languageMap = userList.stream().collect(Collectors.groupingBy(QueryTemplateRelResp.RelPersonDTO::getLanguage));
                for (Map.Entry<String, List<QueryTemplateRelResp.RelPersonDTO>> entry : languageMap.entrySet()) {
                    String language = entry.getKey();
                    List<QueryTemplateRelResp.RelPersonDTO> personList = entry.getValue();
                    Map<String, List<String>> typeMap = Maps.newHashMap();
                    for (QueryTemplateRelResp.RelPersonDTO userDTO : personList) {
                        if (CollectionUtils.isNotEmpty(emailRespList) && emailRespList.contains(userDTO.getValue())) {
                            emailList.add(userDTO.getValue());
                        } else if (CollectionUtils.isNotEmpty(ccEmailRespList) && ccEmailRespList.contains(userDTO.getValue())) {
                            ccEmailList.add(userDTO.getValue());
                        } else if (CollectionUtils.isNotEmpty(bccEmailRespList) && bccEmailRespList.contains(userDTO.getValue())) {
                            bccEmailList.add(userDTO.getValue());
                        }
                    }
                    typeMap.put("email", emailList.stream().distinct().collect(Collectors.toList()));
                    typeMap.put("ccEmail", ccEmailList.stream().distinct().collect(Collectors.toList()));
                    typeMap.put("bccEmail", bccEmailList.stream().distinct().collect(Collectors.toList()));
                    receiverMap.put(language, typeMap);
                }
            }
        }
        // 短信
        else if (MessageChannelEnum.SMS.getFlag().equals(channelFlag)) {
            // 如果管理后台没有配置，则查询传入的收件人手机号
            if (CollectionUtils.isEmpty(queryTemplateRelResp.getContactList())) {
                receiverMap = buildUserIdOrTelephone(channelFlag, msgInfoReq, blackWhiteList);
            } else {
                // 过滤黑名单，添加白名单
                if (CollectionUtils.isNotEmpty(blackWhiteList)) {
                    List<String> blackTelephoneList = Lists.newArrayList();
                    List<IamUserDTO> whiteList = Lists.newArrayList();
                    buildBlackWhiteParamList(channelFlag, blackWhiteList, blackTelephoneList, whiteList);

                    queryTemplateRelResp.getContactList().removeIf(x -> blackTelephoneList.contains(x.getValue()));
                    whiteList.forEach(x -> {
                        QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                        personDTO.setValue(x.getTelephone());
                        personDTO.setLanguage(x.getLanguage());
                        queryTemplateRelResp.getContactList().add(personDTO);
                    });
                }
                Map<String, List<QueryTemplateRelResp.RelPersonDTO>> languageMap = queryTemplateRelResp.getContactList().stream().collect(Collectors.groupingBy(QueryTemplateRelResp.RelPersonDTO::getLanguage));
                for (Map.Entry<String, List<QueryTemplateRelResp.RelPersonDTO>> entry : languageMap.entrySet()) {
                    String language = entry.getKey();
                    List<QueryTemplateRelResp.RelPersonDTO> personList = entry.getValue();
                    Map<String, List<String>> typeMap = Maps.newHashMap();
                    List<String> telephoneList = personList.stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).distinct().collect(Collectors.toList());
                    typeMap.put("telephone", telephoneList);
                    receiverMap.put(language, typeMap);
                }
            }
        }
        // 企微和钉钉
        else if (MessageChannelEnum.WECOM.getFlag().equals(channelFlag) || MessageChannelEnum.DINGTALK.getFlag().equals(channelFlag)) {
            // 如果管理后台没有配置，则取传入的收件人用户
            if (CollectionUtils.isEmpty(queryTemplateRelResp.getContactList())) {
                receiverMap = buildUserIdOrTelephone(channelFlag, msgInfoReq, blackWhiteList);
            } else {
                // 添加白名单
                if (CollectionUtils.isNotEmpty(blackWhiteList)) {
                    List<String> whiteIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.WHITE_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
                    List<IamUserDTO> userInfoList = queryUserInfo(whiteIdList, channelFlag);
                    for (IamUserDTO iamUserDTO : userInfoList) {
                        QueryTemplateRelResp.RelPersonDTO personDTO = new QueryTemplateRelResp.RelPersonDTO();
                        personDTO.setValue(iamUserDTO.getUserId());
                        personDTO.setLanguage(iamUserDTO.getLanguage());
                        queryTemplateRelResp.getContactList().add(personDTO);
                    }
                    // 删除黑名单
                    List<String> blackUserIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.BLACK_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
                    queryTemplateRelResp.getContactList().removeIf(x -> blackUserIdList.contains(x.getValue()));
                }
                // 按语系分组
                Map<String, List<QueryTemplateRelResp.RelPersonDTO>> languageMap = queryTemplateRelResp.getContactList().stream().collect(Collectors.groupingBy(QueryTemplateRelResp.RelPersonDTO::getLanguage));
                for (Map.Entry<String, List<QueryTemplateRelResp.RelPersonDTO>> entry : languageMap.entrySet()) {
                    String language = entry.getKey();
                    List<QueryTemplateRelResp.RelPersonDTO> personList = entry.getValue();
                    Map<String, List<String>> typeMap = Maps.newHashMap();
                    List<String> userIdList = personList.stream().map(QueryTemplateRelResp.RelPersonDTO::getValue).distinct().collect(Collectors.toList());
                    typeMap.put("userList", userIdList);
                    receiverMap.put(language, typeMap);
                }
            }
        }
        return receiverMap;
    }

    /**
     * 查询黑白名单的邮箱/手机号/语系
     *
     * @param channelFlag    渠道
     * @param blackWhiteList 黑白名单对象
     * @param blackEmailList 黑名单参数
     * @param whiteList      白名单参数
     * @return 返回对象
     */
    private void buildBlackWhiteParamList(String channelFlag, List<QueryBlackWhiteListResp> blackWhiteList, List<String> blackEmailList, List<IamUserDTO> whiteList) {
        List<String> blackWhiteIdList = blackWhiteList.stream().map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
        // 黑白名单邮箱/手机号
        List<IamUserDTO> userInfoList = queryUserInfo(blackWhiteIdList, channelFlag);
        List<String> blackIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.BLACK_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
        for (IamUserDTO iamUserDTO : userInfoList) {
            if (blackIdList.contains(iamUserDTO.getUserId())) {
                if (MessageChannelEnum.EMAIL.getFlag().equals(channelFlag)) {
                    blackEmailList.add(iamUserDTO.getEmail());
                } else if (MessageChannelEnum.SMS.getFlag().equals(channelFlag)) {
                    blackEmailList.add(iamUserDTO.getTelephone());
                }
            } else {
                whiteList.add(iamUserDTO);
            }
        }
    }

    /**
     * 查询用户id或手机号
     *
     * @param channelFlag 渠道
     * @param msgInfoReq  请求对象
     * @return 返回
     */
    private Map<String, Map<String, List<String>>> buildUserIdOrTelephone(String channelFlag, SendCommonChannelMsgReq.MsgInfoReq msgInfoReq, List<QueryBlackWhiteListResp> blackWhiteList) {
        Map<String, Map<String, List<String>>> receiverMap = Maps.newHashMap();
        List<String> blackIdList = Lists.newArrayList();
        List<String> whiteIdList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(blackWhiteList)) {
            blackIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.BLACK_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
            whiteIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.WHITE_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
        }
        msgInfoReq.getUserIdList().removeIf(blackIdList::contains);
        msgInfoReq.getUserIdList().addAll(whiteIdList);

        // 查询收件人手机号及语系
        List<IamUserDTO> userInfoList = queryUserInfo(msgInfoReq.getUserIdList(), channelFlag);
        Map<String, List<IamUserDTO>> languageMap = userInfoList.stream().collect(Collectors.groupingBy(IamUserDTO::getLanguage));
        languageMap.forEach((k, v) -> {
            Map<String, List<String>> typeMap = Maps.newHashMap();
            if (MessageChannelEnum.SMS.getFlag().equals(channelFlag)) {
                List<String> telephoneList = v.stream().map(IamUserDTO::getTelephone).collect(Collectors.toList());
                typeMap.put("telephone", telephoneList);
            } else if (MessageChannelEnum.WECOM.getFlag().equals(channelFlag) || MessageChannelEnum.DINGTALK.getFlag().equals(channelFlag)) {
                List<String> userList = v.stream().map(IamUserDTO::getUserId).collect(Collectors.toList());
                typeMap.put("userList", userList);
            }
            receiverMap.put(k, typeMap);
        });
        return receiverMap;
    }

    /**
     * 按多语言分组构建邮箱 发送参数
     *
     * @param channelFlag 渠道
     * @param msgInfoReq  请求对象
     * @return 返回
     */
    private Map<String, Map<String, List<String>>> buildEmail(String channelFlag, SendCommonChannelMsgReq.MsgInfoReq msgInfoReq, List<QueryBlackWhiteListResp> blackWhiteList) {
        Map<String, Map<String, List<String>>> emailMap = Maps.newConcurrentMap();
        // 删除在黑名单中的用户，添加在白名单中的用户
        List<String> blackIdList = Lists.newArrayList();
        List<String> whiteIdList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(blackWhiteList)) {
            blackIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.BLACK_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
            whiteIdList = blackWhiteList.stream().filter(x -> x.getType().equals(Constants.WHITE_TYPE)).map(QueryBlackWhiteListResp::getUserId).collect(Collectors.toList());
        }
        List<String> userIdList = Lists.newArrayList();
        userIdList.addAll(CollectionUtils.isNotEmpty(msgInfoReq.getUserIdList()) ? msgInfoReq.getUserIdList() : Lists.newArrayList());
        userIdList.addAll(CollectionUtils.isNotEmpty(msgInfoReq.getCcUserIdList()) ? msgInfoReq.getCcUserIdList() : Lists.newArrayList());
        userIdList.addAll(CollectionUtils.isNotEmpty(msgInfoReq.getBccUserIdList()) ? msgInfoReq.getBccUserIdList() : Lists.newArrayList());
        userIdList.removeIf(blackIdList::contains);
        userIdList.addAll(whiteIdList);

        // 查询收件人邮箱及语系
        List<IamUserDTO> userInfoList = queryUserInfo(userIdList, channelFlag);
        // 按语系分组构建邮件接收人
        Map<String, List<IamUserDTO>> languageMap = userInfoList.stream().collect(Collectors.groupingBy(IamUserDTO::getLanguage));
        for (Map.Entry<String, List<IamUserDTO>> entry : languageMap.entrySet()) {
            String language = entry.getKey();
            List<IamUserDTO> valueList = entry.getValue();
            Map<String, List<String>> typeMap = Maps.newHashMap();
            List<String> emailList = Lists.newArrayList();
            List<String> ccEmailList = Lists.newArrayList();
            List<String> bccEmailList = Lists.newArrayList();
            for (IamUserDTO userDTO : valueList) {
                if (StringUtils.isNotBlank(userDTO.getEmail()) && ((CollectionUtils.isNotEmpty(msgInfoReq.getUserIdList()) && msgInfoReq.getUserIdList().contains(userDTO.getUserId()))
                        || whiteIdList.contains(userDTO.getUserId()))) {
                    emailList.add(userDTO.getEmail());
                } else if (StringUtils.isNotBlank(userDTO.getEmail()) && CollectionUtils.isNotEmpty(msgInfoReq.getCcUserIdList()) && msgInfoReq.getCcUserIdList().contains(userDTO.getUserId())) {
                    ccEmailList.add(userDTO.getEmail());
                } else if (StringUtils.isNotBlank(userDTO.getEmail()) && CollectionUtils.isNotEmpty(msgInfoReq.getBccUserIdList()) && msgInfoReq.getBccUserIdList().contains(userDTO.getUserId())) {
                    bccEmailList.add(userDTO.getEmail());
                }
            }
            typeMap.put("email", emailList);
            typeMap.put("ccEmail", ccEmailList);
            typeMap.put("bccEmail", bccEmailList);
            emailMap.put(language, typeMap);
        }
        return emailMap;
    }

    /**
     * 查询用户信息
     *
     * @param userIdList  用户id
     * @param channelFlag 渠道
     * @return 返回
     */
    private List<IamUserDTO> queryUserInfo(List<String> userIdList, String channelFlag) {
        List<IamUserDTO> userInfoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(userIdList)) {
            log.error("MessageControllerV2 | send common message query user info param userId is empty, channelFlag:{}", channelFlag);
            return userInfoList;
        }
        userInfoList = iamService.queryUserInfoList(userIdList);
        if (CollectionUtils.isEmpty(userInfoList)) {
            return userInfoList;
        }
        userInfoList.forEach(x -> {
            if (StringUtils.isBlank(x.getLanguage()) || Constants.ZH_CN_LOCALE.equals(x.getLanguage())) {
                x.setLanguage("zh-CN");
            }
            if (StringUtils.isNotBlank(x.getLanguage()) && Constants.ZH_TW_LOCALE.equals(x.getLanguage())) {
                x.setLanguage("zh-TW");
            }
            if (StringUtils.isNotBlank(x.getLanguage()) && Constants.EN_US_LOCALE.equals(x.getLanguage())) {
                x.setLanguage("en-US");
            }
        });
        return userInfoList;
    }

    @Override
    public void sendMsgAndEmail(List<MsgAndEmailBatchDTO> msgAndEmailBatchDTOList){
        log.info("【发送消息/邮件】：{}", JsonUtils.objectToString(msgAndEmailBatchDTOList));
        for(MsgAndEmailBatchDTO msgAndEmail : msgAndEmailBatchDTOList){
            List<String> recivers = msgAndEmail.getRecivers();
            // 1、查询用户的语言别、获取用户邮箱
            List<IamUserDTO> userInfoList = iamService.queryUserInfoList(recivers);

            // 2、获取用户消息发送方式
            List<PersonalizedDto> userMetadataList = iamService.queryUserMetadataList(recivers);
            if(CollectionUtils.isEmpty(userInfoList) || null == msgAndEmail.getContent() || MapUtils.isEmpty(msgAndEmail.getContent().getTitle()) || MapUtils.isEmpty(msgAndEmail.getContent().getMsg())){
                continue;
            }

            List<UserMsgAndEmailDTO> userMsgAndEmailDTOList = new ArrayList<>();
            for(IamUserDTO user : userInfoList){
                UserMsgAndEmailDTO userMsgAndEmail = new UserMsgAndEmailDTO();
                userMsgAndEmail.setUserId(user.getUserId());
                userMsgAndEmail.setLocale(user.getLanguage());
                userMsgAndEmail.setEmail(user.getEmail());
                // 用户消息发送方式
                Optional<PersonalizedDto> optional = userMetadataList.stream().filter(x -> x.getUserId().equals(user.getUserId())).findFirst();
                if(optional.isPresent()){
                    userMsgAndEmail.setMessageSendType(optional.get().getValue());
                }
                userMsgAndEmailDTOList.add(userMsgAndEmail);
            }
            Map<String, List<UserMsgAndEmailDTO>> langUserMap = userMsgAndEmailDTOList.stream().collect(Collectors.groupingBy(UserMsgAndEmailDTO::getLocale));
            for(String lang : langUserMap.keySet()){
                // 获取SSO地址（此处不支持免登）
                String ssoUrl = getSsoUrl(msgAndEmail);
                // 获取移动端跳转链接（不支持免登，由移动端拼接用户的token和语言别）：https://ksd-paas.apps.digiwincloud.com.cn/sso-login?userToken=${mobile_user_token}&dwLang=${locale}&routerLink=editor?id=675ab9cbfadfc00043d2005d
                String mobileDetailUrl = getDetailUrl(msgAndEmail, ssoUrl, MeesqgeConstant.MSG_SEND_TYPE_MOBILE);
                // 获取邮件跳转链接（不支持免登）：https://ksd-paas.apps.digiwincloud.com.cn/editor?id=675ab9cbfadfc00043d2005d
                String emailDetailUrl = getDetailUrl(msgAndEmail, ssoUrl, MeesqgeConstant.MSG_SEND_TYPE_EMAIL);

                // 3、发送工作提醒
                List<String> userList = langUserMap.get(lang).stream().filter(x -> MsgSendTypeEnum.IM.getName().equals(x.getMessageSendType())
                                || MsgSendTypeEnum.IM_AND_EMAIL.getName().equals(x.getMessageSendType())).map(UserMsgAndEmailDTO::getUserId).distinct().collect(Collectors.toList());
                sendImMessage(msgAndEmail, lang, userList, mobileDetailUrl);

                // 4、发送邮件
                List<UserMsgAndEmailDTO> userMsgAndEmailDTOS = langUserMap.get(lang).stream().filter(x -> MsgSendTypeEnum.EMAIL.getName().equals(x.getMessageSendType())
                        || MsgSendTypeEnum.MAIL.getName().equals(x.getMessageSendType())
                        || MsgSendTypeEnum.IM_AND_EMAIL.getName().equals(x.getMessageSendType())).collect(Collectors.toList());
                sendEmail(msgAndEmail, lang, userMsgAndEmailDTOS, emailDetailUrl);
            }
        }
    }

    /**
     *  发送催办im消息
     */
    private void sendImMessage(MsgAndEmailBatchDTO msgAndEmail, String lang, List<String> userList, String detailUrl){
        try {
            // 消息内容
            MessageDO messageDO = new MessageDO();
            messageDO.setGid(UUID.randomUUID().toString());
            messageDO.setSource(msgAndEmail.getSource());
            messageDO.setType(MeesqgeConstant.LINK_URL);
            if(null != msgAndEmail.getNoticeMobileApp()){
                messageDO.setNoticeMobileApp(msgAndEmail.getNoticeMobileApp());
            }else {
                // 默认是推送移动端消息提醒
                messageDO.setNoticeMobileApp(true);
            }
            messageDO.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            messageDO.setSendDate(LocalDateTime.now());
            messageDO.setDetailUrl(detailUrl);

            JSONObject contentMap = new JSONObject();
            contentMap.put("title", msgAndEmail.getContent().getTitle().get(lang));
            contentMap.put("msg", msgAndEmail.getContent().getMsg().get(lang));
            contentMap.put(MeesqgeConstant.LINK_URL, msgAndEmail.getLinkUrl());
            contentMap.put("config", JSONObject.fromObject(msgAndEmail.getConfig()));
            contentMap.put("status",0);
            messageDO.setContent(contentMap);
            messageDO.setTitle(msgAndEmail.getContent().getTitle().get(lang));

            MessageBatchUserDTO messageBatchUserDTO = new MessageBatchUserDTO();
            messageBatchUserDTO.setMessage(messageDO);
            messageBatchUserDTO.setTenantId(messageDO.getTenantId());
            messageBatchUserDTO.setUserIdList(userList);
            List<MessageBatchUserDTO> messageList = new ArrayList<>();
            messageList.add(messageBatchUserDTO);
            log.info("MessageServiceV2Impl-sendImMessage-messageList:{}",JsonUtils.objectToString(messageList));
            // 存储消息
            this.newMessageForBatchUsers(messageList);

            // 发送实时提醒
            List<MessageRemindDTO> appMessageList = new ArrayList<>();
            MessageRemindDTO appMessageDTO = MessageRemindDTO.convert(messageDO);
            List<Map> receiverDTOS = new ArrayList<>();
            userList.forEach(userId -> {
                Map<String, String> receiverDTO = new HashMap<>();
                receiverDTO.put("tenantId", messageDO.getTenantId());
                receiverDTO.put("userId", userId);
                receiverDTOS.add(receiverDTO);
            });
            appMessageDTO.setReceivers(receiverDTOS);
            appMessageList.add(appMessageDTO);
            messageSendService.sendMessageRemindToClient(appMessageList);
        } catch (Exception e) {
            log.error("sendImMessage,error:{}", e.getMessage(),e);
        }
    }

    /**
     * 发送催办邮件
     */
    private void sendEmail(MsgAndEmailBatchDTO msgAndEmail, String lang, List<UserMsgAndEmailDTO> userList, String detailLinkUrl){
        try {
            log.info("sendEmail-langName:{}",lang);

            // 邮件内容
            Map<String,Object> contentMap = new HashMap<>();
            contentMap.put("title", msgAndEmail.getContent().getTitle().get(lang));
            contentMap.put("msg", msgAndEmail.getContent().getMsg().get(lang));
            contentMap.put(MeesqgeConstant.LINK_URL, detailLinkUrl);

            JSONObject emailMsg = new JSONObject();
            emailMsg.put("data", contentMap);

            // 发送邮件
            List<String> contacts = userList.stream().map(UserMsgAndEmailDTO::getEmail).distinct().collect(Collectors.toList());
            JSONObject emailJson = new JSONObject();
            if(StringUtils.isEmpty(msgAndEmail.getEmailTemplateId())){
                // 邮件模板ID 支持邮件模板自定义，不传则用默认模板ATHENA_NOTICE_LINK
                emailJson.put("eventId", MeesqgeConstant.EMAIL_EVENT_ID_ATHENA_NOTICE_LINK);
            } else {
                emailJson.put("eventId", msgAndEmail.getEmailTemplateId());
            }
            emailJson.put("message", emailMsg);
            emailJson.put("contacts", StringUtils.join(contacts, ";"));
            log.info("[MessageServiceV2Impl_sendEmail emailJson]:{}", emailJson);
            emcService.sendEmail(lang, emailJson);
        } catch (Exception e) {
            log.error("sendEmail,error:{}",e);
        }
    }

    /**
     * 获取detailUrl
     * @param msgAndEmail
     * @param ssoUrl SSO回调地址（域名）
     * @return
     */
    private String getDetailUrl(MsgAndEmailBatchDTO msgAndEmail, String ssoUrl , String msgSendType){
        // detailUrl = ssoUrl + "/" + msgAndEmail.getLinkUrl();
        String detailUrl = "";
        // 是否需要免登录：如果ssoLogin为false，就要传带有域名的全路径
        if(msgAndEmail.getConfig() == null || msgAndEmail.getConfig().getSsoLogin() == null || !msgAndEmail.getConfig().getSsoLogin()){
            // 不需要免登录时，则返回带有域名的全路径
            return msgAndEmail.getLinkUrl();
        }

        // 如果ssoUrl为空，则直接返回detailUrl
        if(StringUtils.isEmpty(ssoUrl)){
            return detailUrl;
        }

        // ssoLogin为true，则linkUrl为相对路径,如果相对路径为空，则返回ssoUrl
        if(StringUtils.isEmpty(msgAndEmail.getLinkUrl())){
            return ssoUrl;
        }

        if(MeesqgeConstant.MSG_SEND_TYPE_EMAIL.equals(msgSendType)){
            // https://ksd-paas.apps.digiwincloud.com.cn/editor?id=675ab9cbfadfc00043d2005d
            detailUrl = ssoUrl + "/" + msgAndEmail.getLinkUrl();
        }else if(MeesqgeConstant.MSG_SEND_TYPE_MOBILE.equals(msgSendType)){
            // https://ksd-paas.apps.digiwincloud.com.cn/sso-login?userToken=${mobile_user_token}&dwLang=${locale}&routerLink=editor?id=675ab9cbfadfc00043d2005d
            detailUrl = String.format(MeesqgeConstant.SSO_LOGIN_URL,ssoUrl);
            detailUrl = detailUrl + "&routerLink="+ msgAndEmail.getLinkUrl();
        }

        return detailUrl;
    }

    /**
     * 获取SSO地址
     * @param msgAndEmail
     * @return
     */
    private String getSsoUrl(MsgAndEmailBatchDTO msgAndEmail){
        // SSO回调地址（域名）
        String ssoUrl = "";
        // 查询SEMC的SSO配置，拼接全路径普通地址
        GetSsoUrlReq req = new GetSsoUrlReq();
        // 邮件不支持免登录
        req.setSsoLogin(Boolean.FALSE);
        req.setAppType(msgAndEmail.getConfig().getAppType());
        req.setSsoAppCode(msgAndEmail.getConfig().getSsoAppCode());
        GetSsoUrlResp resp = semcService.getSsoUrl(req);
        if(null != resp){
            ssoUrl = resp.getAddress();
        }
        return ssoUrl;
    }
}
