package com.digiwin.athena.aim.domain.message.service.impl;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.CategoryEnum;
import com.digiwin.athena.aim.common.Constants;
import com.digiwin.athena.aim.common.InterfaceConstant;
import com.digiwin.athena.aim.domain.message.model.MessageBatchUserDTO;
import com.digiwin.athena.aim.domain.message.model.MessageContentDO;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.service.WecomMessageService;
import com.digiwin.athena.aim.infrastructure.emc.EmcService;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.iam.dto.IamUserTokenDTO;
import com.digiwin.athena.aim.util.AESUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.google.common.collect.Lists;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import static com.digiwin.athena.aim.common.Constants.MESSAGE_TYPE_ACTIVITY;
import lombok.extern.slf4j.Slf4j;

/**
 * 企微消息发送
 *
 * @author sungqz
 * @since 2023-12-4
 */
@Slf4j
@Service
public class WecomMessageServiceImpl implements WecomMessageService {

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private EmcService emcService;

    @Autowired
    private IamService iamService;

    @Autowired
    private MessageUtils messageUtils;

    @Resource
    private RestTemplate restTemplate;

    public static final String MESSAGE_LINK_URL = "<a href=\"%s\">%s</a>";

    public static final String ASA_MESSAGE_LINK_URL = "<div class=\\\"gray\\\">%s</div> <div class=\\\"normal\\\">%s</div>";

    private static final String END_TIME_FORM = "YYYY年M月d日";

    /**
     * 异步批量发送企微消息
     *
     * @param messageBatchUserDTO 消息列表
     */
    @Override
    public void sendWecomMessage(MessageBatchUserDTO messageBatchUserDTO) throws UnsupportedEncodingException {
        MessageDO message = messageBatchUserDTO.getMessage();
        // 判断消息是否为空
        if (message == null) {
            return;
        }
        MessageContentDO messageContentDO = JsonUtils.jsonToObject(JsonUtils.objectToString(message.getContent()), MessageContentDO.class);
        if (null == messageContentDO) {
            log.info("sendWecomMessage | content is empty, message:{}", JsonUtils.objectToString(message));
            return;
        }

        if (("ASA".equalsIgnoreCase(message.getSource()) && Constants.MESSAGE_TYPE_TEXT.equalsIgnoreCase(message.getType())) || "AGILE_DATA".equalsIgnoreCase(message.getSource())) {
            // 发送娜娜消息任务(type=text,source=ASA) 或 敏捷数据消息（source=AGILE_DATA）
            sendAsaWecomMessage(messageBatchUserDTO);
            return;
        } else if (!MESSAGE_TYPE_ACTIVITY.equalsIgnoreCase(message.getType())) {
            // 发送任务创建(type=activity,category=NEW)和签核知会(type=activity,category=SIGN_NOTICE)消息
            log.info("sendWecomMessage | message type is not within the range, type:{}", message.getType());
            return;
        }

        JSONObject content = message.getContent();
        String targetTenantId = messageBatchUserDTO.getTenantId();
        StringBuilder msg = new StringBuilder(content.containsKey("title") ? content.getString("title") : "")
                .append(content.containsKey("msg") ? content.getString("msg") : "");

        // 任务创建，需要拼接移动端路由接口，点击实现单点登录并打开任务卡，<a href="xxx">消息内容</a>
        // 链接：{mobileUrl}/mobile/v1/sso/grantOauthAccess?taskId={dataId}&proxyToken={proxyToken}&targetTenantId={targetTenantId}&iamUserToken={iamUserToken}&userId={userId}&tenantId={tenantId}&sourceType=Wechat&isHideAppBar=false
        if (CategoryEnum.NEW.name().equals(message.getCategory()) || CategoryEnum.SIGN_NOTICE.name().equals(message.getCategory())) {
            String taskId = content.getString("id");
            StringBuilder mobileUrl = new StringBuilder(envProperties.getMobileProperties().getUri() + InterfaceConstant.MOBILE_SSO_URL)
                    .append("?taskId=").append(taskId)
                    .append("&targetTenantId=").append(targetTenantId);

            // TODO: 2023/12/11  判断是否需要获取proxyToken
            mobileUrl.append("&proxyToken=").append("");

            // tenantId加密
            String tenantIdDecrypt = targetTenantId;
            try {
                tenantIdDecrypt = AESUtils.encrypt(AESUtils.MOBILE_SSO_SECRET_KEY, targetTenantId);
            } catch (Exception e) {
                log.info("sendWecomMessage AESUtils encrypted tenantId error, tenantId:{}", targetTenantId, e);
            }
            mobileUrl.append("&tenantId=").append(tenantIdDecrypt);

            // 批量获取用户的加密userToken
            List<String> userIdList = messageBatchUserDTO.getUserIdList();
            List<IamUserTokenDTO> userList = userIdList.stream().map(x -> {
                IamUserTokenDTO dto = new IamUserTokenDTO();
                dto.setUserId(x);
                dto.setTenantId(targetTenantId);
                dto.setAppId("athena");
                return dto;
            }).collect(Collectors.toList());
            List<IamUserTokenDTO> tokenDTOList = iamService.queryUserTempToken(userList);
            Map<String, String> userTokenMap = tokenDTOList.stream().collect(Collectors.toMap(IamUserTokenDTO::getUserId, IamUserTokenDTO::getTempToken, (a, b) -> a));

            // 对userId加密，并拼接链接
            for (String userId : userIdList) {
                String userIdDecrypt = userId;
                try {
                    userIdDecrypt = AESUtils.encrypt(AESUtils.MOBILE_SSO_SECRET_KEY, userId);
                } catch (Exception e) {
                    log.info("sendWecomMessage AESUtils encrypted userId error, tenantId:{}", targetTenantId, e);
                }
                String userToken = StringUtils.isNotBlank(userTokenMap.get(userId)) ? userTokenMap.get(userId) : "";
                mobileUrl.append("&userId=").append(userIdDecrypt)
                        .append("&iamUserToken=").append(userToken)
                        .append("&sourceType=Wechat&isHideAppBar=false");
                // 跳转
                ResponseEntity<JSONObject> respEntity = getWecomConfig();
                if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                    JSONArray data = respEntity.getBody().getJSONArray("data");
                    if (data != null && data.size() > 0) {
                        JSONObject jsonObject = data.getJSONObject(0);
                        String agentid = jsonObject.getString("appId");
                        String corpId = jsonObject.getString("corpId");
                        //redirect_uri=envProperties.getPcAddressUri()
                        String redirect_uri = envProperties.getPcAddressUri() + "/semc/router/routingDistribution?taskId=" + taskId + "&appId=" + agentid + "&tenantId=" + targetTenantId;
                        String encode = URLEncoder.encode(redirect_uri, "utf-8");
                        String weChatUrl = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" + corpId + "&redirect_uri=" + encode + "&response_type=code&scope=snsapi_base&state=STATE&agentid=" + agentid + "#wechat_redirect";
                        String text = String.format(MESSAGE_LINK_URL, weChatUrl, msg);
                        emcService.sendWeChat(Collections.singletonList(userId), text, null);
                        log.info("message send success");
                    }
                }
                log.error("Message sending failed");
                /* String text = String.format(MESSAGE_LINK_URL, mobileUrl, msg);
                // 不同的用户发送的消息不同，emc接口不支持，所以循环调用发送企微
                emcService.sendWeChat(Collections.singletonList(userId), text);*/
            }
             /*  // 拼接链接
            HashMap<String, String> userMsgMap = Maps.newHashMap();
            for (String userId : userIdList) {
                String userIdDecrypt = StringUtils.EMPTY;
                try {
                    userIdDecrypt = AESUtils.decrypt(AESUtils.MOBILE_SSO_SECRET_KEY, userId);
                } catch (Exception e) {
                    log.info("sendWecomMessage AESUtils encrypted userId error, tenantId:{}", targetTenantId, e);
                }
                String userToken = StringUtils.isNotBlank(userTokenMap.get(userId)) ? userTokenMap.get(userId) : "";
                mobileUrl.append("&userId=").append(userIdDecrypt)
                        .append("&iamUserToken=").append(userToken)
                        .append("&sourceType=").append("Wechat");
                msg = String.format(msgLinkUrl, mobileUrl, msg);
                userMsgMap.put(userId, msg);
            }
            // 异步调用emc发送企微
            invokeEmcWechatThread.execute(() -> userMsgMap.forEach((userId, msg1) -> emcService.sendWeChat(Collections.singletonList(userId), msg1)));
            */
        }

    }

    private void sendAsaWecomMessage(MessageBatchUserDTO messageBatchUserDTO) throws UnsupportedEncodingException {
        MessageDO message = messageBatchUserDTO.getMessage();

        ResponseEntity<JSONObject> respEntity = getWecomConfig();
        if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            if (data != null && data.size() > 0) {
                for (int i = 0; i < data.size(); i++) {
                    JSONObject jsonObject = data.getJSONObject(i);
                    if (jsonObject.getString("suiteId").equals(envProperties.getNanaSuiteId())) {
                        String agentid = jsonObject.getString("appId");
                        String corpId = jsonObject.getString("corpId");
                        //跳转链接需要前端提供     
                        String redirect_uri = String.format(Constants.NANA_MESSAGE_PC_URL, envProperties.getAniaAddressUri());
                        StringBuilder urlPath = new StringBuilder().append(redirect_uri).append("?");
                        List<MessageDO.TargetAction.Param> urlParams = message.getAction().getUrlParams();
                        for (int j = 0; j < urlParams.size(); j++){
                            urlPath.append(urlParams.get(j).getKey()).append("=").append(urlParams.get(j).getValue()).append("&");
                        }
                        urlPath.append("appId=").append(agentid);
//                        JSONObject nanaParams = new JSONObject();
//                        nanaParams.put("assistantCode", message.getAction().getAssistantCode());
//
//                        if ("AGILE_DATA".equalsIgnoreCase(message.getSource()) && StringUtils.isNotBlank(message.getAction().getType()) && message.getAction().getType().equals("AGILE_DATA_SUBSCRIPTION")) {
//                            //敏捷数据拼接url信息
//                            urlPath = redirect_uri + "?" + String.format(Constants.METRIC_REDIRECT_URL,  "nanaSubscriptionDetail", messageBatchUserDTO.getTenantId(), messageBatchUserDTO.getUserIdList().get(0), AESUtils.encode2(JSON.toJSONString(nanaParams)),
//                                    message.getAction().getAdaParams().containsKey("question") ? message.getAction().getAdaParams().getString("question") : null,
//                                    message.getAction().getAdaParams().containsKey("snapshotId") ? AESUtils.encodeString(message.getAction().getAdaParams().getString("snapshotId")) : null, message.getAction().getAdaParams().containsKey("sendDate") ? message.getAction().getAdaParams().getString("sendDate"): null,
//                                    message.getAction().getAdaParams().containsKey("undeletable") ? message.getAction().getAdaParams().getBoolean("undeletable") : null, message.getAction().getAdaParams().containsKey("dataTipMessage") ? AESUtils.encodeString(message.getAction().getAdaParams().getString("dataTipMessage")) : null,
//                                    message.getAction().getAdaParams().containsKey("method") ? message.getAction().getAdaParams().getString("method") : null, message.getAction().getAdaParams().containsKey("metricIdList") ? AESUtils.encode2(JSON.toJSONString(message.getAction().getAdaParams().getJSONArray("metricIdList"))) : null);
//                        } else {
//                            String workitemId = message.getAction().getWorkitemId();
//                            if (StringUtils.isNotBlank(workitemId)){
//                                nanaParams.put("workitemId", workitemId);
//                            }
//                            urlPath = redirect_uri + "?" + String.format(Constants.NANA_REDIRECT_URL, "nana", messageBatchUserDTO.getTenantId(), messageBatchUserDTO.getUserIdList().get(0), AESUtils.encode2(JSON.toJSONString(nanaParams)));
//                        }
//                        urlPath = urlPath + "&appId="+ agentid;
                        String encode = URLEncoder.encode(urlPath.toString(), "utf-8").replace("+","%20");
                        String weChatUrl = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" + corpId + "&redirect_uri=" + encode + "&response_type=code&scope=snsapi_base&state=STATE&agentid=" + agentid + "#wechat_redirect";
                        // 构造发送参数
                        JSONObject msgJson = new JSONObject();
                        JSONObject dataJsonObject = new JSONObject();
                        dataJsonObject.put("appId", agentid);
                        dataJsonObject.put("userList", messageBatchUserDTO.getUserIdList());
                        dataJsonObject.put("thirdUserList", new String[]{});
                        dataJsonObject.put("messageData", buildMessageData(messageBatchUserDTO, weChatUrl));

                        msgJson.put("message", new com.alibaba.fastjson.JSONObject().fluentPut("data", dataJsonObject));
                        msgJson.put("types", Lists.newArrayList("weCom"));
                        msgJson.put("eventId", envProperties.getWecomNoticeEventId());

                        emcService.sendCommonChannelMsg(messageBatchUserDTO.getMessage().getLocale(), msgJson);

                        log.info("message send success");
                    }
                }
            }
        }
    }

    private ResponseEntity<JSONObject> getWecomConfig() {
        HttpHeaders headers = new HttpHeaders();

        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        //组装请求体
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> respEntity = restTemplate.exchange(envProperties.getEmcUri() + "/api/emc/v1/appconfig/outsource/current", HttpMethod.POST, requestEntity, JSONObject.class);
        return respEntity;
    }

    public static void main(String[] args) throws UnsupportedEncodingException {
        String redirect_uri = "https://semc-test.apps.digiwincloud.com.cn" + "/semc/router/enterpriseWeChatLogin?tenantId=AthenaWF&appId=1000034";
        String encode = URLEncoder.encode(redirect_uri, "utf-8");
        String weChatUrl = "https://open.weixin.qq.com/connect/oauth2/authorize?appid=" + "" + "&redirect_uri=" + encode + "&response_type=code&scope=snsapi_base&state=STATE&agentid=" + "" + "#wechat_redirect";
        System.out.println(weChatUrl);
    }

    private JSONObject buildTaskButtonsMsg(String title, String description, String url, String btntxt) {
        JSONObject taskMsg = new JSONObject();
        taskMsg.put("title", title);
        taskMsg.put("description", description);
        taskMsg.put("url", url);
        taskMsg.put("btntxt", btntxt);
        return taskMsg;
    }

    private String parseDateOfPattern(Date date, String pattern) {
        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        return sdf.format(date);
    }

    private JSONObject buildMessageData(MessageBatchUserDTO messageBatchUserDTO, String weChatUrl) {
        MessageDO message = messageBatchUserDTO.getMessage();
        JSONObject content = message.getContent();
        String msg = content.getString("msg");

        String title;
        String btntxt;
        String description;
        String date = parseDateOfPattern(new Date(), END_TIME_FORM);

        String locale = StringUtils.isNotBlank(message.getLocale()) && message.getLocale().equals(Constants.ZH_CN_LOCALE) ? Constants.ZH_CN_LOCALE : Constants.ZH_TW_LOCALE;
        title = messageUtils.getMessageByLangName("meesage.wecom.title", locale);
        if ("AGILE_DATA".equalsIgnoreCase(message.getSource())) {
            description = "【" + content.getString("title") + "】\n" + msg;
        } else {
            if (StringUtils.isNotBlank(content.getString("msg"))) {
                description = msg;
            } else {
                description = messageUtils.getMessageByLangName("meesage.wecom.description", locale);
            }
        }
        btntxt = messageUtils.getMessageByLangName("message.line.button", locale);

        description = String.format(ASA_MESSAGE_LINK_URL, date, description);
        return buildTaskButtonsMsg(title, description, weChatUrl, btntxt);
    }
}
