package com.digiwin.athena.aim.domain.message.subscriber;

import com.google.common.eventbus.Subscribe;

import com.digiwin.athena.aim.common.Constants;
import com.digiwin.athena.aim.domain.message.event.WecomMessageEvent;
import com.digiwin.athena.aim.domain.message.model.MessageBatchUserDTO;
import com.digiwin.athena.aim.domain.message.service.WecomMessageService;
import com.digiwin.athena.aim.infrastructure.semc.SemcService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.MDC;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * 企微消息事件订阅
 *
 * @author sungqz
 * @since 2023-12-4
 */
@Component
@Slf4j
public class WecomMessageSubscriber {
    @Resource
    private WecomMessageService wecomMessageService;
    @Resource
    private SemcService semcService;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    public static final String CACHE_KEY = "aim:msg:channel:";

    @Subscribe
    public void handle(WecomMessageEvent event) {
        if (MapUtils.isNotEmpty(event.getMdcContext())) {
            MDC.setContextMap(event.getMdcContext());
        }
        log.info("send wecom message event：{}", CollectionUtils.isNotEmpty(event.getMessageBatchUserList()) ? event.getMessageBatchUserList().size() : 0);
        long start = System.currentTimeMillis();
        try {
            for (MessageBatchUserDTO message : event.getMessageBatchUserList()) {
                // 判断是否发送企微
                String messageChannelKey = CACHE_KEY+message.getTenantId();
                String messageChannel = stringRedisTemplate.opsForValue().get(messageChannelKey);
                if (StringUtils.isBlank(messageChannel)) {
                    messageChannel = semcService.queryMessageChannel(message.getTenantId());
                    if (StringUtils.isBlank(messageChannel)) {
                        stringRedisTemplate.opsForValue().set(messageChannelKey, "noExist", 1, TimeUnit.DAYS);
                    }
                }
                log.info("send wecom message for batch users | message channel:{}, message:{}", messageChannel, message);
                if (StringUtils.isNotBlank(messageChannel) && messageChannel.contains(Constants.MESSAGE_TYPE_WECOM)) {
                    wecomMessageService.sendWecomMessage(message);
                }
            }
        } catch (Exception ex) {
            log.error("send wecom message occur error", ex);
        } finally {
            log.info("send wecom message event结束时间：{}", System.currentTimeMillis() - start);
            try {
                if (MapUtils.isNotEmpty(event.getMdcContext())) {
                    MDC.clear();
                }
            } catch (Exception e) {
                log.error("logMdcRemoveEx", e);
            }
        }
    }
}
