package com.digiwin.athena.aim.infrastructure.concurrent.pool;

import com.jugg.agile.framework.core.config.JaPropertyListener;
import com.jugg.agile.framework.core.util.concurrent.migration.JaTtlExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.*;

/**
 * 动态线程池抽象类<br></>
 * 后期支持线程池的细节监控
 */
@Slf4j
public abstract class AbstractDynamicThreadPool implements InitializingBean, ExecutorService {

    protected ExecutorService executorService;

    protected DynamicThreadPoolBean threadPoolExecutor;

    @Override
    public void afterPropertiesSet() throws Exception {
        //初始化线程池
        initializeThreadPool();
        //监听配置变更
        configureChangeListener();
        //资源释放钩子
        addShutdownHook();
    }

    /**
     * 监听配置变更
     */
    private void configureChangeListener() {
        try {
            // 添加配置变更监听
            JaPropertyListener.addCommonListener(() -> {
                DynamicParam dynamicParam = transferParameter();
                changeThreadPoolConfig(dynamicParam.getCoreSize(), dynamicParam.getMaxSize());
                String chanageAfterLogStr = threadPoolExecutor.printThreadPoolStatus();
                log.info("changeThreadPool:" + chanageAfterLogStr);
            });
        } catch (Throwable e) {
            log.error("refresh DynamicThreadPool config", e);
        }
    }

    /**
     * 预留给自定义线程池初始化
     */
    private void initializeThreadPool() {
        threadPoolExecutor = buildThreadPool();
        executorService = JaTtlExecutors.getTtlExecutorService(threadPoolExecutor);
        String chanageBeforeLogStr = threadPoolExecutor.printThreadPoolStatus();
        log.info("initializeThreadPool:" + chanageBeforeLogStr);
    }

    /**
     * 预留子类构建线程池对象<br>
     *
     * @return
     */
    protected abstract DynamicThreadPoolBean buildThreadPool();

    /**
     * 预留给自定义线程池实现自定义配置变更
     */
    public abstract DynamicParam transferParameter();

    /**
     * 修改线程池核心参数
     *
     * @param coreSize 核心线程数
     * @param maxSize  最大线程数
     */
    private void changeThreadPoolConfig(int coreSize, int maxSize) {
        threadPoolExecutor.changeThreadPoolConfig(coreSize, maxSize);
    }

    /**
     * 钩子方法
     */
    protected void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            if (executorService != null && !executorService.isShutdown()) {
                executorService.shutdown();
            }
        }));
    }

    @Override
    public Future<?> submit(Runnable task) {
        return executorService.submit(task);
    }

    @Override
    public <T> Future<T> submit(Callable<T> task) {
        return executorService.submit(task);
    }

    @Override
    public void execute(Runnable command) {
        executorService.execute(command);
    }

    @Override
    public void shutdown() {
        executorService.shutdown();
    }

    @Override
    public List<Runnable> shutdownNow() {
        return executorService.shutdownNow();
    }

    @Override
    public boolean isShutdown() {
        return executorService.isShutdown();
    }

    @Override
    public boolean isTerminated() {
        return executorService.isTerminated();
    }

    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executorService.awaitTermination(timeout, unit);
    }

    @Override
    public <T> Future<T> submit(Runnable task, T result) {
        return executorService.submit(task, result);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return executorService.invokeAll(tasks);
    }

    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return executorService.invokeAll(tasks, timeout, unit);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return executorService.invokeAny(tasks);
    }

    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return executorService.invokeAny(tasks, timeout, unit);
    }

    /**
     * 动态线程池涉及可变更参数
     */
    static class DynamicParam {
        private Integer coreSize;
        private Integer maxSize;

        public DynamicParam(Integer coreSize, Integer maxSize) {
            this.coreSize = coreSize;
            this.maxSize = maxSize;
        }

        public Integer getCoreSize() {
            return coreSize;
        }

        public Integer getMaxSize() {
            return maxSize;
        }
    }
}
