package com.digiwin.athena.aim.infrastructure.concurrent.pool;

import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.*;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 自定义动态线程池
 *
 * @author zhq
 */
@Slf4j
public class DynamicThreadPoolBean extends ThreadPoolExecutor {

    private String poolName = "DynamicThreadPoolDefault";

    /**
     * 队列容量
     */
    private int queueCapacity;

    /**
     * 队列默认容量
     */
    private static final int DEFAULT_QUEUESIZE = 1000;

    /**
     * 固定阀值为80%
     */
    private static final Float THRESHOLD_FACTOR = 0.8f;

    /**
     * 通知间隔时间默认5分钟
     */
    private static final Long INTERVAL = 5 * 60 * 1000L;

    /**
     * 通知的时间
     */
    private Long maxPoolSizeWarningTime;
    private Long queueCapacityWarningTime;

    private final ReentrantLock lock = new ReentrantLock();

    private DynamicThreadPoolBean(int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
    }

    public DynamicThreadPoolBean(String poolName, int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler) {
        this(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
        this.poolName = poolName;
        this.setQueueCapacity(DEFAULT_QUEUESIZE);
    }

    public DynamicThreadPoolBean(String poolName, int corePoolSize, int maximumPoolSize, long keepAliveTime, TimeUnit unit, BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory, RejectedExecutionHandler handler, Integer queueSize) {
        this(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, threadFactory, handler);
        this.poolName = poolName;
        this.setQueueCapacity(queueSize);
    }

    public String getPoolName() {
        return poolName;
    }

    public int getQueueCapacity() {
        return queueCapacity;
    }

    /**
     * 仅是记录创建线程池时传入的工作队列的个数
     *
     * @param queueCapacity
     */
    public void setQueueCapacity(int queueCapacity) {
        this.queueCapacity = queueCapacity;
    }


    /**
     * 修改线程池配置
     *
     * @param coreSize
     * @param maxSize
     */
    protected void changeThreadPoolConfig(Integer coreSize, Integer maxSize) {
        this.setCorePoolSize(coreSize);
        this.setMaximumPoolSize(maxSize);
    }

    @Override
    protected void beforeExecute(Thread t, Runnable r) {
        lock.lock();
        try {
            String printLog = printThreadPoolStatus();
            log.info("beforeExecute " + printLog);
            logWarningNotice();
            super.beforeExecute(t, r);
        } catch (Exception e) {
            log.error("SpiderThreadPool beforeExecute error!", e);
            Thread.currentThread().interrupt();
        } finally {
            lock.unlock();
        }
    }

    @Override
    protected void afterExecute(Runnable r, Throwable t) {
        super.afterExecute(r, t);
        logWarningNotice();

    }

    @Override
    protected void terminated() {
        super.terminated();
    }

    public int getQueueTaskSize() {
        return this.getQueue().size();
    }


    public void logWarningNotice() {
        int maximumPoolSize = this.getMaximumPoolSize();
        int currentPoolSize = this.getPoolSize();
        boolean maxPoolSizeWarningTimeFlag = maxPoolSizeWarningTime == null || (System.currentTimeMillis() - maxPoolSizeWarningTime) > INTERVAL;
        boolean queueCapacityWarningTimeFlag = queueCapacityWarningTime == null || (System.currentTimeMillis() - queueCapacityWarningTime) > INTERVAL;
        if ((float) currentPoolSize / maximumPoolSize > THRESHOLD_FACTOR && maxPoolSizeWarningTimeFlag) {
            maxPoolSizeWarningTime = System.currentTimeMillis();
            log.warn(poolName + "->线程池最大线程数已超过阈值，当前最大线程数：" + maximumPoolSize + "，当前活跃线程数：" + currentPoolSize + "，当前线程负载：" + (float) currentPoolSize / maximumPoolSize + "，请及时调整线程池配置");
        }
        if ((float) this.getQueueTaskSize() / queueCapacity > THRESHOLD_FACTOR && queueCapacityWarningTimeFlag) {
            queueCapacityWarningTime = System.currentTimeMillis();
            log.warn(poolName + "->线程池队列已超过阈值，当前队列容量：" + queueCapacity + "，当前队列任务数：" + this.getQueueTaskSize() + "，当前队列负载：" + (float) this.getQueueTaskSize() / queueCapacity + "，请及时调整线程池配置");
        }
    }

    /**
     * 打印线程池状态
     *
     * @return
     */
    public String printThreadPoolStatus() {
        return String.format(poolName + "->core_size:%s;max_size:%s;active_size:%s;pool_size:%s;queue_size:%s;task_count:%s;completed_count:%s;largest_pool_count:%s", this.getCorePoolSize(), this.getMaximumPoolSize(), this.getActiveCount(), this.getPoolSize(), this.getQueue().size(), this.getTaskCount(), this.getCompletedTaskCount(), this.getLargestPoolSize());
    }

}
