package com.digiwin.athena.aim.infrastructure.emc.impl;

import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.Constants;
import com.digiwin.athena.aim.common.InterfaceConstant;
import com.digiwin.athena.aim.domain.message.model.MessageCenterConfigDTO;
import com.digiwin.athena.aim.domain.message.model.MessageCenterTenantChannelConfig;
import com.digiwin.athena.aim.infrastructure.emc.EmcService;
import com.digiwin.athena.aim.infrastructure.eoc.dto.EmpDTO;
import com.digiwin.athena.aim.infrastructure.thememap.dto.IamDdMessageDTO;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.iam.sdk.meta.dto.IamResultBDTO;
import com.digiwin.athena.iam.sdk.meta.dto.IamResultDTO;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.*;

/**
 * @program: emc
 */
@Slf4j
@Service
public class EmcServiceImpl implements EmcService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;


    @Override
    public int senDdMessage(IamDdMessageDTO iamDdMessageDTO) {
        String url = envProperties.getEmcUri() + InterfaceConstant.IAM_MESSAGE_EMAIL;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("Digi-Middleware-Auth-App", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        String local = iamDdMessageDTO.getLocale();
        headers.add("Accept-Language", StringUtils.isEmpty(local) ? Constants.ZH_CN_LOCALE : local);
        JSONObject reqObject = JSONObject.fromObject(this);
        reqObject.put("eventId", envProperties.getNoticeEventId() + "_" + iamDdMessageDTO.getAppId());
        JSONObject messageObject = JSONObject.fromObject(this);
        messageObject.put("data", iamDdMessageDTO);
        reqObject.put("message", messageObject);
        reqObject.put("types", new String[]{"dingTalk"});
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);
        try {
            log.info("请求EMC发送钉钉消息 入参:{}", JsonUtils.objectToString(httpEntity));
            ResponseEntity<BaseResultDTO<IamDdMessageDTO>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO<IamDdMessageDTO>>() {
                    });
            log.info("请求EMC发送钉钉消息end，入参：{}, 响应内容：{}", JsonUtils.objectToString(iamDdMessageDTO), JsonUtils.objectToString(respEntity.getBody()));
        } catch (BusinessException ex) {
            log.error("请求EMC发送钉钉消息报错，入参：{}, ex：{}", JsonUtils.objectToString(iamDdMessageDTO), ex);
        }
        return 0;
    }

    /**
     * 发送企微消息
     *
     * @param userIdList 消息发送用户列表
     * @param msg        消息内容
     * @return 返回对象
     */
    @Override
    public void sendWeChat(List<String> userIdList, String msg, String appId) {
        String url = envProperties.getEmcUri() + InterfaceConstant.EMC_MESSAGE_WECHAT;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        JSONObject reqObject = JSONObject.fromObject(this);
        reqObject.put("msgType", "text");
        if (StringUtils.isNotBlank(appId)){
            reqObject.put("appId", appId);
        }
        reqObject.put("content", msg);
        reqObject.put("userIds", userIdList);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);
        try {
            log.info("invoke emc sendWeChat, url:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            ResponseEntity<Object> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<Object>() {
                    });
            log.info("invoke emc sendWeChat success, url:{}, response：{}", url, JsonUtils.objectToString(respEntity.getBody()));
        } catch (Exception ex) {
            log.error("invoke emc sendWeChat error, url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
        }
    }

    /**
     * 各渠道发送消息
     *
     * @param language 语系
     * @param msgJson  消息体
     */
    @Override
    public void sendCommonChannelMsg(String language, JSONObject msgJson) {
        String url = envProperties.getEmcUri() + InterfaceConstant.EMC_MESSAGE_COMMON_CHANNEL;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("Accept-Language", language);

        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(msgJson, headers);
        try {
            log.info("start invoke emc send common channel msg. param:{}", httpEntity);
            ResponseEntity<HashMap> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, HashMap.class);
            if (respEntity.getBody() == null || !requestSuccess(respEntity.getBody().get("code"), respEntity.getBody().get("success"))) {
                log.error("invoke emc send common channel msg fail. url:{}, result:{}", url, respEntity);
                throw BusinessException.create("invoke emc send common channel msg fail. param:" + httpEntity + ", result:" + respEntity);
            }
            log.info("invoke emc send common channel msg success. param:{}, result:{}", httpEntity, respEntity);
        } catch (Exception ex) {
            log.error("invoke emc send common channel msg error, url:{}, request:{}, errorMessage:{}", url, httpEntity, ex.getMessage());
            throw BusinessException.create("invoke emc send common channel msg error:" + ex.getMessage() + ", param:" + httpEntity);
        }
    }

    /**
     * 返回结果校验
     *
     * @param code    状态码
     * @param success 是否成功
     * @return 返回对象
     */
    private boolean requestSuccess(Object code, Object success) {
        if (null == code || null == success) {
            return false;
        }
        return StringUtils.equals(String.valueOf(code), String.valueOf(org.apache.http.HttpStatus.SC_OK)) && Boolean.parseBoolean(String.valueOf(success));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void sendEmail(String language, JSONObject content) {
        log.info("[SendEmail] EmcServiceImpl start: {}", content);
        String url = envProperties.getEmcUri() + InterfaceConstant.IAM_MESSAGE_EMAIL;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("Accept-Language", StringUtils.isEmpty(language) ? Constants.ZH_CN_LOCALE : language);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        HttpEntity<?> httpEntity = new HttpEntity<>(content, headers);

        try {
            ResponseEntity<HashMap> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, HashMap.class);
            Object code = respEntity.getBody().get("code");
            Object success = respEntity.getBody().get("success");
            if (!requestSuccess(code, success)) {
                log.error("调用emc发送邮件失败，{}", "status code: " + org.apache.http.HttpStatus.SC_OK + ", code: " + code + ", message: " + JsonUtils.objectToString(respEntity.getBody().get("message")));
            }
        } catch (Exception ex) {
            log.error("请求EMC发邮件报错，入参：{}, ex：{}", JsonUtils.objectToString(content), ex);
        }
    }

    /**
     * 根据场景id获取场景配置
     *
     * @param tenantId
     * @param appCode
     * @param sceneId
     * @return
     */
    @Override
    public List<MessageCenterConfigDTO> getMsgConfigList(String tenantId, String appCode, String sceneId) {

        String url = envProperties.getEmcUri() + InterfaceConstant.EMC_MESSAGE_CONFIG_LIST;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(headers);

        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("source", appCode);
        param.put("athenaMessageId", sceneId);
        try {
            log.info("[getMsgConfigList]请求EMC 查询场景配置 入参:tenantId = {},source = {},athenaMessageId = {} ", tenantId, appCode, sceneId);
            ResponseEntity<IamResultDTO<List<MessageCenterConfigDTO>>> respEntity = this.restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<IamResultDTO<List<MessageCenterConfigDTO>>>() {
                    }, param);
            log.info("[getMsgConfigList]请求EMC 查询场景配置， 响应内容：{}", JsonUtils.objectToString(respEntity.getBody()));
            if (null != respEntity.getBody() && 200 == respEntity.getBody().getCode()) {
                return respEntity.getBody().data();
            }
        } catch (BusinessException ex) {
            log.error("请求EMC查询场景配置失败，入参:tenantId = {},source = {},athenaMessageId = {}, ex：{}", tenantId, appCode, sceneId, ex);
        }
        return new ArrayList<>();
    }

    @Override
    public List<MessageCenterTenantChannelConfig> getMsgTenantChannelConfigList() {
        String url = envProperties.getEmcUri() + InterfaceConstant.EMC_MESSAGE_CHANNEL_LIST;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(headers);

        try {
            log.info("请求EMC 查询租户渠道配置 入参:tenantId = {}", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            ResponseEntity<IamResultDTO<List<MessageCenterTenantChannelConfig>>> respEntity = this.restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<IamResultDTO<List<MessageCenterTenantChannelConfig>>>() {
                    });
            log.info("请求EMC 查询租户渠道配置 , 响应内容：{}", JsonUtils.objectToString(respEntity.getBody()));
            if (null != respEntity.getBody() && 200 == respEntity.getBody().getCode()) {
                return respEntity.getBody().getData();
            }
        } catch (BusinessException ex) {
            log.error("请求EMC 查询租户渠道配置失败，入参:tenantId = {}, ex：{}", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(), ex);
        }
        return new ArrayList<>();
    }


    @Override
    public Map getAppConfig(JSONObject param) {
        Map map = Maps.newHashMap();
        String url = envProperties.getEmcUri() + InterfaceConstant.API_EMC_V1_APPCONFIG_FIND;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(param,headers);
        try {
            log.info("调用emc查询钉钉内部应用配置开始，入参是：{}",JsonUtils.objectToString(httpEntity));
            ResponseEntity<HashMap> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, HashMap.class);
            log.info("调用emc查询钉钉内部应用配置结束，返参是：{}",JsonUtils.objectToString(respEntity));
            Object code = respEntity.getBody().get("code");
            Object success = respEntity.getBody().get("success");
            if (!requestSuccess(code, success)) {
                log.error("调用emc查询钉钉内部应用配置失败，{}", "status code: " + org.apache.http.HttpStatus.SC_OK + ", code: " + code + ", message: " + JsonUtils.objectToString(respEntity.getBody().get("message")));
            }
            if (Objects.nonNull(respEntity.getBody().get("data"))){
                Object data = respEntity.getBody().get("data");
                map.put("corpId", JSONObject.fromObject(data).getString("corpId"));
                map.put("agentId", JSONObject.fromObject(data).getString("appId"));
            }
        } catch (Exception ex) {
            log.error("请求EMC查询钉钉内部应用配置失败，入参：{}, ex：{}", JsonUtils.objectToString(httpEntity), ex);
        }
        return map;
    }

    @Override
    public ResponseEntity<JSONObject> getWecomConfig() {
        HttpHeaders headers = new HttpHeaders();

        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        //组装请求体
        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<JSONObject> respEntity = restTemplate.exchange(envProperties.getEmcUri() + "/api/emc/v1/appconfig/outsource/current", HttpMethod.POST, requestEntity, JSONObject.class);
        return respEntity;
    }
}
