package com.digiwin.athena.aim.infrastructure.iam.impl;

import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.MeesqgeConstant;
import com.digiwin.athena.aim.infrastructure.iam.IamService;
import com.digiwin.athena.aim.infrastructure.iam.dto.IamUserDTO;
import com.digiwin.athena.aim.infrastructure.iam.dto.IamUserTokenDTO;
import com.digiwin.athena.aim.infrastructure.iam.dto.PersonalizedDto;
import com.digiwin.athena.aim.infrastructure.iam.dto.UserLangMetadataDTO;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import net.sf.json.JsonConfig;
import org.apache.commons.collections.CollectionUtils;
import org.checkerframework.checker.units.qual.A;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @program: aim
 * @description:
 * @author: Hang Liu
 * @create: 2023-06-27 13:56
 */
@Slf4j
@Service
public class IamServiceImpl implements IamService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    MessageUtils messageUtils;


    //获取当前登入租户的corpId
    private static final String TENANT_METADATA_CORPID = "/api/iam/v2/tenant/metadata/corpid/from?comeFrom=dingding";

    public static final String CATALOG_ID_BASIC = "basic";

    public static final String MESSAGE_NOTICE_TYPE_KEY = "MessageSendType";

    public static final String DAP_ACCEPT_LANGUAGE = "dapAcceptLanguage";

    /**
     * 查询当前租户下指定角色下的用户清单的信息
     *
     * @param roleId 角色id
     */
    @Override
    public List<IamUserDTO> getUserListByRole(String roleId) {

        String url = envProperties.getIamUri() + "/api/iam/v2/association/userinrole";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        // headers.add(GlobalConstant.IAM_USER_TOKEN, "480bb1a4-c75b-46f7-91c9-907337485b5e");

        Map<String, Object> params = new HashMap<>();
        params.put("roleId", roleId);

        HttpEntity httpEntity = new HttpEntity(params, headers);
        try {
            ResponseEntity<List<IamUserDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<List<IamUserDTO>>() {
                    }, params);
            if (null != responseEntity.getBody()) {
                return responseEntity.getBody();
            } else {
                log.error("请求AIM查询当前租户下指定角色下的用户清单的信息报错，参数：{}", params);
                String errMsg = MessageFormat.format(messageUtils.getMessage("exception.aim.user.list"),
                        JsonUtils.objectToString(params));
                throw BusinessException.create(errMsg);
            }
        } catch (BusinessException ex) {
            throw ex;
        } catch (Exception ex) {
            String errMsg = MessageFormat.format(messageUtils.getMessage("exception.aim.user.list"),
                    JsonUtils.objectToString(params));
            throw BusinessException.create(errMsg, ex);
        }
    }

    @Override
    public String getUserMetadataAllTenant(String userId, int type, int lineOrDingding) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        log.info("getUserMetadataAllTenant-userId:{},type:{}", userId, type);
        Map<String, Object> params = new HashMap<>();
        params.put("userId", userId);
        params.put("tenantSid",
                0);
        if (type == 1) {
            //获取用户配置的语言类别
            params.put("catalogId", "basic");
            params.put("key", "dapAcceptLanguage");
        } else if (type == 2) {
            //获取用户line绑定信息
            params.put("catalogId", "integrationUserId");
            if (Objects.equals(1,lineOrDingding)) {
                params.put("key", "line_Athena");
            } else{
                params.put("tenantSid",
                        AppAuthContextHolder.getContext().getAuthoredUser().getTenantSid());
                params.put("key", "dingding");
            }
        }

        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + "/api/iam/v2/usermetadata/user/get";

        try {
            log.info(JsonUtils.objectToString(httpEntity));
            ResponseEntity<UserLangMetadataDTO> respEntity = restTemplate.postForEntity(url, httpEntity, UserLangMetadataDTO.class);
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                UserLangMetadataDTO baseResult = respEntity.getBody();
                if (null != baseResult && HttpStatus.OK.value() == baseResult.getCode()) {
                    Object value = Optional.ofNullable(baseResult.getData()).map(data -> data.get("value")).orElse(null);
                    if (type == 1) {
                        //用户配置的语言类别
                        //如果用户设置了语言别，则使用用户的语言别，如果没有使用语言别，则使用默认的语言别
                        return Objects.nonNull(value) ? String.valueOf(value) : "zh_CN";
                    } else if (type == 2) {
                        //用户line绑定信息
                        return Objects.nonNull(value) ? String.valueOf(value) : null;
                    }
                }
            }
            throw BusinessException.create(respEntity.getStatusCodeValue(), JsonUtils.objectToString(respEntity.getBody()));
        } catch (Exception ex) {
            String errMsg = null;
            if (type == 1) {
                errMsg = MessageFormat.format(messageUtils.getMessage("exception.aim.user.lang"),
                        JsonUtils.objectToString(params));
            } else if (type == 2) {
                errMsg = MessageFormat.format(messageUtils.getMessage("exception.aim.user.line"),
                        JsonUtils.objectToString(params));
            }
            throw BusinessException.create(errMsg, ex);
        }
    }

    /**
     * 批量获取用户临时token
     *
     * @param userIdList 用户列表
     * @return 返回对象
     */
    @Override
    public List<IamUserTokenDTO> queryUserTempToken(List<IamUserTokenDTO> userIdList) {
        String url = envProperties.getIamUri() + "/api/iam/v2/user/tempToken/list";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        Map<String, Object> params = new HashMap<>();
        params.put("userList", userIdList);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<JSONObject> respEntity;
        try {
            respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query user temp token error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            return null;
        }
        if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            return JSONArray.toList(data, new IamUserTokenDTO(), new JsonConfig());
        } else {
            log.error("query user temp token fail. url:{}, param:{}, result:{}", url, JsonUtils.objectToString(httpEntity), respEntity);
            return null;
        }
    }

    /**
     * 批量获取用户信息
     *
     * @param userIdList 用户列表
     * @return 返回对象
     */
    @Override
    public List<IamUserDTO> queryUserInfoList(List<String> userIdList) {
        if(CollectionUtils.isEmpty(userIdList)){
            return new ArrayList<>();
        }
        userIdList = userIdList.stream().distinct().collect(Collectors.toList());
        String url = envProperties.getIamUri() + "/api/iam/v2/tenant/user/info";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        Map<String, Object> params = new HashMap<>();
        params.put("userIds", userIdList);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);

        ResponseEntity<JSONObject> respEntity;
        try {
            log.info("start query user Info list. param:{}", httpEntity);
            respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
        } catch (Exception ex) {
            log.error("query user Info list error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            throw BusinessException.create("query user Info list error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
        }
        if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
            JSONArray data = respEntity.getBody().getJSONArray("data");
            log.info("uery user Info list success. param:{}, result:{}", JsonUtils.objectToString(httpEntity), data);
            return JSONArray.toList(data, new IamUserDTO(), new JsonConfig());
        } else {
            log.error("query user Info list fail. url:{}, param:{}, result:{}", url, JsonUtils.objectToString(httpEntity), respEntity);
            throw BusinessException.create("query user Info list fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
        }
    }

@Override
public String getTenantCorpid() {
        // header信息
        HttpHeaders headers = new HttpHeaders();
        headers.add("Content-Type", "application/json");
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        headers.add("digi-middleware-auth-user", userToken);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());

        HttpEntity<Map<String, Object>> requestEntity = new HttpEntity<>(null, headers);
        StringBuilder url = new StringBuilder(envProperties.getIamUri()).append(TENANT_METADATA_CORPID);
        try {
            ResponseEntity<JSONObject> respEntity = restTemplate.exchange(url.toString(), HttpMethod.GET,
                    requestEntity, JSONObject.class);
            if (org.apache.http.HttpStatus.SC_OK == respEntity.getStatusCodeValue() && respEntity.getBody() != null) {
                String data = respEntity.getBody().getString("data");
                return data;
            }
        } catch (Exception ex) {
            log.error("从IAM获取当前登入租户的corpId异常, token: {}, errorMsg:{}", userToken, ex.getMessage());
            return "";
        }
        return "";
    }

    /**
     * 批量获取用户的自定义属性值，也可获取其他附加属性(一批最多查询500个)
     *
     * @param userIdList 用户列表
     * @return 返回对象
     */
    @Override
    public List<PersonalizedDto> queryUserMetadataList(List<String> userIdList) {
        List<PersonalizedDto> result = new ArrayList<>();
        if(CollectionUtils.isEmpty(userIdList)){
            return result;
        }
        userIdList = userIdList.stream().distinct().collect(Collectors.toList());
        List<List<String>> userIdListList = Lists.partition(userIdList, MeesqgeConstant.MAX_QUERY_SIZE);

        String url = envProperties.getIamUri() + "/api/iam/v2/usermetadata/user/get/batch";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        Map<String, Object> params = new HashMap<>();
        params.put("tenantSid", AppAuthContextHolder.getContext().getAuthoredUser().getTenantSid());
        params.put("catalogId", CATALOG_ID_BASIC);
        params.put("key", MESSAGE_NOTICE_TYPE_KEY);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        userIdListList.forEach(userIds -> {
            params.put("userIdList", userIds);
            ResponseEntity<JSONObject> respEntity;
            try {
                log.info("start query user metadata list. param:{}", httpEntity);
                respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
            } catch (Exception ex) {
                log.error("query user metadata list error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
                throw BusinessException.create("query user metadata list error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
            }
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && respEntity.getBody() != null && CollectionUtils.isNotEmpty(respEntity.getBody().getJSONArray("data"))) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                log.info("query user metadata list success. param:{}, result:{}", JsonUtils.objectToString(httpEntity), data);
                result.addAll(JSONArray.toList(data, new PersonalizedDto(), new JsonConfig()));
            } else {
                log.error("query user metadata list fail. url:{}, param:{}, result:{}", url, JsonUtils.objectToString(httpEntity), respEntity);
                throw BusinessException.create("query user metadata list fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
            }
        });
        return result;
    }

    @Override
    public List<PersonalizedDto> queryUserLangMetadataList(List<String> userIdList) {
        List<PersonalizedDto> result = new ArrayList<>();
        if(CollectionUtils.isEmpty(userIdList)){
            return result;
        }
        userIdList = userIdList.stream().distinct().collect(Collectors.toList());
        List<List<String>> userIdListList = Lists.partition(userIdList, MeesqgeConstant.MAX_QUERY_SIZE);

        String url = envProperties.getIamUri() + "/api/iam/v2/usermetadata/user/get/batch";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add("digi-middleware-auth-app", envProperties.getAppToken());
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        Map<String, Object> params = new HashMap<>();
        params.put("tenantSid", 0);
        params.put("catalogId", CATALOG_ID_BASIC);
        params.put("key", DAP_ACCEPT_LANGUAGE);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(params, headers);
        userIdListList.forEach(userIds -> {
            params.put("userIdList", userIds);
            ResponseEntity<JSONObject> respEntity;
            try {
                log.info("start query user metadata list. param:{}", httpEntity);
                respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, JSONObject.class);
            } catch (Exception ex) {
                log.error("query user metadata list error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
                throw BusinessException.create("query user metadata list error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
            }
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && respEntity.getBody() != null && CollectionUtils.isNotEmpty(respEntity.getBody().getJSONArray("data"))) {
                JSONArray data = respEntity.getBody().getJSONArray("data");
                log.info("query user metadata list success. param:{}, result:{}", JsonUtils.objectToString(httpEntity), data);
                result.addAll(JSONArray.toList(data, new PersonalizedDto(), new JsonConfig()));
            } else {
                log.error("query user metadata list fail. url:{}, param:{}, result:{}", url, JsonUtils.objectToString(httpEntity), respEntity);
                throw BusinessException.create("query user metadata list fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
            }
        });
        return result;
    }
}
