package com.digiwin.athena.aim.infrastructure.mobile;

import com.digiwin.athena.aim.infrastructure.mobile.MobileProperties.OauthProperties;
import com.digiwin.athena.aim.infrastructure.mobile.entity.AppTokenDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * AppOauthServiceIMpl Description
 *
 * @author majianfu
 * @date 2021/7/23
 * @since
 */
@Component
@Slf4j
public class AppOauthServiceImpl implements AppOauthService {
    private static final String TOKEN_CACHE_KEY_FORMAT = "aim:mobile:accessToken:%s";

    // ttl：48小时
    private static final long TOKEN_CACHE_TTL_HOURS = 48;

    @Resource
    private RestTemplate restTemplate;

    @Resource
    private StringRedisTemplate stringRedisTemplate;

    /**
     * {@inheritDoc}
     */
    @Override
    public AppTokenDTO getAccessToken(OauthProperties oauthProperties, boolean flushCache) {
        String cacheKey = String.format(TOKEN_CACHE_KEY_FORMAT, oauthProperties.getClientId());
        if (flushCache) {
            stringRedisTemplate.delete(cacheKey);
        } else {
            String cacheValue = stringRedisTemplate.opsForValue().get(cacheKey);
            if (StringUtils.isNotBlank(cacheValue)) {
                log.info("[AppOauthServiceImpl.getAccessToken] get access_token(clientId: {}) from cache.", oauthProperties.getClientId());
                return JsonUtils.jsonToObject(cacheValue, AppTokenDTO.class);
            }
        }

        AppTokenDTO appTokenDTO = getAccessToken(oauthProperties);
        if (null != appTokenDTO) {
            stringRedisTemplate.opsForValue().set(cacheKey, JsonUtils.objectToString(appTokenDTO), TOKEN_CACHE_TTL_HOURS, TimeUnit.HOURS);
        }
        return appTokenDTO;
    }

    public AppTokenDTO getAccessToken(OauthProperties oauthProperties) {
        log.info("[AppOauthServiceImpl.getAccessToken] get access_token(clientId: {}) by http request.", oauthProperties.getClientId());
        String uri = UriComponentsBuilder.fromUriString(oauthProperties.getUri())
                .queryParam("grant_type", oauthProperties.getGrantType())
                .queryParam("client_id", oauthProperties.getClientId())
                .queryParam("client_secret", oauthProperties.getClientSecret())
                .toUriString();

        ResponseEntity<AppTokenDTO> respEntity = restTemplate.getForEntity(uri, AppTokenDTO.class);
        if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
            return respEntity.getBody();
        } else {
            throw BusinessException.create(respEntity.getStatusCodeValue(), String.valueOf(respEntity.getBody()));
        }
    }
}
