package com.digiwin.athena.aim.infrastructure.mobile;

import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.infrastructure.mobile.entity.AppTokenDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.RetryableAction;
import com.digiwin.athena.appcore.util.RetryableOperationToolkit;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Resource;

/**
 * AppSendMsgServiceImpl Description
 *
 * @author majianfu
 * @date 2021/7/23
 * @since
 */
@Component
@Slf4j
public class AppSendMsgServiceImpl implements AppSendMsgService {
    @Resource
    private RestTemplate restTemplate;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private AppOauthService appOauthService;

    /**
     * {@inheritDoc}
     */
    @Override
    public void sendAppMsg(JSONObject message) {
        if (null == message || message.isEmpty()) {
            return;
        }

        // 最大重试次数为1,，即：可发送2次请求
        int maxRetryTimes = 1;
        RetryableOperationToolkit.operate(maxRetryTimes, new RetryableAction<ResponseEntity<JSONObject>>() {
            @Override
            public ResponseEntity<JSONObject> active(int retryTime) {
                // retryTime == 0，为第一次请求，无需刷新缓存；retryTime > 0，即因为access_token过期，需刷新缓存重新获取access_token
                boolean flushCache = 0 == retryTime ? false : true;
                // 获取移动APP access_token
                AppTokenDTO appTokenDTO = appOauthService.getAccessToken(envProperties.getMobileProperties().getOauth(), flushCache);
                // 发送移动APP 通知消息
                return sendAppMsg(appTokenDTO.getAccess_token(), message);
            }

            @Override
            public boolean needRetry(ResponseEntity<JSONObject> respEntity) {
                if (isOK(respEntity)) {
                    // httpStatus == 200 && std_data.execution.sql_code == "0"，请求成功
                    return false;
                } else if (HttpStatus.UNAUTHORIZED.value() == respEntity.getStatusCodeValue()) {
                    // httpStatus == 401, access_token过期，需要重新获取access_token后再请求
                    return true;
                } else {
                    // 其他错误，无需重试，直接抛出业务异常
                    throw BusinessException.create(respEntity.getStatusCodeValue(), String.valueOf(respEntity.getBody()));
                }
            }
        });
    }

    private boolean isOK(ResponseEntity<JSONObject> respEntity) {
        // httpStatus == 200 && std_data.execution.sql_code == "0"，请求成功
        if (HttpStatus.OK.value() != respEntity.getStatusCodeValue()) {
            return false;
        }
        // 没有响应内容
        JSONObject body = respEntity.getBody();
        if (null == body || body.isEmpty() || !body.containsKey("std_data")) {
            return false;
        }

        JSONObject stdData = body.getJSONObject("std_data");
        if (null == stdData || stdData.isEmpty() || !stdData.containsKey("execution")) {
            return false;
        }

        JSONObject execution = stdData.getJSONObject("execution");
        if (null == execution || execution.isEmpty() || !execution.containsKey("sql_code")) {
            return false;
        }
        return "0".equals(execution.getString("sql_code"));
    }

    private ResponseEntity<JSONObject> sendAppMsg(String accessToken, JSONObject message) {
        String uri = UriComponentsBuilder.fromUriString(envProperties.getMobileProperties().getMessage().getUri())
                .queryParam("access_token", accessToken)
                .toUriString();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity(message, headers);
        ResponseEntity<JSONObject> resp = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, JSONObject.class);
        log.info("sendAppMsg success. req:{}, resp:{}", message, resp);
        return resp;
    }
}
