package com.digiwin.athena.aim.infrastructure.mongo;

import com.digiwin.athena.aim.domain.message.model.CiConstant;
import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.util.DistributeUtils;
import com.digiwin.athena.aim.util.ValidateUtils;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Indexes;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.redis.core.RedisTemplate;

import javax.annotation.Resource;
import java.util.concurrent.ConcurrentHashMap;

/**
 * AbstractMongoMessageMapper Description
 *
 * @author majianfu
 * @date 2021/4/26
 * @since
 */
public abstract class AbstractMongoMessageMapper {

    /**
     * 消息模板的集合
     */
    protected static final String TEMPLATE_COLLECTION_NAME = "template-collection";


    /**
     * 应用到期提醒的集合
     */
    protected static final String APP_EXPIRATION_MESSAGE_COLLECTION_NAME_SUFFIX = "-app-expire-message-collection";

    /**
     * 消息中心发送消息记录
     */
    protected static final String MESSAGE_CENTER_RECORD_COLLECTION_NAME_SUFFIX = "-message-center-record-collection";

    protected static final String COLLECTION_NAME_SUFFIX = "-message-collection";

    private static final String LOCK_NAME_SUFFIX = ":createMongoCollection:lock";

    protected static final String ORDER_CREATETIME = "createDate";

    protected static final long DEFAULT_LIMIT = 2000;


    private ConcurrentHashMap<String, String> existCollections = new ConcurrentHashMap<>();

    @Resource
    private RedisTemplate redisTemplate;

    @Resource
    protected MongoTemplate msgMongoTemplate;


    /**
     * 消息模板的集合
     *
     * @return
     */
    public String getCollectionNameForTempTemplate() {
        return TEMPLATE_COLLECTION_NAME;
    }

    /**
     * 应用到期提醒的集合
     *
     * @param tenantId 租户id
     * @return
     */
    public String getCollectionNameForAppExpirationMessage(String tenantId) {
        ValidateUtils.checkTenantId(tenantId);
        return StringUtils.trim(tenantId) + APP_EXPIRATION_MESSAGE_COLLECTION_NAME_SUFFIX;
    }


    public String getCollectionName(MessageDO messageDO) {
        return getCollectionName(messageDO.getTenantId());
    }

    public String getCollectionName(String tenantId) {
        ValidateUtils.checkTenantId(tenantId);
        return StringUtils.trim(tenantId) + COLLECTION_NAME_SUFFIX;
    }

    /**
     * 应用到期提醒的集合
     *
     * @param tenantId 租户id
     * @return
     */
    public String getCollectionNameForMessageCenterRecord(String tenantId) {
        ValidateUtils.checkTenantId(tenantId);
        return StringUtils.trim(tenantId) + MESSAGE_CENTER_RECORD_COLLECTION_NAME_SUFFIX;
    }

    /**
     * 创建集合
     *
     * @param collectionName 集合名称
     * @param collectionFlag 集合类型
     */
    protected void createCollectionNecessary(String collectionName, String collectionFlag) {
        if (existCollections.contains(collectionName)) {
            return;
        }

        if (msgMongoTemplate.collectionExists(collectionName)) {
            existCollections.put(collectionName, "1");
            return;
        }

        String lockName = collectionName + LOCK_NAME_SUFFIX;
        DistributeUtils.tryLock30s(redisTemplate, lockName, () -> {
            if (msgMongoTemplate.collectionExists(collectionName)) {
                existCollections.put(collectionName, "1");
                // 集合已存在
                return;
            }

            // 创建集合
            msgMongoTemplate.createCollection(collectionName);
            // 创建消息集合的索引
            if (StringUtils.isBlank(collectionFlag)) {
                createIndex(collectionName);
            } else if (CiConstant.CollectionFlagEnum.TEMPLATE.getFlag().equals(collectionFlag)) {   // 创建模板集合的索引
                createIndexForTemplateCollection(collectionName);
            } else if (CiConstant.CollectionFlagEnum.APP_EXPIRATION_MESSAGE.getFlag().equals(collectionFlag)) {   // 创建消息提醒集合的索引
                createIndexForAppExpirationCollection(collectionName);
            }
            existCollections.put(collectionName, "1");
        });
    }

    private void createIndex(String collectionName) {
        //该参数为索引的属性配置
        IndexOptions indexOptions = new IndexOptions()
                .unique(true)
                .background(true)
                .name("UNIQUE_IDX_GID");

        // 在gid上创建hashed类型索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("gid", -1), indexOptions);
        // 创建userid的索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("userId", 1));
        //创建createDate的索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("createDate", -1));

        Bson index1 = Indexes.compoundIndex(Indexes.ascending("userId"), Indexes.descending("createDate"));
        IndexOptions indexOptions1 = new IndexOptions().background(true);
        msgMongoTemplate.getCollection(collectionName).createIndex(index1, indexOptions1);

        Bson index2 = Indexes.compoundIndex(Indexes.ascending("userId"), Indexes.ascending("state"), Indexes.descending("createDate"));
        IndexOptions indexOptions2 = new IndexOptions().background(true);
        msgMongoTemplate.getCollection(collectionName).createIndex(index2, indexOptions2);
    }

    /**
     * 创建模板集合的索引
     *
     * @param collectionName 集合名称
     */
    private void createIndexForTemplateCollection(String collectionName) {
        // 该参数为索引的属性配置
        IndexOptions indexOptions = new IndexOptions().unique(true).background(true).name("UNIQUE_IDX_GID");
        // 在gid上创建hashed类型索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("gid", -1), indexOptions);
        // 创建模板code的索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("code", 1));
    }

    /**
     * 创建应用到期消息集合的索引
     *
     * @param collectionName 集合名称
     */
    private void createIndexForAppExpirationCollection(String collectionName) {
        // 该参数为索引的属性配置
        IndexOptions indexOptions = new IndexOptions().unique(true).background(true).name("UNIQUE_IDX_GID");
        // 在gid上创建hashed类型索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("gid", -1), indexOptions);
        // 创建userid的索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("userId", 1));
        //创建createDate的索引
        msgMongoTemplate.getCollection(collectionName).createIndex(new Document("createDate", -1));
    }

    /**
     * 通过聚合函数查询对应的统计总数
     *
     * @param collectionName 集合名称(租户信息）
     * @param criteria       查询条件
     * @param limit          限制数量
     * @param orderFields    降序排序字段（默认降序）
     * @return 统计结果
     */
    protected long countByCondition(String collectionName, Criteria criteria, long limit, String... orderFields) {
        // 创建一个Aggregation对象，定义聚合操作的入口点
        Aggregation aggregation = Aggregation.newAggregation(Aggregation.match(criteria), Aggregation.sort(Sort.Direction.DESC, orderFields), Aggregation.limit(limit),
                Aggregation.count().as("total"));
        AggregationResults<Document> result = msgMongoTemplate.aggregate(aggregation, collectionName, Document.class);
        if (CollectionUtils.isNotEmpty(result.getMappedResults())) {
            Document document = result.getMappedResults().get(0);
            return MapUtils.getLong(document, "total");
        } else {
            return 0;
        }

    }

}
