package com.digiwin.athena.aim.infrastructure.mongo;

import com.digiwin.athena.aim.domain.message.model.MessageDO;
import com.digiwin.athena.aim.domain.message.model.MsgStateEnum;
import com.digiwin.athena.aim.domain.message.repository.MessageMapper;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import static org.springframework.data.mongodb.core.aggregation.Aggregation.*;

@Component
public class MongoMessageMapper extends AbstractMongoMessageMapper implements MessageMapper {
    @Override
    public List<MessageDO> selectMsgSummaryBySubTypeCategory(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId())
                .and("subTypeCategory").is(condition.getSubTypeCategory());
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }

        Aggregation aggregation = Aggregation.newAggregation(match(criteria),
                sort(Sort.Direction.DESC, "createDate"));

        AggregationResults<MessageDO> result = msgMongoTemplate.aggregate(aggregation, getCollectionName(condition), MessageDO.class);
        return result.getMappedResults();
    }

    @Override
    public long deleteReadMsg(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId())
                .and("state").is(MsgStateEnum.READ.getState());
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }
        if (null != condition.getSubTypeCategory()) {
            criteria.and("subTypeCategory").is(condition.getSubTypeCategory());
        }

        Query query = new Query(criteria);
        DeleteResult result = msgMongoTemplate.remove(query, getCollectionName(condition));
        return result.getDeletedCount();
    }

    @Override
    public long changeStateToRead(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId())
                .andOperator(new Criteria().orOperator(Criteria.where("state").is(MsgStateEnum.UNREAD.getState()), Criteria.where("state").is(null)));

        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }
        if (null != condition.getSubTypeCategory()) {
            criteria.and("subTypeCategory").is(condition.getSubTypeCategory());
        }

        Query query = new Query(criteria);

        Update update = new Update();
        update.set("state", MsgStateEnum.READ.getState())
                .set("modifyDate", LocalDateTime.now());

        UpdateResult result = msgMongoTemplate.updateMulti(query, update, getCollectionName(condition));
        return result.getModifiedCount();
    }

    /**
     * 查询每个subTypeCategory分组类的数据量
     *
     * @param condition
     * @return
     */
    @Override
    public List<Map> selectUnreadCountGroupBySubTypeCategory(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId());
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }
        criteria.and("state").is(MsgStateEnum.UNREAD.getState());

        Aggregation aggregation = Aggregation.newAggregation(match(criteria),
                group("subTypeCategory").count().as("unreadCount"),
                project("unreadCount").and("subTypeCategory").previousOperation());

        AggregationResults<Map> results = msgMongoTemplate.aggregate(aggregation, getCollectionName(condition), Map.class);
        return results.getMappedResults();
    }

    /**
     * 查询subTypeCategory分组中的最新一条数据
     *
     * @param condition
     * @return
     */
    @Override
    public List<MessageDO> selectLatestGroupBySubTypeCategory(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId());
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }

        Aggregation aggregation = Aggregation.newAggregation(
                // 第一步：挑选所需的字段，类似select *，*所代表的字段内容
//                Aggregation.project("_id", "gid", "userId", "tenantId", "type", "subType",
//                        "subTypeCategory", "category", "importance", "source", "state", "title", "content", "sendDate", "createDate"),
                // 第二步：sql where 语句筛选符合条件的记录
                match(criteria),
                // 第三步：order by，由于createDate上没有创建索引，并且_id的顺序和createDate的顺序都是递增，因此根据_id倒序排序即可
                sort(Sort.Direction.DESC, "gid"),
                // 第四步：分组，并挑出每个分组的第一条记录
                group("subTypeCategory")
                        .first("gid").as("gid").first("userId").as("userId").first("tenantId").as("tenantId")
                        .first("type").as("type").first("subType").as("subType").first("subTypeCategory").as("subTypeCategory")
                        .first("category").as("category").first("importance").as("importance").first("source").as("source")
                        .first("state").as("state").first("title").as("title").first("content").as("content")
                        .first("sendDate").as("sendDate").first("createDate").as("createDate"),
                // 第五步：重新挑选字段
                project(MessageDO.class)
//                Aggregation.project("gid", "userId", "tenantId", "type", "subType",
//                        "subTypeCategory", "category", "importance", "source", "state", "title", "content", "sendDate", "createDate")
        );

        AggregationResults<MessageDO> results = msgMongoTemplate.aggregate(aggregation, getCollectionName(condition), MessageDO.class);
        return results.getMappedResults();
    }

    @Override
    public List<MessageDO> selectWorkNewsMsgSummary(MessageDO condition) {
        Criteria criteria = Criteria.where("userId").is(condition.getUserId());
        Aggregation aggregation = Aggregation.newAggregation(match(criteria),
                sort(Sort.Direction.DESC, "createDate"));

        AggregationResults<MessageDO> result = msgMongoTemplate.aggregate(aggregation, getCollectionName(condition), MessageDO.class);
        return result.getMappedResults();
    }

    @Override
    public MessageDO insert(MessageDO messageDO) {
        String collectionName = getCollectionName(messageDO);

        LocalDateTime createDate = LocalDateTime.now();
        messageDO.setCreateDate(createDate);
        messageDO.setModifyDate(createDate);
        messageDO.setHasReadDetail(Boolean.FALSE);

        // 在gid上创建索引
        createCollectionNecessary(collectionName, null);
        return msgMongoTemplate.insert(messageDO, collectionName);
    }

    @Override
    public long selectUnreadCount(MessageDO condition) {
        if (StringUtils.isEmpty(condition.getTenantId())) {
            //如果没有租户，不进行进一步处理，因为供应商登录的时候，就没有租户ID
            return 0L;
        }

        Criteria criteria = Criteria
                .where("userId").is(condition.getUserId())
                .and("state").is(MsgStateEnum.UNREAD.getState());
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }
        if (!StringUtils.isEmpty(condition.getChannelType())) {
            Criteria channelTypeCriteria = new Criteria().orOperator(
                    Criteria.where("channelType").is(condition.getChannelType()),  // channelType 等于特定值
                    Criteria.where("channelType").exists(false)                    // channelType 字段不存在
            );
            criteria.andOperator(channelTypeCriteria);
        }

        return countByCondition(getCollectionName(condition), criteria, DEFAULT_LIMIT, ORDER_CREATETIME);
    }

}
