package com.digiwin.athena.aim.infrastructure.mongo;


import com.digiwin.athena.aim.api.dto.Page;
import com.digiwin.athena.aim.domain.message.model.*;
import com.digiwin.athena.aim.domain.message.repository.MessageMapperV2;
import com.digiwin.athena.aim.util.DateUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.collect.Lists;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Repository;

import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * MessageMapperV2Impl Description
 *
 * @author majianfu
 * @date 2021/4/26
 * @since
 */
@Slf4j
@Repository
public class MongoMessageMapperV2 extends AbstractMongoMessageMapper implements MessageMapperV2 {

    @Override
    public PageImpl<MessageDO> pagingQuery(MessageDO condition, Page page) {
//        Pageable pageable = PageRequest.of(page.getPageNum(), page.getPageSize(), Sort.by(Sort.Direction.DESC, "gid"));
        Pageable pageable = PageRequest.of(page.getPageNum(), page.getPageSize());
        if (StringUtils.isEmpty(condition.getTenantId())) {
            return new PageImpl<>(new ArrayList<>(), PageRequest.of(page.getPageNum(), page.getPageSize()), 0);
        }
        Criteria criteria = Criteria.where("userId").is(condition.getUserId());
        criteria.and("deleted").ne(Boolean.TRUE);
        if (StringUtils.isNotBlank(condition.getType())) {
            criteria.and("type").is(condition.getType());
        }
        if (StringUtils.isNotBlank(condition.getSubType())) {
            criteria.and("subType").is(condition.getSubType());
        }
        if (StringUtils.isNotBlank(condition.getSubTypeCategory())) {
            criteria.and("subTypeCategory").is(condition.getSubTypeCategory());
        }
        if (StringUtils.isNotBlank(condition.getCategory())) {
            criteria.and("category").is(condition.getCategory());
        }
        if (null != condition.getImportance()) {
            criteria.and("importance").is(condition.getImportance());
        }
        if (StringUtils.isNotBlank(condition.getSource())) {
            criteria.and("source").is(condition.getSource());
        }
        if (null != condition.getState()) {
            criteria.and("state").is(condition.getState());
        }

        if (StringUtils.isNotBlank(condition.getChannelType())) {
            Criteria channelTypeCriteria = new Criteria().orOperator(
                    Criteria.where("channelType").is(condition.getChannelType()),  // channelType 等于特定值
                    Criteria.where("channelType").exists(false)                    // channelType 字段不存在
            );
            criteria.andOperator(channelTypeCriteria);
        }

        if (StringUtils.isNotBlank(condition.getStartTimeStr()) || StringUtils.isNotBlank(condition.getEndTimeStr())) {
            if (StringUtils.isNotBlank(condition.getStartTimeStr()) &&  StringUtils.isNotBlank(condition.getEndTimeStr())) {
                criteria.and("createDate").gte(DateUtils.StringToDate(condition.getStartTimeStr()+" 00:00:00"))
                        .lte(DateUtils.StringToDate(condition.getEndTimeStr()+" 23:59:59"));
            }
            // 如果 startDate 不为空，且 endDate 为空，则只添加 $gte 条件
            else if(StringUtils.isNotBlank(condition.getStartTimeStr())){
                criteria.and("createDate").gte(DateUtils.StringToDate(condition.getStartTimeStr()+" 00:00:00"));
            }
            // 如果 endDate 不为空，且 startDate 为空，则只添加 $lte 条件
            else if (StringUtils.isNotBlank(condition.getEndTimeStr())) {
                criteria.and("createDate").lte(DateUtils.StringToDate(condition.getEndTimeStr()+" 23:59:59"));
            }
        }
        //关键词搜索
        if (StringUtils.isNotBlank(condition.getKey())) {
            List<String> gidList=filterByKey(criteria,condition);
            //关键词搜索结果为空，直接返回
            if(CollectionUtils.isEmpty(gidList)){
                return new PageImpl<>(new ArrayList<MessageDO>(), pageable, 0L);
            }
            criteria.and("gid").in(gidList);
        }
        long count = countByCondition(getCollectionName(condition), criteria, DEFAULT_LIMIT, ORDER_CREATETIME);
        if (BooleanUtils.isTrue(condition.getUnterminate())){
            //过滤终止消息  且和前端约定 当unterminate是true时,category并不会作为传参
            criteria.and("category").ne("TERMINATE");
        }

        List<MessageDO> messageList = msgMongoTemplate.find(new Query().addCriteria(criteria).with(pageable).with(Sort.by(Sort.Direction.DESC, "createDate")),
                MessageDO.class,
                getCollectionName(condition));

        return new PageImpl<>(messageList, pageable, count);
    }

    @Override
    public MessageRemoveDTO updateDeleted(String channelType, AuthoredUser authoredUser) {
        Criteria criteria = Criteria.where("userId").is(authoredUser.getUserId())
                .and("state").is(MsgStateEnum.READ.getState())
                .and("deleted").ne(Boolean.TRUE);

        if (StringUtils.isNotBlank(channelType)) {
            Criteria channelTypeCriteria = new Criteria().orOperator(
                    Criteria.where("channelType").is(channelType),  // channelType 等于特定值
                    Criteria.where("channelType").exists(false)  // channelType 字段不存在
            );
            criteria.andOperator(channelTypeCriteria);
        }

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC, "createDate"));
        query.limit(1000);
        List<MessageDO> messageList = msgMongoTemplate.find(query,
                MessageDO.class,
                getCollectionName(authoredUser.getTenantId()));
        int size = messageList.size();

        if (size != 0){
            // 分批处理 防止超时
            List<String> gids = messageList.stream()
                    .map(MessageDO::getGid)
                    .collect(Collectors.toList());
            log.error("提示详细清除缓存总条数:{}",gids.size());
            List<List<String>> parts = Lists.partition(gids, 100);
            log.error("提示详细清除缓存parts条数:{}",parts.size());

            parts.stream().forEach(list -> {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                Date now = new Date();
                String formattedDate = sdf.format(now);
               log.error("提示详细清除缓存条数:{},时间是:{}",list.size(),formattedDate);

                Query updateQuery = new Query(Criteria.where("gid").in(list));
                Update update = new Update();
                update.set("deleted",Boolean.TRUE);
                msgMongoTemplate.updateMulti(updateQuery, update, getCollectionName(authoredUser.getTenantId()));
            });

        }

        MessageRemoveDTO messageRemoveDTO= new MessageRemoveDTO();
        messageRemoveDTO.setRemoveNum(size);
        return messageRemoveDTO;
    }

    /**
     *  根据关键词过滤得出符合条件的key
     * @param criteria
     * @param condition
     * @return
     */
    public List<String> filterByKey(Criteria criteria,MessageDO condition){
        List<String> gidList=new ArrayList<>();
        List<MessageDO> messageListAll = msgMongoTemplate.find(new Query().addCriteria(criteria).with(Sort.by(Sort.Direction.DESC, "createDate")),
                MessageDO.class,
                getCollectionName(condition));
        if(CollectionUtils.isEmpty(messageListAll)){
            return gidList;
        }
        for(MessageDO message:messageListAll){
            String content=JsonUtils.objectToString(message.getContent());
            if(content.contains(condition.getKey())){
                gidList.add(message.getGid());
            }
        }
        return gidList;
    }

    @Override
    public long readMessageByGid(String channelType, List<String> gidList, AuthoredUser authoredUser) {
        if (StringUtils.isEmpty(authoredUser.getTenantId())) {
            //如果没有租户，不进行进一步处理，因为供应商登录的时候，就没有租户ID
            return 0L;
        }
        List<Criteria> criteriaList = new ArrayList<>();
        criteriaList.add(Criteria.where("userId").is(authoredUser.getUserId()));

        Criteria stateCriteria = new Criteria().orOperator(
                Criteria.where("state").is(MsgStateEnum.UNREAD.getState()),
                Criteria.where("state").is(null)
        );

//        Criteria criteria = Criteria.where("userId").is(authoredUser.getUserId());
        if (CollectionUtils.isNotEmpty(gidList)) {
            criteriaList.add(new Criteria().andOperator(
                    Criteria.where("gid").in(gidList),
                    stateCriteria
            ));
//            criteria.and("gid").in(gidList)
//                    .andOperator(new Criteria().orOperator(Criteria.where("state").is(MsgStateEnum.UNREAD.getState()), Criteria.where("state").is(null)));
        } else {
            criteriaList.add(stateCriteria);
//            criteria.andOperator(new Criteria().orOperator(Criteria.where("state").is(MsgStateEnum.UNREAD.getState()), Criteria.where("state").is(null)));
        }

        if (StringUtils.isNotBlank(channelType)) {
            criteriaList.add(new Criteria().orOperator(
                    Criteria.where("channelType").is(channelType),
                    Criteria.where("channelType").exists(false)
            ));
        }
        Criteria finalCriteria = new Criteria().andOperator(criteriaList.toArray(new Criteria[0]));

        Query query = new Query(finalCriteria);

        Update update = new Update();
        update.set("state", MsgStateEnum.READ.getState())
                .set("modifyDate", LocalDateTime.now());

        UpdateResult result = msgMongoTemplate.updateMulti(query, update, getCollectionName(authoredUser.getTenantId()));
        return result.getModifiedCount();
    }

    @Override
    public long readMessageDetailByGid(String gid, AuthoredUser authoredUser) {
        if (StringUtils.isEmpty(authoredUser.getTenantId())) {
            //如果没有租户，不进行进一步处理，因为供应商登录的时候，就没有租户ID
            return 0L;
        }

        Criteria criteria = Criteria.where("userId").is(authoredUser.getUserId())
                .and("gid").is(gid);

        Query query = new Query(criteria);

        Update update = new Update();
        update.set("hasReadDetail", Boolean.TRUE)
                .set("modifyDate", LocalDateTime.now());

        UpdateResult result = msgMongoTemplate.updateMulti(query, update, getCollectionName(authoredUser.getTenantId()));
        return result.getModifiedCount();
    }

    @Override
    public void batchInsert(List<MessageDO> messageList) {
        if (CollectionUtils.isEmpty(messageList)) {
            return;
        }
        String collectionName = getCollectionName(messageList.get(0));
        // 在gid上创建索引
        createCollectionNecessary(collectionName, null);

        LocalDateTime createDate = LocalDateTime.now();
        messageList.forEach(message -> {
            message.setCreateDate(createDate);
            message.setModifyDate(createDate);
            message.setHasReadDetail(Boolean.FALSE);
            message.setState(MsgStateEnum.UNREAD.getState());
        });

        // 并行批量插入
        BulkOperations operations = this.msgMongoTemplate.bulkOps(BulkOperations.BulkMode.UNORDERED, collectionName);
        operations.insert(messageList);
        operations.execute();
    }

    @Override
    public void clearProjectAndTaskMessage(String tenantId, List<String> projectCodeList, List<String> taskCodeList) {
        if (CollectionUtils.isNotEmpty(projectCodeList)) {
            Criteria projectCri = Criteria.where("type").is("task")
                    .and("subType").in(projectCodeList);
            this.msgMongoTemplate.remove(new Query(projectCri), getCollectionName(tenantId));
        }

        if (CollectionUtils.isNotEmpty(taskCodeList)) {
            Criteria taskCri = Criteria.where("type").is("activity")
                    .and("subType").in(taskCodeList);
            this.msgMongoTemplate.remove(new Query(taskCri), getCollectionName(tenantId));
        }
    }

    /**
     * 应用到期提醒 批量发送消息
     *
     * @param messageList 消息
     */
    @Override
    public void insertAppExpirationMessage(List<ExpireMessageReq.MessageDTO> messageList) {
        if (CollectionUtils.isEmpty(messageList)) {
            return;
        }
        String collectionName = getCollectionNameForAppExpirationMessage(messageList.get(0).getTenantId());
        // 在gid上创建索引
        createCollectionNecessary(collectionName, CiConstant.CollectionFlagEnum.APP_EXPIRATION_MESSAGE.getFlag());

        LocalDateTime createDate = LocalDateTime.now();
        messageList.forEach(message -> {
            message.setCreateDate(createDate);
            message.setModifyDate(createDate);
        });
        // 并行批量插入
        BulkOperations operations = this.msgMongoTemplate.bulkOps(BulkOperations.BulkMode.UNORDERED, collectionName);
        operations.insert(messageList);
        operations.execute();
    }

    /**
     * 应用到期提醒 查询消息列表
     *
     * @param condition 查询条件
     */
    @Override
    public List<MessageDO> queryAppExpirationMessage(MessageDO condition) {
        // 查询当前用户 + 未到应用截止提醒时间的消息
        Criteria criteria = Criteria.where("userId").is(condition.getUserId())
                .and("endTime").gte(LocalDateTime.now());
        Query query = new Query(criteria);

        String collectionName = getCollectionNameForAppExpirationMessage(condition.getTenantId());
        List<MessageDO> messageDOList = msgMongoTemplate.find(query.with(Sort.by(Sort.Direction.ASC, "createDate")),
                MessageDO.class, collectionName);
        log.info("MongoMessageMapperV2 query app expiration message success. result:{}", messageDOList);

        // 更新时间，目的是前端每天要展示一次提醒，根据跟更新时间是否是当天做展示
        if (CollectionUtils.isNotEmpty(messageDOList)) {
            Update update = new Update();
            update.set("modifyDate", LocalDateTime.now());
            UpdateResult result = msgMongoTemplate.updateMulti(query, update, collectionName);
        }
        return messageDOList;
    }

    /**
     * 应用到期提醒 应用续约，即删除该应用的到期提醒
     *
     * @param appExpireMessageDO 请求体
     * @return
     */
    @Override
    public void deleteAppExpirationMessage(AppExpireMessageDO appExpireMessageDO) {
        String collectionName = getCollectionNameForAppExpirationMessage(appExpireMessageDO.getTenantId());
        Criteria criteria = Criteria.where("appCode").is(appExpireMessageDO.getAppCode());
        msgMongoTemplate.remove(new Query(criteria), collectionName);
    }
}
