package com.digiwin.athena.aim.infrastructure.semc.impl;

import com.digiwin.athena.aim.app.env.EnvProperties;
import com.digiwin.athena.aim.common.InterfaceConstant;
import com.digiwin.athena.aim.infrastructure.semc.SemcService;
import com.digiwin.athena.aim.infrastructure.semc.dto.*;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.List;

/**
 * semc service
 *
 * @author sungqz
 * @since 2023-12-4
 */
@Slf4j
@Service
public class SemcServiceImpl implements SemcService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 查询租户设置的消息发送渠道
     *
     * @return 渠道
     */
    @Override
    public String queryMessageChannel(String tenantId) {
        String url = envProperties.getSemcUri() + InterfaceConstant.SEMC_TENANT_MESSAGE_CHANNEL + "?tenantId=" + tenantId;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        JSONObject reqObject = JSONObject.fromObject(this);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);

        ResponseEntity<BaseResultDTO<String>> respEntity;
        try {
            respEntity = this.restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
        } catch (BusinessException ex) {
            log.error("query message channel error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            return null;
        }
        // 获取消息的发送渠道
        if (respEntity.getStatusCodeValue() == HttpStatus.OK.value() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
            return respEntity.getBody().getResponse();
        } else {
            log.error("query message channel fail. status:{}, url:{}, param:{}, result:{}", respEntity.getStatusCodeValue(), url, JsonUtils.objectToString(httpEntity), respEntity.getBody());
            return null;
        }
    }

    /**
     * 查询各渠道模板信息
     *
     * @param channelFlag 渠道标识 email:邮件 sms:短信 wecom:企微 dingTalk:钉钉
     * @param eventId     事件id
     * @param tenantId    租户id
     * @return 对象
     */
    @Override
    public List<QueryTemplateListResp> queryTemplateList(String channelFlag, String eventId, String remark, String tenantId) {
        String url = envProperties.getSemcUri() + InterfaceConstant.SEMC_TENANT_MESSAGE_TEMPLATE;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        JSONObject reqObject = JSONObject.fromObject(this);
        reqObject.put("channelFlag", channelFlag);
        reqObject.put("eventId", eventId);
        reqObject.put("remark", remark);
        reqObject.put("tenantId", tenantId);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);

        ResponseEntity<BaseResultDTO<List<QueryTemplateListResp>>> respEntity;
        try {
            log.info("start query template list. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<QueryTemplateListResp>>>() {
            });
        } catch (BusinessException ex) {
            log.error("query template list error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            throw BusinessException.create("query template list error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
        }
        if (respEntity.getStatusCodeValue() == HttpStatus.OK.value() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
            return respEntity.getBody().getResponse();
        } else {
            log.error("query template list fail. status:{}, url:{}, param:{}, result:{}", respEntity.getStatusCodeValue(), url, JsonUtils.objectToString(httpEntity), respEntity.getBody());
            throw BusinessException.create("query template list fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
        }
    }

    /**
     * 查看模板接收人
     *
     * @param tenantId    租户id
     * @param eventId     事件id
     * @param channelFlag 渠道标识 email:邮件 sms:短信 wecom:企微 dingTalk:钉钉
     * @param templateId 模板主键id
     * @param language    语系 简体：zh-CN 繁体：zh-TW 英文：en-US
     * @return 返回对象
     */
    @Override
    public QueryTemplateRelResp queryRelPerson(String tenantId, String eventId, String channelFlag, String templateId, String language) {
        String url = envProperties.getSemcUri() + InterfaceConstant.SEMC_TENANT_MESSAGE_REL_PERSON;

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        JSONObject reqObject = JSONObject.fromObject(this);
        reqObject.put("tenantId", tenantId);
        reqObject.put("templateId", templateId);
        reqObject.put("channelFlag", channelFlag);
        reqObject.put("eventId", eventId);
        reqObject.put("language", language);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);

        ResponseEntity<BaseResultDTO<QueryTemplateRelResp>> respEntity;
        try {
            log.info("start semc query rel person list. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<QueryTemplateRelResp>>() {
            });
        } catch (BusinessException ex) {
            log.error("query semc rel person error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            throw BusinessException.create("query rel person error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
        }
        if (respEntity.getStatusCodeValue() == HttpStatus.OK.value() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
            return respEntity.getBody().getResponse();
        } else {
            log.error("query semc rel person fail. status:{}, url:{}, param:{}, result:{}", respEntity.getStatusCodeValue(), url, JsonUtils.objectToString(httpEntity), respEntity.getBody());
            throw BusinessException.create("query rel person fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
        }
    }

    /**
     * 查询黑白名单用户id
     *
     * @param tenantId 租户id
     * @param type     名单类型 0:黑名单 1:白名单
     * @param key      搜索关键字
     * @return 返回对象
     */
    @Override
    public List<QueryBlackWhiteListResp> queryBlackWhiteUserList(String tenantId, Integer type, String key) {
        key = StringUtils.isBlank(key) ? StringUtils.EMPTY : key;
        StringBuilder stringBuilder = new StringBuilder(envProperties.getSemcUri()).append(InterfaceConstant.SEMC_TENANT_BLACK_WHITE);
        StringBuilder url = type == null ? stringBuilder.append("?type=&key=").append(key).append("&tenantId=").append(tenantId)
                : stringBuilder.append("?type=").append("&key=").append(key).append("&tenantId=").append(tenantId);

        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        JSONObject reqObject = JSONObject.fromObject(this);
        HttpEntity<JSONObject> httpEntity = new HttpEntity<>(reqObject, headers);

        ResponseEntity<BaseResultDTO<List<QueryBlackWhiteListResp>>> respEntity;
        try {
            log.info("start query blackWhite user list. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            respEntity = this.restTemplate.exchange(url.toString(), HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<QueryBlackWhiteListResp>>>() {
            });
        } catch (BusinessException ex) {
            log.error("query blackWhite user list error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            throw BusinessException.create("query blackWhite user list error:" + ex.getMessage() + ", param:" + JsonUtils.objectToString(httpEntity));
        }
        if (respEntity.getStatusCodeValue() == HttpStatus.OK.value() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
            return respEntity.getBody().getResponse();
        } else {
            log.error("query blackWhite user list fail. status:{}, url:{}, param:{}, result:{}", respEntity.getStatusCodeValue(), url, JsonUtils.objectToString(httpEntity), respEntity.getBody());
            throw BusinessException.create("query blackWhite user list fail. param:" + JsonUtils.objectToString(httpEntity) + ", result:" + respEntity);
        }
    }

    @Override
    public GetSsoUrlResp getSsoUrl(GetSsoUrlReq req) {
        String url = envProperties.getSemcUri() + InterfaceConstant.SEMC_GET_SSO_URL;
        HttpHeaders headers = new HttpHeaders();
        headers.add("digi-middleware-auth-user", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        HttpEntity<GetSsoUrlReq> httpEntity = new HttpEntity<>(req, headers);
        ResponseEntity<BaseResultDTO<GetSsoUrlResp>> respEntity;
        try {
            log.info("start semc getSsoUrlt. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity));
            respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<GetSsoUrlResp>>() {
            });
        } catch (BusinessException ex) {
            log.error("getSsoUrl error. url:{}, param:{}", url, JsonUtils.objectToString(httpEntity), ex);
            return null;
        }
        if (respEntity.getStatusCodeValue() == HttpStatus.OK.value() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
            return respEntity.getBody().getResponse();
        } else {
            log.error("getSsoUrl fail. status:{}, url:{}, param:{}, result:{}", respEntity.getStatusCodeValue(), url, JsonUtils.objectToString(httpEntity), respEntity.getBody());
            return null;
        }
    }
}

