package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.annotate.OperateAuthorityVerify;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtApprovedStatusDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.enums.APIExportImportEnums;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtMdcProp;
import com.digiwin.athena.apimgmt.model.ApprovedStatus;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.service.util.DmcFileServiceUtil;
import com.digiwin.athena.apimgmt.service.util.StandardApiExportServiceUtil;
import com.digiwin.athena.apimgmt.service.util.StandardApiFileReadServiceUtil;
import com.digiwin.athena.apimgmt.services.ApiMgmtApiVersionService;
import com.digiwin.athena.apimgmt.services.ApiMgmtSyncMdcServiceProcessor;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.FileOutputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * API定版
 * /restful/standard/apimgmt/ApiComfirmed/Update
 */
@Slf4j
@Service
@OperateAuthorityVerify
public class ApiComfirmedUpdateService extends AbstractApiService {

    @Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    ApiMgmtApprovedStatusDao approvedStatusDao;

    @Autowired
    private ApiMgmtSyncMdcServiceProcessor syncMdcServiceProcessor;

    @Autowired
    private ApiMgmtApiVersionService apiVersionService;

    @Autowired
    private ApiMgmtMdcProp mdcProp;

    public ApiComfirmedUpdateService() {
        super();
        jsonSchemaFileName = ApimgmtSchemaConstant.API_REVIEW_SCHEMA;
    }

    /**
     * 定版
     *
     * @param validatorResult validatorResult
     * @return map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = tStateCode.getDescription();
        ArrayList<Long> tListApiVerIdNeedToSyncMetadata = new ArrayList<>();
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.API_LIST);
        // 取得語系
        String tLocale = ApiMgmtServiceContextHolder.getLocale();
        // header沒傳語系的話，默認回傳英文
        tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
        // by songwq 20230625 根据token获取租户ID，为了后续检查定版API是否为当前设计租户ID校验
        String tToken =  ApiMgmtServiceContextHolder.getToken();
        String routerKey = ApiMgmtServiceContextHolder.getRouterKey();
        String tTenantId = ApiMgmtServiceContextHolder.getTenantId();
        String tUserId = ApiMgmtServiceContextHolder.getUserId();
        Map<String, List<Map<String, String>>> tTotalApiErrorMap = new HashMap<>();
        for (JsonNode tApiNode : tApiArrayNode) {
            String apiName = tApiNode.get(ApiAttributeEnum.apiName.toString()).asText();
            String version = tApiNode.get(ApiAttributeEnum.version.toString()).asText();
            Map<String, String> tConditionMap = new HashMap<>();
            tConditionMap.put(ApiAttributeEnum.apiName.toString(), apiName);
            tConditionMap.put(ApiAttributeEnum.version.toString(), version);
            if (tApiNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
                tConditionMap.put(ApiAttributeEnum.tenantId.toString(), tApiNode.get(ApiAttributeEnum.tenantId.toString()).asText());
            }
            StandardApiVersion tStandardApiVersion = standardApiVersionDao.fetchApiVersion(tConditionMap);
            // by songwq 20230625 增加检查定版API是否为当前设计租户ID校验，如果不是则不支持定版
            // by songwq 20230724 fix: 修复bug59700 处理历史数据设计租户ID为空的情况 对于历史设计者租户id为空支持进版
            if (tStandardApiVersion != null && tStandardApiVersion.getStandardApi() != null
                    && (StringUtil.isEmpty(tStandardApiVersion.getStandardApi().getDesignTenantId())
                    || tTenantId.equals(tStandardApiVersion.getStandardApi().getDesignTenantId()))) {
                List<Map<String, String>> tSetStdApiResult = new ArrayList<>();
                //if (Float.parseFloat(version) > 1.0) {
                if (!apiVersionService.judgeNewBranch(version)) {
                    String tPreviousVersion = ApiVersionServiceUtil.getPreviousVersion(version);
                    tConditionMap.put(ApiAttributeEnum.version.toString(), tPreviousVersion);
                    StandardApiVersion standardApiVersion = standardApiVersionDao.fetchApiVersion(tConditionMap);
                    if (null != standardApiVersion && null != standardApiVersion.getApprovedStatus()
                            && 4L != standardApiVersion.getApprovedStatus().getId()) {
                        // 异常 当前API的上一个版本未定版
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREVI_IS_DEVELOPING.toString(), null, null, 0, 0, tLocale);
                        String tApiError = apiName + "/" + version;
                        tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                    } else {
                        // 已定版
                        ApprovedStatus tApprovedStatus = approvedStatusDao.get(4L);
                        tStandardApiVersion.setApprovedStatus(tApprovedStatus);
                        tStandardApiVersion.setApprovedTime(LocalDateTime.now());
                        standardApiVersionDao.saveOrupdate(tStandardApiVersion);
                        tListApiVerIdNeedToSyncMetadata.add(tStandardApiVersion.getId());
                    }
                } else {
                    //查询父类API是否已经定版
                    StandardApi standardApi = tStandardApiVersion.getStandardApi();
                    if (!StringUtil.isEmptyOrSpace(standardApi.getParentApiName())) {
                        List<StandardApiVersion> standardApiVersionList = standardApiVersionDao.fetchApiVersionsByName(standardApi.getParentApiName(), standardApi.getParentBranch(), standardApi.getParentApiTenantId());
                        standardApiVersionList = standardApiVersionList.stream().filter(x -> x.getApprovedStatus().getId() == ApimgmtConstant.APPROVED_STATUS_COMFIRMED).collect(Collectors.toList());
                        if (CollUtil.isEmpty(standardApiVersionList)) {
                            StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PARENT_API_NOT_CONFIRMED.toString(), null, null, 0, 0, tLocale);
                            String tApiError = apiName + "/" + version;
                            tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                            continue;
                        }
                    }
                    // 已定版
                    ApprovedStatus tApprovedStatus = approvedStatusDao.get(4L);
                    tStandardApiVersion.setApprovedStatus(tApprovedStatus);
                    tStandardApiVersion.setApprovedTime(LocalDateTime.now());
                    standardApiVersionDao.saveOrupdate(tStandardApiVersion);
                    tListApiVerIdNeedToSyncMetadata.add(tStandardApiVersion.getId());
                }
            }
        }
        if (!tListApiVerIdNeedToSyncMetadata.isEmpty() && CollUtil.isNotEmpty(mdcProp.getConfirmedUrlList())) {
            // 同步MDC元數據 定版只会同步正式区
            for (String url : mdcProp.getConfirmedUrlList()) {
                syncMdcServiceProcessor.runSyncMdcMetadata(
                        tListApiVerIdNeedToSyncMetadata, url, tUserId, tToken, routerKey);
            }
        }
        if (MapUtil.isNotEmpty(tTotalApiErrorMap)) {
            log.info("tTotalApiErrorMap: " + tTotalApiErrorMap);
            return result(tStateCode, tDescription, tTotalApiErrorMap);
        }
        Map<String, Object> tResponse = new HashMap<>();
        // 組成回傳訊息
        tResponse.put(ApimgmtConstant.CODE, "000");
        tResponse.put(ApimgmtConstant.RESULT, ApimgmtConstant.SUCCESS);
        return tResponse;
    }

    private Map<String, Object> result(StateCode tStateCode, String tDescription,
                                       Map<String, List<Map<String, String>>> pTotalApiErrorMap) throws Exception {
        ObjectNode tResponseNode = mapper.createObjectNode();
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
        tStateCode = getStateCode(ValidateStateEnum.IMPORT_SPEC_FORMAT_ERROR.getCode());
        tDescription = tStateCode.getDescription();
        // 取得樣板
        HSSFWorkbook tHSSFWorkbook = StandardApiExportServiceUtil.getTemplateFile(false);
        Sheet tErrorSheet = tHSSFWorkbook.createSheet("錯誤明細");
        StandardApiFileReadServiceUtil.writeImportErrorFile(tErrorSheet, pTotalApiErrorMap);
        String tFileName = ApiMgmtApplicationParameter._TEMPLATE_FILE_NAME;
        String tFileExportPath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + tFileName;
        FileOutputStream fileOut = new FileOutputStream(tFileExportPath);
        tHSSFWorkbook.write(fileOut);
        fileOut.close();
        int pFailedNum = pTotalApiErrorMap.keySet().size();
        //上傳檔案並提供下載連結
        DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tFileExportPath, tFileName, true, pFailedNum);
        tResponseNode.put("code", tStateCode.getCode());
        tResponseNode.put("description", tDescription);
        return converJsonNodeToMap(tResponseJsonNode);
    }
}
