package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.dto.ApiParamTextResDto;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.auth.IApiMgmtIdentityService;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.util.ApiDataNameServiceUtil;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.services.ApiMgmtIamVerifyService;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 取得單支API詳細信息
 * /restful/standard/apimgmt/ApiDetail/Get
 */
@Slf4j
@Service
public class ApiDetailGetService extends AbstractApiService {

    @Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    ApiMgmtStandardApiDataNameDao standardApiDataNameDao;

    @Autowired
    ApiMgmtTenantConfigDao tenantConfigDao;

    @Autowired
    private ApiMgmtProjectVersionRelationDao projectVersionRelationDao;

    @Autowired
    private ApiMgmtProjectDao projectDao;
    @Autowired
    private ApiMgmtIamVerifyService iamVerifyService;

    @Autowired
    private ApiMgmtProp prop;

    @Autowired
    private ApiMgmtProjectVersionDao projectVersionDao;
    @Autowired
    private ApiParamTextService apiParamTextService;
    @Autowired
    private IApiMgmtIdentityService identityService;

    public ApiDetailGetService() {
        super();
        jsonSchemaFileName = ApimgmtSchemaConstant.API_DETAIL_GET_SCHEMA;
    }

    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, IOException {

        // 取得語系
        String tLocale = ApiMgmtServiceContextHolder.getLocale();
        // header沒傳語系的話，默認回傳英文
        tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
        // 取得request node
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        // 建立response node
        ObjectNode tResponseNode = mapper.createObjectNode();
        StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
        // 取得入參api名稱
        String tApiName = tRequestJsonNode.get(ApiAttributeEnum.apiName.toString()).asText();
        // 取得入參api版號
        String tVersion = "";
        String branch = ApimgmtConstant.DEFAULT_BRANCH;
        // 存在则获取当前版本讯息 不存在则获取最新版本讯息
        if (tRequestJsonNode.has(ApiAttributeEnum.version.toString())) {
            tVersion = tRequestJsonNode.get(ApiAttributeEnum.version.toString()).asText();
            // branch
            branch = ApiVersionServiceUtil.getBranchFromVersion(tVersion);
        }
        String tTenant = "";
        // 取得入參租戶id
        if (tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
            tTenant = tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()).asText();
        }
        String pTenantId = ApiMgmtServiceContextHolder.getTenantId();
        String teamType = ApiMgmtServiceContextHolder.getTeamType();
        // 判斷租戶是否可查看全部API
        boolean tViewAllApi = tenantConfigDao.viewAllApiTenantExist(pTenantId);
        // 找出api的所有版本號
        List<StandardApiVersion> tStandardApiVersionList = standardApiVersionDao.fetchApiVersionsByName(tApiName, branch, pTenantId, tTenant, teamType, tViewAllApi);
        // 找出入參對應版本號的相關資訊
        StandardApiVersion tStandardApiVersion = null;
        if (CollUtil.isEmpty(tStandardApiVersionList)) {
            // 需要兼容paas环境的历史数据
            if (prop.isEnvType() && !tApiName.startsWith(ApimgmtConstant.API_PAAS_REGEX)) {
                tApiName = ApimgmtConstant.API_PAAS_REGEX + tApiName;
                tStandardApiVersionList = standardApiVersionDao.fetchApiVersionsByName(tApiName, branch, pTenantId, tTenant, teamType, tViewAllApi);
            }
        }
        if (StrUtil.isNotBlank(tVersion)) {
            for (StandardApiVersion standardApiVersion : tStandardApiVersionList) {
                // 有找到對應版本號數據
                if (standardApiVersion.getVersion().equals(tVersion)) {
                    tStandardApiVersion = standardApiVersion;
                }
            }
        } else {
            if (CollUtil.isNotEmpty(tStandardApiVersionList)) {
                tStandardApiVersionList.sort((o1, o2) -> o2.getApprovedTime().compareTo(o1.getApprovedTime()));
                tStandardApiVersion = tStandardApiVersionList.get(0);
            }
        }
        // 有找到對應版本號數據
        if (tStandardApiVersion != null) {
            // 組成response
            StandardApi tStandardApi = tStandardApiVersion.getStandardApi();

            String apiType = tStandardApi.getApiType();
            String protocol = tStandardApi.getProtocol();
            String apiPath = tStandardApi.getApiPath();
            String requestMethod = tStandardApi.getRequestMethod();
            String tApiNameResponse = tStandardApi.getName();
            String tParentApiNameResponse = tStandardApi.getParentApiName();
            String tParentBranchResponse = tStandardApi.getParentBranch();
            String tParentApiTenantIdResponse = tStandardApi.getParentApiTenantId();
            String tApiDescription;
            String tApiRemark;
            String tApiTag;
            String tApiVersionString = tStandardApiVersion.getVersion();
            String tApplicant = tStandardApiVersion.getApplicant();
            String tApprovedStatus;
            String tEditHistory = tStandardApiVersion.getUpdateHistory() == null ? "" : tStandardApiVersion.getUpdateHistory();
            String tInvokeType = tStandardApi.getStandardApiSyncType().getNameEnUs();
            String tRequester = tStandardApi.getRequester();
            String tProvider = tStandardApi.getProvider();
            Long tPaging = tStandardApi.getStandardApiPaging().getId();
            int tStatusId = tStandardApiVersion.getApprovedStatus().getId().intValue();
            String approvedTime = DateUtil.formatLocalDateTime(tStandardApiVersion.getApprovedTime());
            int tCategoryId = tStandardApi.getStandardApiCategory().getId().intValue();
            String tCategoryName;
            String tGroupName;
            int tGroup = tStandardApi.getStandardApiGroup().getId().intValue();
            String tIsBatch = tStandardApi.getIsBatch() ? "Y" : "N";
            String tIdempotency = tStandardApiVersion.getIdempotency() ? "Y" : "N";
            String tMsgFormat = tStandardApi.getMsgFormat();
            String tTenantId = tStandardApi.getTenantId() != null || StringUtil.isEmptyOrSpace(tStandardApi.getTenantId()) ? tStandardApi.getTenantId() : "";
            switch (tLocale) {
                case ApimgmtConstant.ZH_CN:
                    tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhCn();
                    tApiDescription = tStandardApi.getDescriptionZhCn();
                    tApiRemark = tStandardApi.getRemarkZhCn();
                    tApiTag = tStandardApi.getTagZhCn() == null ? "" : tStandardApi.getTagZhCn();
                    tGroupName = tStandardApi.getStandardApiGroup().getNameZhCn();
                    tCategoryName = tStandardApi.getStandardApiCategory().getNameZhCn();
                    break;
                case ApimgmtConstant.ZH_TW:
                    tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhTw();
                    tApiDescription = tStandardApi.getDescriptionZhTw();
                    tApiRemark = tStandardApi.getRemarkZhTw();
                    tApiTag = tStandardApi.getTagZhTw() == null ? "" : tStandardApi.getTagZhTw();
                    tGroupName = tStandardApi.getStandardApiGroup().getNameZhTw();
                    tCategoryName = tStandardApi.getStandardApiCategory().getNameZhTw();
                    break;
                case ApimgmtConstant.EN_US:
                default:
                    tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameEnUs();
                    tApiDescription = tStandardApi.getDescriptionEnUs();
                    tApiRemark = tStandardApi.getRemarkEnUs();
                    tApiTag = tStandardApi.getTagEnUs() == null ? "" : tStandardApi.getTagEnUs();
                    tGroupName = tStandardApi.getStandardApiGroup().getNameEnUs();
                    tCategoryName = tStandardApi.getStandardApiCategory().getNameEnUs();
                    break;
            }
            ObjectNode tApiDescriptionMultilingualNode = mapper.createObjectNode();
            tApiDescriptionMultilingualNode.put(LocaleEnum.ZH_TW.getType(), tStandardApi.getDescriptionZhTw());
            tApiDescriptionMultilingualNode.put(LocaleEnum.ZH_CN.getType(), tStandardApi.getDescriptionZhCn());
            tApiDescriptionMultilingualNode.put(LocaleEnum.EN_US.getType(), tStandardApi.getDescriptionEnUs());
            tResponseNode.set(ApiAttributeEnum.apiDescriptionMultilingual.toString(), tApiDescriptionMultilingualNode);
            ObjectNode tApiRemarkMultilingualNode = mapper.createObjectNode();
            tApiRemarkMultilingualNode.put(LocaleEnum.ZH_TW.getType(), tStandardApi.getRemarkZhTw());
            tApiRemarkMultilingualNode.put(LocaleEnum.ZH_CN.getType(), tStandardApi.getRemarkZhCn());
            tApiRemarkMultilingualNode.put(LocaleEnum.EN_US.getType(), tStandardApi.getRemarkEnUs());
            tResponseNode.set(ApiAttributeEnum.apiRemarkMultilingual.toString(), tApiRemarkMultilingualNode);
            ObjectNode tApiTagMultilingualNode = mapper.createObjectNode();
            tApiTagMultilingualNode.put(LocaleEnum.ZH_TW.getType(), tStandardApi.getTagZhTw() == null ? "" : tStandardApi.getTagZhTw());
            tApiTagMultilingualNode.put(LocaleEnum.ZH_CN.getType(), tStandardApi.getTagZhCn() == null ? "" : tStandardApi.getTagZhCn());
            tApiTagMultilingualNode.put(LocaleEnum.EN_US.getType(), tStandardApi.getTagEnUs() == null ? "" : tStandardApi.getTagEnUs());
            tResponseNode.set(ApiAttributeEnum.apiTagMultilingual.toString(), tApiTagMultilingualNode);

            // api名稱
            tResponseNode.put(ApiAttributeEnum.apiName.toString(), tApiNameResponse);
            tResponseNode.put(ApiAttributeEnum.apiId.toString(), tStandardApi.getId());
            tResponseNode.put(ApiAttributeEnum.apiType.toString(), apiType);
            tResponseNode.put(ApiAttributeEnum.branch.toString(), branch);
            tResponseNode.put(ApiAttributeEnum.apiPath.toString(), apiPath);
            tResponseNode.put(ApiAttributeEnum.protocol.toString(), protocol);
            tResponseNode.put(ApiAttributeEnum.requestMethod.toString(), requestMethod);
            if (StrUtil.isNotBlank(tStandardApi.getUseProduct())) {
                tResponseNode.set(ApiAttributeEnum.useProduct.toString(), mapper.readTree(tStandardApi.getUseProduct()));
            }
            tResponseNode.put(ApiAttributeEnum.source.toString(), tStandardApi.getSourceProduct());

            if (tStandardApi.getTextId()!=null){
                ApiParamTextResDto apiParamText = apiParamTextService.getApiParamText(tStandardApi.getTextId());
                String json = mapper.writeValueAsString(apiParamText);
                tResponseNode.put(ApiAttributeEnum.apiParamText.toString(), mapper.readTree(json));
                tResponseNode.put(ApiAttributeEnum.textId.toString(),tStandardApi.getTextId());
            }

            List<ProjectVersionRelation> projectVersionRelationList = projectVersionRelationDao.getByApiId(tStandardApiVersion.getStandardApi().getId());
            if (CollUtil.isNotEmpty(projectVersionRelationList)) {
                ProjectVersionRelation projectVersionRelation = projectVersionRelationList.get(0);
                tResponseNode.put(ApiAttributeEnum.projectId.toString(), projectVersionRelation.getProjectId());
                tResponseNode.put(ApiAttributeEnum.projectName.toString(), projectDao.get(projectVersionRelation.getProjectId()).getProjectName());
                tResponseNode.put(ApiAttributeEnum.projectCode.toString(), tResponseNode.get(ApiAttributeEnum.projectName.toString())
                        .asText());
                tResponseNode.put(ApiAttributeEnum.hasVersions.toString(), !projectVersionDao.getByProjectId(projectVersionRelation.getProjectId()).isEmpty());

                List<Long> projectVersionIdList = projectVersionRelationList.stream().map(ProjectVersionRelation::getProjectVersionId).filter(Objects::nonNull).collect(Collectors.toList());
                if (CollUtil.isNotEmpty(projectVersionIdList)) {
                    ArrayNode projectVersionIdArrayNode = mapper.createArrayNode();
                    for (Long projectVersionId : projectVersionIdList) {
                        projectVersionIdArrayNode.add(projectVersionId);
                    }
                    tResponseNode.set(ApiAttributeEnum.projectVersionIdList.toString(), projectVersionIdArrayNode);

                    List<ProjectVersion> projectVersionList = projectVersionDao.queryProjectVersionByIdList(projectVersionIdList);
                    ArrayNode projectVersionNameArrayNode = mapper.createArrayNode();
                    for (ProjectVersion projectVersion : projectVersionList) {
                        projectVersionNameArrayNode.add(projectVersion.getProjectVersionName());
                    }
                    tResponseNode.set(ApiAttributeEnum.projectVersionNameList.toString(), projectVersionNameArrayNode);
                }
            } else {
                tResponseNode.put(ApiAttributeEnum.projectId.toString(), ApimgmtConstant.DEFAULT_PROJECT);
                tResponseNode.put(ApiAttributeEnum.projectName.toString(), ApimgmtConstant.DEFAULT_PROJECT_NAME);
            }
            // api父类名稱
            tResponseNode.put(ApiAttributeEnum.parentApiName.toString(), tParentApiNameResponse);
            tResponseNode.put(ApiAttributeEnum.parentBranch.toString(), tParentBranchResponse);
            // api父类租户Id
            tResponseNode.put(ApiAttributeEnum.parentApiTenantId.toString(), tParentApiTenantIdResponse);
            // api說明
            tResponseNode.put(ApiAttributeEnum.apiDescription.toString(), tApiDescription);
            // 產品類別
            tResponseNode.put(ApiAttributeEnum.apiTag.toString(), tApiTag);
            // 備註
            tResponseNode.put(ApiAttributeEnum.apiRemark.toString(), tApiRemark);
            // 版本號
            tResponseNode.put(ApiAttributeEnum.apiVersion.toString(), tApiVersionString);
            // 租戶ID
            tResponseNode.put(ApiAttributeEnum.tenantId.toString(), tTenantId);
            // 申請人
            tResponseNode.put(ApiAttributeEnum.applicant.toString(), tApplicant);
            // 當前狀態
            tResponseNode.put(ApiAttributeEnum.approvedStatus.toString(), tApprovedStatus);
            // 审核时间
            tResponseNode.put(ApiAttributeEnum.approvedTime.toString(), approvedTime);
            // 修改紀錄
            tResponseNode.put(ApiAttributeEnum.editHistory.toString(), tEditHistory);
            // 发起方
            tResponseNode.put(ApiAttributeEnum.requester.toString(), tRequester);
            // 接收方
            tResponseNode.put(ApiAttributeEnum.provider.toString(), tProvider);
            // 幂等性
            tResponseNode.put(ApiAttributeEnum.idempotency.toString(), tIdempotency);
            // 调用模式
            tResponseNode.put(ApiAttributeEnum.invokeType.toString(), tInvokeType);
            // 分頁模式
            tResponseNode.put(ApiAttributeEnum.paging.toString(), tPaging);
            // 審核狀態代號
            tResponseNode.put(ApiAttributeEnum.statusId.toString(), tStatusId);
            // 信息類別代號
            tResponseNode.put(ApiAttributeEnum.categoryId.toString(), tCategoryId);
            // 信息類別
            tResponseNode.put(ApiAttributeEnum.categoryName.toString(), tCategoryName);
            // 信息大類代號
            tResponseNode.put(ApiAttributeEnum.group.toString(), tGroup);
            // 信息大類
            tResponseNode.put(ApiAttributeEnum.groupName.toString(), tGroupName);
            // 須分包
            tResponseNode.put(ApiAttributeEnum.isBatch.toString(), tIsBatch);
            // 信息格式
            tResponseNode.put(ApiAttributeEnum.msgFormat.toString(), tMsgFormat);
            // by songwq 20230625 新增设计租户ID
            tResponseNode.put(ApiAttributeEnum.designTenantId.toString(), tStandardApi.getDesignTenantId());

            if (StrUtil.isNotBlank(tStandardApi.getDesignTenantId())) {
                try {
                    tResponseNode.put(ApiAttributeEnum.designTenantName.toString(), identityService.getTenantName(tStandardApi.getDesignTenantId()));
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }

            // 版本號清單
            ArrayNode tApiVersionListAryNode = mapper.createArrayNode();
            for (StandardApiVersion standardApiVersion : tStandardApiVersionList) {
                tApiVersionListAryNode.add(standardApiVersion.getVersion());
            }
            tResponseNode.set(ApiAttributeEnum.apiVersionList.toString(), tApiVersionListAryNode);
            // 信息范本下载
            ObjectNode tMessageExampleDownloadNode = mapper.createObjectNode();
            // 开发者角色（发起方）
            ObjectNode tRequesterSideNode = mapper.createObjectNode();
            // 开发者角色（接受方）
            ObjectNode tProviderSideNode = mapper.createObjectNode();
            // 发起方产品
            ArrayNode tHostProductAryNode = mapper.createArrayNode();
            // 接收方产品
            ArrayNode tServiceProductAryNode = mapper.createArrayNode();
            // 信息格式
            ArrayNode tMsgFormatAryNode = mapper.createArrayNode();
            tResponseNode.set(ApiAttributeEnum.messageExampleDownload.toString(), tMessageExampleDownloadNode);
            tMessageExampleDownloadNode.set(ApiAttributeEnum.requesterSide.toString(), tRequesterSideNode);
            tMessageExampleDownloadNode.set(ApiAttributeEnum.providerSide.toString(), tProviderSideNode);
            tRequesterSideNode.set(ApiAttributeEnum.hostProduct.toString(), tHostProductAryNode);
            tRequesterSideNode.set(ApiAttributeEnum.serviceProduct.toString(), tServiceProductAryNode);
            tRequesterSideNode.set(ApiAttributeEnum.msgFormat.toString(), tMsgFormatAryNode);
            tProviderSideNode.set(ApiAttributeEnum.hostProduct.toString(), tHostProductAryNode);
            tProviderSideNode.set(ApiAttributeEnum.serviceProduct.toString(), tServiceProductAryNode);
            tProviderSideNode.set(ApiAttributeEnum.msgFormat.toString(), tMsgFormatAryNode);
            String[] tRequesterProducts = tStandardApi.getRequester().split(",");
            String[] tProviderProducts = tStandardApi.getProvider().split(",");
            for (String tRequester1 : tRequesterProducts) {
                tHostProductAryNode.add(tRequester1);
            }
            for (String tProvider1 : tProviderProducts) {
                tServiceProductAryNode.add(tProvider1);
            }
//            MessageFormatEnum[] tMessageFormatEnums = MessageFormatEnum.values();
            if (StrUtil.isNotBlank(tMsgFormat)) {
                Arrays.stream(tMsgFormat.split(",")).forEach(s->tMsgFormatAryNode.add(s));
            }
//            for (MessageFormatEnum tMessageFormatEnum : tMessageFormatEnums) {
//                if (!(tMessageFormatEnum.equals(MessageFormatEnum.JSON10) || (tInvokeType.equals("fasync") && tMessageFormatEnum.equals(MessageFormatEnum.XML)))) {
//                    tMsgFormatAryNode.add(tMessageFormatEnum.toString());
//                }
//            }
            // 取得詞彙
            List<StandardApiDataName> tStandardApiDataNameList = standardApiDataNameDao.getByApiVerId(tStandardApiVersion.getId(), true);
            tStandardApiDataNameList = tStandardApiDataNameList.stream().distinct().collect(Collectors.toList());
            // Request信息内容
            ApiDataNameServiceUtil.setReaponseMessageSpec(tResponseNode, tStandardApiDataNameList, ApiAttributeEnum.requestMessageSpec.toString(), 1, tLocale);
            // Response信息内容（成功）
            ApiDataNameServiceUtil.setReaponseMessageSpec(tResponseNode, tStandardApiDataNameList, ApiAttributeEnum.responseMessageSuccessSpec.toString(), 2, tLocale);
            // Response信息内容（失敗）
            ApiDataNameServiceUtil.setReaponseMessageSpec(tResponseNode, tStandardApiDataNameList, ApiAttributeEnum.responseMessageFailedSpec.toString(), 3, tLocale);
        } else {
            // 沒找到對應版本號數據
            tStateCode = getStateCode(ValidateStateEnum.API_NAME_VER_NOTFOUND.getCode());
        }
        String tDescription = tStateCode.getDescription();
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
        return converJsonNodeToMap(tResponseJsonNode);
    }
}
