package com.digiwin.athena.apimgmt.apiservice;

import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.common.model.dto.FileDto;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStateCodeDao;
import com.digiwin.athena.apimgmt.enums.APIExportImportEnums;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.service.util.DmcFileServiceUtil;
import com.digiwin.athena.apimgmt.service.util.StandardApiFileReadServiceUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * API规格导入前检查
 * /restful/standard/apimgmt/ApiImport/Check
 */
@Slf4j
@Service
public class ApiImportCheckService {
	ObjectMapper mapper = new ObjectMapper();

	@Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;
	@Autowired
    private ApiMgmtStateCodeDao stateCodeDao;

	public ApiImportCheckService() {
		// Constructor
	}

	/**
	 * API导入前置检查
	 * @param file 文件
	 * @param request 请求体
	 * @return Map
	 * @throws BaseException BaseException
	 */
	public Map<String, Object> execute(FileDto file) throws BaseException {
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口："
				+ this.getClass());
		// 建立response node
		ObjectNode tResponseNode = mapper.createObjectNode();
		StateCode tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = tStateCode.getDescription();
		StringBuilder tDescriptionBuilder = new StringBuilder();
		try {
			// 取得token
            String tUserId = ApiMgmtServiceContextHolder.getUserId();
			String tUserName = ApiMgmtServiceContextHolder.getUserName();
			InputStream fileIn = file.getInputStream();
			// 取得副檔名
			String fileType = FilenameUtils.getExtension(file.getFileName());
			// 轉成Workbook
			Workbook tWorkbook = StandardApiFileReadServiceUtil.getWorkBook(fileType, fileIn);
			ObjectNode tCheckResult = mapper.createObjectNode();
			//草稿
			int tDraftDuplicateCount = 0;
			//审核中
			int tUnderReviewDuplicateCount = 0;
			//开发中
			int tDevelopingDuplicateCount = 0;
			//已定版
			int tConfirmedDuplicateCount = 0;
			String tDuplicateType = null;
			String tOtherUser = null;
			Map<String, Integer> tOtherUserDuplicateRecord = new HashMap<>();
			boolean tApiAlreadyExist = false;
			int tAlreadyExistCount = 0;
			log.info("讀入的Excel有" + tWorkbook.getNumberOfSheets() + "個Sheets.");
			// sheet0-4為樣板，從第5個起為讀取模版
			for (int tSheetNumber = ApimgmtConstant.NEW_API_SHEET_NUMBER; tSheetNumber < tWorkbook.getNumberOfSheets(); tSheetNumber++) {
				// 隱藏的sheet不讀取
				if (tWorkbook.isSheetHidden(tSheetNumber)) {
					continue;
				}
				StandardApi tStandardApi = new StandardApi();
				Sheet tSheet = tWorkbook.getSheetAt(tSheetNumber);
				// 讀取StandardAPI資訊
				Map<String, String> tSetStdApiResult = StandardApiFileReadServiceUtil.setStandardApiBasicInfo(tStandardApi, tUserName, tSheet);
				if (!StringUtil.isEmptyOrSpace(tSetStdApiResult.get(APIExportImportEnums.RESULT_MESSAGE.toString()))) {
                      //存在异常 -应该抛出
					tStateCode = getStateCode(ValidateStateEnum.IMPORT_SPEC_FORMAT_ERROR.getCode());
					tSetStdApiResult.put("code", tStateCode.getCode());
					tSetStdApiResult.put("description", "API基础信息检测异常: " + tSetStdApiResult.get(APIExportImportEnums.RESULT_MESSAGE.toString()));
					return mapper.convertValue(tSetStdApiResult, new TypeReference<Map<String, Object>>() {
					});
				}
				String tApiName = tStandardApi.getName();
				String tTenantId = tStandardApi.getTenantId();
				String tVersion = tStandardApi.getStandardApiVersions().get(0).getVersion();
				List<StandardApiVersion> tExistStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, tTenantId, tVersion);
				// API已存在
				if (!tExistStandardApiVersions.isEmpty()) {
					// 同一個user id下重複
					if (Objects.equals(tUserId, tExistStandardApiVersions.get(0).getUserId())) {
						switch (String.valueOf(tExistStandardApiVersions.get(0).getApprovedStatus().getId())) {
						// 已存在于草稿中
						case "1":
							tDraftDuplicateCount++;
							tDuplicateType = "1";
							break;
						// 正在审核中
						case "2":
							tUnderReviewDuplicateCount++;
							tDuplicateType = "2";
							break;
						// 正在开发中
						case "3":
							tDevelopingDuplicateCount++;
							tDuplicateType = "3";
							break;
						// 已定版
						case "4":
							tConfirmedDuplicateCount++;
							tDuplicateType = "4";
							break;
						default:
							break;
						}
					} else {
						tDuplicateType = "5";
						tOtherUser = tExistStandardApiVersions.get(0).getUserId();
						if (tOtherUserDuplicateRecord.get(tExistStandardApiVersions.get(0).getUserId()) == null) {
							tOtherUserDuplicateRecord.put(tExistStandardApiVersions.get(0).getUserId(), 1);
						} else {
							int i = tOtherUserDuplicateRecord.get(tExistStandardApiVersions.get(0).getUserId());
							tOtherUserDuplicateRecord.put(tExistStandardApiVersions.get(0).getUserId(), i + 1);
						}
					}
					tApiAlreadyExist = true;
					tAlreadyExistCount++;
					StandardApiFileReadServiceUtil.addDuplicateDescription(tWorkbook, tSheet, tDuplicateType, tOtherUser);
				}
			}
			// API已存在，則要提供詳細的下載連結
			if (tApiAlreadyExist) {
                String tDetailFilePath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + file.getFileName();
				String tDetailFileName = file.getFileName();
				FileOutputStream fileOut = new FileOutputStream(tDetailFilePath);
				tWorkbook.write(fileOut);
				fileOut.close();
				DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tDetailFilePath, tDetailFileName, true, 0);
				tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.IMPORT_SPEC_DUPLICATE_API.getCode());
				tDescription = tStateCode.getDescription();
			}
			if (tDraftDuplicateCount != 0) {
				tCheckResult.put(ApiAttributeEnum.DraftDuplicate.toString(), tDraftDuplicateCount);
			}
			if (tUnderReviewDuplicateCount != 0) {
				tCheckResult.put(ApiAttributeEnum.UnderReviewDuplicate.toString(), tUnderReviewDuplicateCount);
			}
			if (tDevelopingDuplicateCount != 0) {
				tCheckResult.put(ApiAttributeEnum.DevelopingDuplicate.toString(), tDevelopingDuplicateCount);
			}
			if (tConfirmedDuplicateCount != 0) {
				tCheckResult.put(ApiAttributeEnum.ConfirmedDuplicate.toString(), tConfirmedDuplicateCount);
			}
			if (!tOtherUserDuplicateRecord.isEmpty()) {
				ArrayNode tOtherUserDuplicateAryNode = mapper.createArrayNode();
				for (String tUser : tOtherUserDuplicateRecord.keySet()) {
					ObjectNode tUserDuplicateNode = mapper.createObjectNode();
					tUserDuplicateNode.put(ApiAttributeEnum.userId.toString(), tUser);
					tUserDuplicateNode.put(ApiAttributeEnum.duplicateNumber.toString(), tOtherUserDuplicateRecord.get(tUser));
					tOtherUserDuplicateAryNode.add(tUserDuplicateNode);
				}
				tCheckResult.set(ApiAttributeEnum.OtherUserDuplicate.toString(), tOtherUserDuplicateAryNode);
			}
			tResponseNode.set(ApiAttributeEnum.checkResult.toString(), tCheckResult);
			tResponseNode.put(ApiAttributeEnum.wholeError.toString(), tAlreadyExistCount == tWorkbook.getNumberOfSheets() - ApimgmtConstant.NEW_API_SHEET_NUMBER);
		} catch (BaseException e) {
			log.error(e.getMessage(), e);
			tStateCode = getStateCode(e.getStateEnum().getCode());
			tDescriptionBuilder.append(tStateCode.getDescription());
			if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
				tDescriptionBuilder.append(ApimgmtConstant.BLANK).append(e.getMessage());
			}
			tDescription = tDescriptionBuilder.toString();
		} catch (Exception e) {
			log.error(e.getMessage(), e);
			tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
			tDescriptionBuilder.append(tStateCode.getDescription()).append(ApimgmtConstant.BLANK).append(e.getClass().toString()).append(":").append(e.getMessage());
			tDescription = tDescriptionBuilder.toString();
		}
		tResponseNode.put("code", tStateCode.getCode());
		tResponseNode.put("description", tDescription);
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口：" + this.getClass() + "，回傳訊息：" + tResponseNode.toString());
		return mapper.convertValue(tResponseNode, new TypeReference<Map<String, Object>>() {
		});
	}

	protected StateCode getStateCode(String pStateCode) {
		return stateCodeDao.getStateCodeByCode(pStateCode);
	}
}
