package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.apimgmt.annotate.OperateAuthorityVerify;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtApprovedStatusDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtSyncMdcFailDataDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtMdcProp;
import com.digiwin.athena.apimgmt.model.ApprovedStatus;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.services.ApiMgmtSyncMdcServiceProcessor;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * API規格取回
 * /restful/standard/apimgmt/ApiSpecRetrieve/Update
 */
@Slf4j
@Service
@OperateAuthorityVerify
public class ApiSpecRetrieveUpdateService extends AbstractApiService {

    @Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

    @Autowired
    ApiMgmtApprovedStatusDao approvedStatusDao;

    @Autowired
    private ApiMgmtSyncMdcFailDataDao syncMdcFailDataDao;

    @Autowired
    private ApiMgmtSyncMdcServiceProcessor syncMdcServiceProcessor;

    @Autowired
    private ApiMgmtMdcProp mdcProp;

    public ApiSpecRetrieveUpdateService() {
        super();
        jsonSchemaFileName = ApimgmtSchemaConstant.API_REVIEW_SCHEMA;
    }

    /**
     * API 取回
     *
     * @param validatorResult validatorResult
     * @return Map
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        String tToken = ApiMgmtServiceContextHolder.getToken();
        String routerKey = ApiMgmtServiceContextHolder.getRouterKey();
        String tUserId = ApiMgmtServiceContextHolder.getUserId();
        ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.API_LIST);
        List<Long> versionList = new ArrayList<>();
        for (JsonNode tApiNode : tApiArrayNode) {
            Map<String, String> tConditionMap = new HashMap<>();
            tConditionMap.put(ApiAttributeEnum.apiName.toString(), tApiNode.get(ApiAttributeEnum.apiName.toString()).asText());
            tConditionMap.put(ApiAttributeEnum.version.toString(), tApiNode.get(ApiAttributeEnum.version.toString()).asText());
            if (tApiNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
                tConditionMap.put(ApiAttributeEnum.tenantId.toString(), tApiNode.get(ApiAttributeEnum.tenantId.toString()).asText());
            }
            StandardApiVersion tStandardApiVersion = standardApiVersionDao.fetchApiVersion(tConditionMap);
            if (tStandardApiVersion != null) {
                // 状态非开发中跳过
                if (tStandardApiVersion.getApprovedStatus().getId() != ApimgmtConstant.APPROVED_STATUS_DEVELOPING) {
                    continue;
                }

                ApprovedStatus tApprovedStatus = approvedStatusDao.get(1L);
                tStandardApiVersion.setApprovedStatus(tApprovedStatus);
                tStandardApiVersion.setApprovedTime(LocalDateTime.now());
                standardApiVersionDao.saveOrupdate(tStandardApiVersion);
                // 删除定时任务相关数据
                syncMdcFailDataDao.removeByApiVerId(tStandardApiVersion.getId());
                versionList.add(tStandardApiVersion.getId());
            }
        }
        if (CollUtil.isNotEmpty(versionList) && CollUtil.isNotEmpty(mdcProp.getReviewUrlList())) {
            for (String url : mdcProp.getReviewUrlList()) {
                syncMdcServiceProcessor.runSyncMdcMetadataDel(versionList, url, true, tUserId, tToken, routerKey);
            }
        }
        Map<String, Object> tResponse = new HashMap<>();
        // 組成回傳訊息
        tResponse.put(ApimgmtConstant.CODE, "000");
        tResponse.put(ApimgmtConstant.RESULT, ApimgmtConstant.SUCCESS);
        return tResponse;
    }
}
