package com.digiwin.athena.apimgmt.apiservice;

import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardDataNameDao;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.StandardDataName;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * 詞彙快速查找
 */
@Slf4j
@Service
public class DataNameSearchService implements InitializingBean {


	@Autowired
	private ApiMgmtStandardDataNameDao standardDataNameDao;

	private static TreeSet<AbstractSearchModel> dataNameTreeSet;

	private static TreeSet<AbstractSearchModel> descTWTreeSet;

	private static TreeSet<AbstractSearchModel> descCNTreeSet;

	/**
	 * /restful/standard/apimgmt/ApiDateNameSearch/Get
	 * @param dataName dataName
	 * @return Map
	 * @throws Exception Exception
	 */
	public Map<String, Object> execute(String dataName) throws Exception {
		log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口：" + this.getClass() + "，原始訊息： {\"dataName\":\"" + dataName + "\"}");
		// 取得語系
		String tLocale = ApiMgmtServiceContextHolder.getLocale();
		// header沒傳語系的話，默認回傳英文
		tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
		Map<String, Object> tResponse = new LinkedHashMap<>();
		List<DataNameModel> tDataNameModelList = new ArrayList<>();
		int count = 0;
		TreeSet<AbstractSearchModel> searchSet = dataNameTreeSet;
		if (!dataName.matches(ApimgmtConstant.REGEX_DATA_NAME)) {
			switch (tLocale) {
				case ApimgmtConstant.ZH_TW:
				searchSet = descTWTreeSet;
				break;
				case ApimgmtConstant.ZH_CN:
				searchSet = descCNTreeSet;
				break;
			}
		}
		Set<AbstractSearchModel> sub = searchSet.subSet(new CompareModel(dataName), new CompareModel(dataName + Character.MAX_VALUE));
		for (AbstractSearchModel abstractSearchModel : sub) {
			tDataNameModelList.add(abstractSearchModel.getDataNameModel());
			if (++count > ApimgmtConstant.MAX_COUNT) {
				break;
			}
		}
		if (count < ApimgmtConstant.MAX_COUNT) {
			for (AbstractSearchModel temp : searchSet) {
				if (temp.getEqualValue().contains(dataName) && !tDataNameModelList.contains(temp.getDataNameModel())) {
					tDataNameModelList.add(temp.getDataNameModel());
					if (++count > ApimgmtConstant.MAX_COUNT) {
						break;
					}
				}
			}
		}
		for (DataNameModel tDataNameModel : tDataNameModelList) {
			Map<String, Object> tDataNameMap = new HashMap<>();
			Map<String, Object> tMultilingualMap = new HashMap<>();
			tMultilingualMap.put(ApimgmtConstant.ZH_TW, tDataNameModel.getDescTW());
			tMultilingualMap.put(ApimgmtConstant.ZH_CN, tDataNameModel.getDescCN());
			tMultilingualMap.put(ApimgmtConstant.EN_US, tDataNameModel.getDescEn());
			tDataNameMap.put(ApimgmtConstant.DATA_DESC_MULT, tMultilingualMap);
			tDataNameMap.put(ApimgmtConstant.DATA_NAME, tDataNameModel.getDataName());
			String tDesc = tDataNameModel.getDescCN();
			switch (tLocale) {
			case ApimgmtConstant.ZH_TW:
				tDesc = tDataNameModel.getDescTW();
				break;
			case ApimgmtConstant.ZH_CN:
				tDesc = tDataNameModel.getDescEn();
				break;
			}
			tResponse.put(tDataNameModel.getDataName() + "(" + tDesc + ")", tDataNameMap);
		}
		return tResponse;
	}

	public void reloadDataNameToTreeSet() {
		TreeSet<AbstractSearchModel> dataNameTreeSetTemp = new TreeSet<>();
		TreeSet<AbstractSearchModel> descTWTreeSetTemp = new TreeSet<>();
		TreeSet<AbstractSearchModel> descCNTreeSetTemp = new TreeSet<>();
		List<StandardDataName> standardDataNameList = standardDataNameDao.getDataNameByApprovedStatus(4L, false);
		for (StandardDataName standardDataName : standardDataNameList) {
			DataNameModel model = new DataNameModel(standardDataName.getDataName(), standardDataName.getDescriptionZhTw(), standardDataName.getDescriptionZhCn(), standardDataName.getDescriptionEnUs());
			dataNameTreeSetTemp.add(new DataNameSearchModel(model));
			descTWTreeSetTemp.add(new DescTWSearchModel(model));
			descCNTreeSetTemp.add(new DescCNSearchModel(model));
		}
		dataNameTreeSet = dataNameTreeSetTemp;
		descTWTreeSet = descTWTreeSetTemp;
		descCNTreeSet = descCNTreeSetTemp;
	}

    @Override
    public void afterPropertiesSet() {
        reloadDataNameToTreeSet();
    }
}

class DataNameModel {
	/**
	 * 词典名称
	 */
	private String dataName;

	/**
	 * 描述繁中
	 */
	private String descTW;

	/**
	 * 描述简中
	 */
	private String descCN;

	/**
	 * 描述英文
	 */
	private String descEn;

	public DataNameModel(String dataName, String descTW, String descCN, String descEn) {
		this.dataName = dataName;
		this.descTW = descTW;
		this.descCN = descCN;
		this.descEn = descEn;
	}

	public String getDataName() {
		return dataName;
	}

	public String getDescTW() {
		return descTW;
	}

	public String getDescCN() {
		return descCN;
	}

	public String getDescEn() {
		return descEn;
	}

	@Override
	public boolean equals(Object obj) {
	    if (null == obj) {
	        return  false;
        }
        if (obj.getClass() != this.getClass()) {
            return false;
		}
        if (obj == this) {
            return true;
        }
		DataNameModel model = (DataNameModel) obj;
		return getDataName().equals(model.getDataName());
	}

	@Override
	public int hashCode() {
		return Objects.hash(dataName);
	}
}

abstract class AbstractSearchModel implements Comparable<AbstractSearchModel> {

	private DataNameModel dataNameModel;

	public AbstractSearchModel(DataNameModel dataNameModel) {
		this.dataNameModel = dataNameModel;
	}

	public DataNameModel getDataNameModel() {
		return dataNameModel;
	}

	public String getDataName() {
		return dataNameModel.getDataName();
	}

	public String getDescTW() {
		return dataNameModel.getDescTW();
	}

	public String getDescCN() {
		return dataNameModel.getDescCN();
	}

	public String getDescEn() {
		return dataNameModel.getDescEn();
	}

	@Override
	public boolean equals(Object obj) {
        if (null == obj) {
            return  false;
        }
        if (obj.getClass() != this.getClass()) {
            return false;
        }
        if (obj == this) {
            return true;
        }
		AbstractSearchModel model = (AbstractSearchModel) obj;
		return getEqualValue().equals(model.getEqualValue());
	}

    @Override
    public int hashCode() {
        return Objects.hash(getEqualValue());
    }

	@Override
	public int compareTo(AbstractSearchModel arg0) {
		return this.getCompareValue().compareTo(arg0.getCompareValue());
	}

	abstract String getEqualValue();

	abstract String getCompareValue();

}

class DataNameSearchModel extends AbstractSearchModel {
	public DataNameSearchModel(DataNameModel dataNameModel) {
		super(dataNameModel);
	}

	@Override
	String getEqualValue() {
		return getDataName();
	}

	@Override
	String getCompareValue() {
		return getDataName();
	}

}

class DescTWSearchModel extends AbstractSearchModel {
	public DescTWSearchModel(DataNameModel dataNameModel) {
		super(dataNameModel);
	}

	@Override
	String getEqualValue() {
		return getDescTW();
	}

	@Override
	String getCompareValue() {
		return getDescTW() + getDataName();
	}

}

class DescCNSearchModel extends AbstractSearchModel {

	public DescCNSearchModel(DataNameModel dataNameModel) {
		super(dataNameModel);
	}

	@Override
	String getEqualValue() {
		return getDescCN();
	}

	@Override
	String getCompareValue() {
		return getDescCN() + getDataName();
	}

}

class CompareModel extends AbstractSearchModel {

	String compareValue;

	public CompareModel(String compareValue) {
		super(null);
		this.compareValue = compareValue;
	}

	@Override
	String getEqualValue() {
		return compareValue;
	}

	@Override
	String getCompareValue() {
		return compareValue;
	}

}
