package com.digiwin.athena.apimgmt.infra.dap.http;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.stream.StreamUtil;
import com.digiwin.athena.apimgmt.common.util.JsonUtil;
import com.digiwin.athena.apimgmt.infra.http.*;
import com.digiwin.http.client.DWHttpClient;
import com.digiwin.http.client.DWRequestOption;
import com.digiwin.http.client.exception.DWHttpFailedException;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.util.EntityUtils;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.stream.Collectors;

/**
 * DAP HTTP客户端实现
 * 实现IHttpClient接口，封装DAP框架的HTTP客户端功能
 */
@Component
public class DapApiMgmtHttpClientWrapper implements ApiMgmtHttpClientWrapper {

    private final DWHttpClient dwHttpClient;

    public DapApiMgmtHttpClientWrapper(DWHttpClient dwHttpClient) {
        this.dwHttpClient = dwHttpClient;
    }

    @Override
    public <T> T execute(HttpRequest request, Class<T> responseType) {
        return execute(request, responseType, null);
    }


    @Override
    public <T> T execute(HttpRequest request, Class<T> responseType, RequestOption options) {
        HttpUriRequest realRequest = buildRequest(request, options);

        return exchange(responseType, options, realRequest);
    }

    private <T> T exchange(Class<T> responseType, RequestOption options, HttpUriRequest realRequest) {
        DWRequestOption dwOption = new DWRequestOption(false);

        BasicHttpContext httpContext = new BasicHttpContext();
        if (options != null) {
            if (!options.isRetryEnabled()) {
                httpContext.setAttribute("Time-Task", "Time-Task");
            }
        }

        try {
            if (responseType == null || HttpResponse.class.isAssignableFrom(responseType)) {
                org.apache.http.HttpResponse resp = dwHttpClient.execute(realRequest, null, dwOption, httpContext);
                //noinspection unchecked
                return (T) transformResponse(resp);
            }

            return dwHttpClient.execute(realRequest, responseType, null, dwOption, httpContext);
        } catch (DWHttpFailedException e) {
            throw new DapHttpFailedException(e);
        } catch (IOException e) {
            throw new DapHttpFailedException("http请求异常！", e);
        }
    }


    @Override
    public <T> T uploadFile(HttpRequest request, MultipartFile file, Class<T> responseType) {
        return uploadFile(request, file, responseType, null);
    }

    @Override
    public <T> T uploadFile(HttpRequest request, MultipartFile file, Class<T> responseType, RequestOption options) {
        HttpUriRequest realRequest = buildMultipartRequest(request, file, options);

        return exchange(responseType, options, realRequest);
    }

    protected HttpUriRequest buildRequest(HttpRequest request, RequestOption options) {
        RequestConfig.Builder config = setOptions(options);

        Object body = request.getBody();
        HttpEntity entity;
        if (body instanceof HttpEntity) {
            entity = (HttpEntity) body;
        } else {
            StringEntity strEntity = new StringEntity(JsonUtil.serialize(body), ContentType.APPLICATION_JSON);
            strEntity.setContentEncoding("UTF-8");
            strEntity.setContentType("application/json");

            entity = strEntity;
        }

        return RequestBuilder.create(request.getMethod())
                .setConfig(config.build())
                .setEntity(entity)
                .setUri(request.getUrl())
                .build();
    }

    protected HttpUriRequest buildMultipartRequest(HttpRequest request, MultipartFile file, RequestOption options) {
        RequestConfig.Builder config = setOptions(options);

        MultipartEntityBuilder entityBuilder = MultipartEntityBuilder.create();

        getBodyMap(request).forEach((k, v) ->
                entityBuilder.addTextBody(k, JsonUtil.serialize(v), ContentType.APPLICATION_JSON));

        entityBuilder.addBinaryBody(file.getName(), file.getInputStream(),
                ContentType.create(file.getContentType()),
                file.getFileName());

        RequestBuilder builder = RequestBuilder.create(request.getMethod())
                .setConfig(config.build())
                .setEntity(entityBuilder.build())
                .setUri(request.getUrl());

        MapUtil.emptyIfNull(request.getHeaders()).forEach(builder::addHeader);

        return builder.build();
    }

    protected RequestConfig.Builder setOptions(RequestOption options) {
        RequestConfig.Builder config = RequestConfig.custom();
        if (options != null) {
            if (options.getTimeout() != null) {
                config.setSocketTimeout(options.getTimeout());
            }
        }
        return config;
    }

    protected HttpResponse transformResponse(org.apache.http.HttpResponse resp) throws IOException {
        return HttpResponse.builder()
                .statusCode(resp.getStatusLine().getStatusCode())
                .header(StreamUtil.of(resp.getAllHeaders())
                        .collect(Collectors.toMap(NameValuePair::getName, NameValuePair::getValue)))
                .is(resp.getEntity().getContent())
                .bodyConverter(() -> {
                    try {
                        return EntityUtils.toString(resp.getEntity());
                    } catch (IOException e) {
                        throw new DapHttpFailedException("解析响应体失败！", e);
                    }
                })
                .build();
    }
}