package com.digiwin.athena.atmc.application.service.query.backlog.impl;

import com.digiwin.athena.atmc.application.service.restful.ptm.PtmQService;
import com.digiwin.athena.atmc.application.service.query.backlog.PtmTransformQueryCountingQryService;
import com.digiwin.athena.atmc.application.configuration.EnvProperties;
import com.digiwin.athena.atmc.application.utils.SummaryMessagesUtils;
import com.digiwin.athena.atmc.core.meta.constants.BpmConstant;
//import com.digiwin.athena.atmc.core.meta.utils.SummaryMessagesUtils;
import com.digiwin.athena.atmc.core.meta.constants.GlobalConstant;
import com.digiwin.athena.atmc.core.meta.enums.ActivityStateEnum;
import com.digiwin.athena.atmc.core.meta.enums.BpmEngineCategoryEnum;
import com.digiwin.athena.atmc.core.meta.enums.ptm.PtmBacklogTypeEnum;
import com.digiwin.athena.atmc.infrastructure.mapper.biz.migration.PtmBacklogMapper;
import com.digiwin.athena.atmc.infrastructure.pojo.bo.migration.*;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmBacklog;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Author hebaokai
 * Date 2024/8/15 16:30
 **/
@Slf4j
@Service
public class PtmTransformQueryCountingQryServiceImpl implements PtmTransformQueryCountingQryService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    PtmBacklogMapper ptmBacklogMapper;

    @Autowired
    PtmQService ptmService;

    /**
     * 是否启用了PTM
     */
    @Override
    public boolean enablePTM() {
        return StringUtils.isNotBlank(envProperties.getPtmUri());
    }

    /**
     * 查询当前租户指定用户的待办列表
     */
    @Override
    public List<BacklogBO> queryBacklogList(String tenantId, String userId) {
        List<BacklogBO> result = new ArrayList<>();
        if (!enablePTM()) {
            return result;
        }

        return queryBacklogList(tenantId, Arrays.asList(userId));
    }

    /**
     * 查询当前租户指定用户的待办列表
     *
     * @return 如果不存在，返回空列表
     */
    @Override
    public List<BacklogBO> queryBacklogList(String tenantId, List<String> userIds) {
        List<BacklogBO> result = new ArrayList<>();
        // 是否配置了ptm url
        if (!enablePTM()) {
            return result;
        }
        // 获取任务列表
        List<PtmTaskCardItemBO> ptmTaskCardItemBOList = ptmService.queryTaskCardList(userIds);
        // 普通任务卡
        result.addAll(dealBacklogList(ptmTaskCardItemBOList));
        // 逾时任务卡
        List<PtmBacklogBO> overdueList = ptmBacklogMapper.selectNoFinishedOverdue(tenantId, userIds);
        result.addAll(dealOverdueBacklog(overdueList));
        // 数据一致性异常排除任务卡
        List<PtmBacklogBO> exceptionList = ptmBacklogMapper.selectNoFinishedException(tenantId, userIds);
        result.addAll(dealExceptionData(exceptionList));
        if (CollectionUtils.isNotEmpty(result)) {
            result.stream().forEach(backLog -> backLog.setDataFrom(BpmConstant.DATA_FROM_PTM));
        }

        return result;
    }

    /**
     * 获取业务数据不一致异常排除任务卡
     * PS：queryBacklogList(String tenantId, List<String> userIds)方法整块代码平挪
     */
    public List<BacklogBO> dealExceptionData(List<PtmBacklogBO> exceptionList) {
        List<BacklogBO> result = new ArrayList<>();
        //数据一致性异常排除任务卡
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(exceptionList)) {
            List<PtmProjectRecordBO> ptmProjectRecordList = ptmService.getProjectRecordByProjectIdsV2(exceptionList.stream().map(PtmBacklog::getProjectId).distinct().collect(Collectors.toList()), false);
            List<PtmTaskRecordBO> ptmTaskRecordList = ptmService.getTaskRecordByTaskIds(exceptionList.stream().map(PtmBacklog::getTaskId).distinct().collect(Collectors.toList()));
            exceptionList.forEach(exception -> {
                PtmTaskRecordBO ptmTaskRecordBO = ptmTaskRecordList.stream().filter(x -> Objects.equals(x.getId(), exception.getTaskId())).findFirst().get();
                PtmProjectRecordBO ptmProjectRecordBO = ptmProjectRecordList.stream().filter(x -> Objects.equals(x.getId(), exception.getProjectId())).findFirst().get();
                BacklogBO bl = BacklogBO.builder().id(exception.getBacklogId()).submitId(exception.getSubmitId()).name(GlobalConstant.DATA_UNIFORMITY_BACKLOG_NAME).subName(ptmTaskRecordBO.getTaskName()).taskName(ptmProjectRecordBO.getProjectName()).taskTargetName(null)       //不需要
                        .taskStartTime(ptmProjectRecordBO.getStartTime()).taskEndTime(ptmProjectRecordBO.getEndTime()).taskImportance(ptmProjectRecordBO.getImportance()).importance(exception.getFavorite() > 0)      //不用处理
                        .content(ptmTaskRecordBO.getTaskName()).importanceReadOnly(false)  //不用处理
                        .importanceSource(null).uri(null).startTime(ptmTaskRecordBO.getCreateTime()).endTime(LocalDateTime.now().plusMonths(1))     //永不逾期
                        .tmTaskId(ptmProjectRecordBO.getProjectDefCode()).tmActivityId(ptmTaskRecordBO.getTaskDefCode()).type(SummaryMessagesUtils.translateBacklogType(exception.getType())).tenantId(ptmTaskRecordBO.getTenantId()).actionDefined(true).bpmActivityId(ptmTaskRecordBO.getId()).tmPattern(ptmTaskRecordBO.getTaskDefPattern()).tmCategory(ptmTaskRecordBO.getTaskDefCategory()).checkItems(null).operation(null).summaryLayout(null).summaryLayoutStr(null).error(null).isOwner(true).performerId(exception.getPerformerId()).performerName(exception.getPerformerName()).ownerUserId(exception.getPerformerId()).ownerUserName(exception.getPerformerName()).readCount(exception.getReadCount()).exception(ptmTaskRecordBO.getHasException()).overdue(false).overdueDays(0).overdueHours(0).overdueMinutes(0).reassignFromId(0L).approvalState(exception.getApprovalState()).todoItems(null).emergency(ptmProjectRecordBO.getEmergency() == 25).eocCode(ptmProjectRecordBO.getEocCode()).eocName(ptmProjectRecordBO.getEocName()).eocType(ptmProjectRecordBO.getEocType()).processSerialNumber(ptmTaskRecordBO.getProcessSerialNumber()).agentBeginDate(null).agentBeginDate(null).overdueWorkitemId(exception.getOverdueBacklogId()).sourceTenantName(StringUtils.equalsIgnoreCase(exception.getTenantId(), exception.getTargetTenantId()) ? null : exception.getTenantName()).workItemId(exception.getBacklogId()).proxyToken(ptmTaskRecordBO.getProxyToken()).historyMessage(null).finishActionId(exception.getClosed() ? exception.getActionId() : null).tmActivityName(ptmTaskRecordBO.getTaskDefName()).dataFrom(BpmConstant.DATA_FROM_PTM).summary(exception.getSummary()).historyMessage(exception.getHistoryMessage()).searchMessage(exception.getSearchMessage()).engineType(BpmEngineCategoryEnum.TaskEngine.getValue()).closedTime(ptmTaskRecordBO.getClosedTime()).createTime(ptmTaskRecordBO.getCreateTime()).targetTenantId(exception.getTargetTenantId()).hasAppPermission(checkByType(exception.getType())).build();

                result.add(bl);
            });
        }
        return result;
    }

    /**
     * 获取逾时任务卡
     * PS：queryBacklogList(String tenantId, List<String> userIds)方法整块代码平挪
     */
    public List<BacklogBO> dealOverdueBacklog(List<PtmBacklogBO> overdueList) {
        List<BacklogBO> result = new ArrayList<>();
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(overdueList)) {

            List<PtmProjectRecordBO> ptmProjectRecordList = ptmService.getProjectRecordByProjectIdsV2(overdueList.stream().map(PtmBacklog::getProjectId).distinct().collect(Collectors.toList()), false);
            List<PtmTaskRecordBO> ptmTaskRecordList = ptmService.getTaskRecordByTaskIds(overdueList.stream().map(PtmBacklog::getTaskId).distinct().collect(Collectors.toList()));

            overdueList.forEach(overdue -> {
                PtmTaskRecordBO ptmTaskRecordBO = ptmTaskRecordList.stream().filter(x -> Objects.equals(x.getId(), overdue.getTaskId())).findFirst().get();
                PtmProjectRecordBO ptmProjectRecordBO = ptmProjectRecordList.stream().filter(x -> Objects.equals(x.getId(), overdue.getProjectId())).findFirst().get();
                BacklogBO bl = BacklogBO.builder().id(overdue.getBacklogId()).submitId(overdue.getSubmitId()).name(GlobalConstant.OVERDUE_BACKLOG_NAME).subName(ptmTaskRecordBO.getTaskName()).taskName(ptmProjectRecordBO.getProjectName()).taskTargetName(null)       //不需要
                        .taskStartTime(ptmProjectRecordBO.getStartTime()).taskEndTime(ptmProjectRecordBO.getEndTime()).taskImportance(ptmProjectRecordBO.getImportance()).importance(overdue.getFavorite() > 0)      //不用处理
                        .content(ptmTaskRecordBO.getTaskName()).importanceReadOnly(false)  //不用处理
                        .importanceSource(null).uri(null).startTime(ptmTaskRecordBO.getCreateTime()).endTime(LocalDateTime.now().plusMonths(1))     //永不逾期
                        .tmTaskId(ptmProjectRecordBO.getProjectDefCode()).tmActivityId(ptmTaskRecordBO.getTaskDefCode()).type(SummaryMessagesUtils.translateBacklogType(overdue.getType())).tenantId(ptmTaskRecordBO.getTenantId()).actionDefined(true).bpmActivityId(ptmTaskRecordBO.getId()).tmPattern(ptmTaskRecordBO.getTaskDefPattern()).tmCategory(ptmTaskRecordBO.getTaskDefCategory()).checkItems(null).operation(null).summaryLayout(null).summaryLayoutStr(null).error(null).isOwner(true).performerId(overdue.getPerformerId()).performerName(overdue.getPerformerName()).ownerUserId(overdue.getPerformerId()).ownerUserName(overdue.getPerformerName()).readCount(overdue.getReadCount()).exception(ptmTaskRecordBO.getHasException()).overdue(false).overdueDays(0).overdueHours(0).overdueMinutes(0).reassignFromId(0L).approvalState(overdue.getApprovalState()).todoItems(null).emergency(ptmProjectRecordBO.getEmergency() == 25).eocCode(ptmProjectRecordBO.getEocCode()).eocName(ptmProjectRecordBO.getEocName()).eocType(ptmProjectRecordBO.getEocType()).processSerialNumber(ptmTaskRecordBO.getProcessSerialNumber()).agentBeginDate(null).agentBeginDate(null).overdueWorkitemId(overdue.getOverdueBacklogId()).sourceTenantName(StringUtils.equalsIgnoreCase(overdue.getTenantId(), overdue.getTargetTenantId()) ? null : overdue.getTenantName()).workItemId(overdue.getBacklogId()).proxyToken(ptmTaskRecordBO.getProxyToken()).historyMessage(null).finishActionId(overdue.getClosed() ? overdue.getActionId() : null).tmActivityName(ptmTaskRecordBO.getTaskDefName()).dataFrom(BpmConstant.DATA_FROM_PTM).summary(overdue.getSummary()).historyMessage(overdue.getHistoryMessage()).searchMessage(overdue.getSearchMessage()).dataChangeRead(overdue.getDataChangeRead()).engineType(BpmEngineCategoryEnum.TaskEngine.getValue()).closedTime(ptmTaskRecordBO.getClosedTime()).createTime(ptmTaskRecordBO.getCreateTime()).targetTenantId(overdue.getTargetTenantId()).hasAppPermission(checkByType(overdue.getType())).build();

                result.add(bl);
            });
        }
        return result;
    }

    /**
     * 整理普通任务的列表信息
     * PS：queryBacklogList(String tenantId, List<String> userIds)方法整块代码平挪
     */
    public List<BacklogBO> dealBacklogList(List<PtmTaskCardItemBO> ptmTaskCardItemBOList) {
        List<BacklogBO> result = new ArrayList<>();
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(ptmTaskCardItemBOList)) {
            List<Long> backlogIdList = ptmTaskCardItemBOList.stream().map(PtmTaskCardItemBO::getBacklogId).distinct().collect(Collectors.toList());
            boolean backlogIdListIsEmpty = org.apache.commons.collections4.CollectionUtils.isEmpty(backlogIdList);
            if (!backlogIdListIsEmpty) {
                Collections.sort(backlogIdList);
            }
            // 查询ATMC代办信息
            List<PtmBacklogBO> ptmBacklogList = ptmBacklogMapper.selectByBacklogIds(backlogIdList, backlogIdListIsEmpty ? null : backlogIdList.get(0), backlogIdListIsEmpty ? null : backlogIdList.get(backlogIdList.size() - 1));
            // 查询PTM项目卡记录信息
            List<PtmProjectRecordBO> ptmProjectRecordList = ptmService.getProjectRecordByProjectIdsV2(ptmTaskCardItemBOList.stream().map(PtmTaskCardItemBO::getProjectId).distinct().collect(Collectors.toList()), false);
            // 查询PTM任务卡记录信息
            List<PtmTaskRecordBO> ptmTaskRecordList = ptmService.getTaskRecordByTaskIds(ptmTaskCardItemBOList.stream().map(PtmTaskCardItemBO::getTaskId).distinct().collect(Collectors.toList()));
            // 获取合并任务数据
            Map<Long, PtmMergeBacklogBO> mergeBacklogMap = getMergeBacklogData(ptmTaskCardItemBOList);

            for (PtmTaskCardItemBO ptmTaskCardItemBO : ptmTaskCardItemBOList) {
                // 任务卡对应的PTM项目卡记录
                Optional<PtmProjectRecordBO> ptmProjectRecordOption = ptmProjectRecordList.stream().filter(x -> Objects.equals(x.getId(), ptmTaskCardItemBO.getProjectId())).findFirst();
                PtmProjectRecordBO ptmProjectRecordBO = ptmProjectRecordOption.orElseGet(ptmProjectRecordOption::get);
                // 任务卡对应的PTM任务卡记录
                Optional<PtmTaskRecordBO> ptmTaskRecordOption = ptmTaskRecordList.stream().filter(x -> Objects.equals(x.getId(), ptmTaskCardItemBO.getTaskId())).findFirst();
                PtmTaskRecordBO ptmTaskRecord = ptmTaskRecordOption.orElseGet(ptmTaskRecordOption::get);
                // 过滤没有入ATMC库的任务卡
                PtmBacklogBO ptmBacklog = new PtmBacklogBO();
                if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(ptmBacklogList)) {
                    Optional<PtmBacklogBO> ptmBacklogOptional = ptmBacklogList.stream().filter(x -> Objects.equals(x.getBacklogId(), ptmTaskCardItemBO.getBacklogId())).findFirst();

                    if (ptmBacklogOptional.isPresent()) {
                        // ATMC 侧记录任务卡
                        ptmBacklog = ptmBacklogOptional.get();
                    }
                }
                // 合并任务卡信息
                PtmMergeBacklogBO ptmMergeBacklogBO = mergeBacklogMap.get(ptmTaskCardItemBO.getBacklogId());
                // 组装任务卡数据
                BacklogBO bl = BacklogBO.builder()
                        .id(ptmTaskCardItemBO.getBacklogId())
                        .submitId(ptmBacklog.getSubmitId())
                        .name(ptmTaskCardItemBO.getTaskName())
                        .subName(null)
                        .taskName(ptmProjectRecordBO.getProjectName())
                        .taskTargetName(null)       //不需要
                        .taskStartTime(ptmProjectRecordBO.getStartTime())
                        .taskEndTime(ptmProjectRecordBO.getEndTime())
                        .taskImportance(ptmProjectRecordBO.getImportance())
                        .importance(ptmBacklog.getFavorite() > 0)      //不用处理
                        .content(ptmTaskCardItemBO.getTaskName())
                        .importanceReadOnly(false)  //不用处理
                        .importanceSource(null)
                        .uri(null)
                        .startTime(ptmTaskCardItemBO.getCreateTime())
                        .endTime(ptmTaskCardItemBO.getPlanEndTime())
                        .tmTaskId(ptmProjectRecordBO.getProjectDefCode())
                        .tmActivityId(ptmTaskCardItemBO.getTaskDefCode())
                        .type(SummaryMessagesUtils.translateBacklogType(ptmBacklog.getType()))
                        .tenantId(StringUtils.isBlank(ptmTaskCardItemBO.getFromTenantId()) ? ptmTaskCardItemBO.getTenantId() : ptmTaskCardItemBO.getFromTenantId())
                        .targetTenantId(StringUtils.isBlank(ptmTaskCardItemBO.getTenantId()) ? ptmTaskCardItemBO.getFromTenantId() : ptmTaskCardItemBO.getTenantId())
                        .actionDefined(true)
                        .bpmActivityId(ptmTaskCardItemBO.getTaskId())
                        .tmPattern(ptmTaskCardItemBO.getTaskDefPattern())
                        .tmCategory(ptmTaskCardItemBO.getTaskDefCategory())
                        .checkItems(null)
                        .operation(null)
                        .summaryLayout(null)
                        .summaryLayoutStr(null)
                        .error(null)
                        .isOwner(true)
                        .performerId(ptmTaskCardItemBO.getPerformerId())
                        .performerName(ptmTaskCardItemBO.getPerformerName())
                        .ownerUserId(ptmTaskCardItemBO.getPerformerId())
                        .ownerUserName(ptmTaskCardItemBO.getPerformerName())
                        .readCount(ptmBacklog.getReadCount())
                        .exception(ptmTaskRecord.getHasException())
                        .overdue(false)
                        .overdueDays(0)
                        .overdueHours(0)
                        .overdueMinutes(0)
                        .reassignFromId(0L)
                        .approvalState(ptmBacklog.getApprovalState())
                        .todoItems(null)
                        .emergency(ptmProjectRecordBO.getEmergency() == 25)
                        .eocCode(ptmProjectRecordBO.getEocCode())
                        .eocName(ptmProjectRecordBO.getEocName())
                        .eocType(ptmProjectRecordBO.getEocType())
                        .processSerialNumber(ptmTaskRecord.getProcessSerialNumber())
                        .agentBeginDate(null)
                        .agentBeginDate(null)
                        .overdueWorkitemId(0L)
                        .sourceTenantName(StringUtils.equalsIgnoreCase(ptmBacklog.getTenantId(), ptmBacklog.getTargetTenantId()) ? null : ptmBacklog.getTenantName())
                        .workItemId(ptmTaskCardItemBO.getBacklogId())
                        .proxyToken(ptmTaskCardItemBO.getProxyToken())
                        .historyMessage(null)
                        .finishActionId(Boolean.TRUE.equals(ptmBacklog.getClosed()) ? ptmBacklog.getActionId() : null)
                        .tmActivityName(ptmTaskCardItemBO.getTaskDefName())
                        .dataFrom(BpmConstant.DATA_FROM_PTM).summary(ptmBacklog.getSummary())
                        .historyMessage(ptmBacklog.getHistoryMessage())
                        .searchMessage(ptmBacklog.getSearchMessage())
                        .merge(ptmBacklog.getMerge())
                        .dataChangeRead(ptmBacklog.getDataChangeRead())
                        .engineType(BpmEngineCategoryEnum.TaskEngine.getValue())
                        .closedTime(ptmTaskCardItemBO.getClosedTime())
                        .createTime(ptmTaskCardItemBO.getCreateTime())
                        .hasAppPermission(ptmBacklog.getType() != null ? checkByType(ptmBacklog.getType()) : null)
                        .auxiliaryTenantId(ptmTaskCardItemBO.getAuxiliaryTenantId())
                        .auxiliaryTenantName(ptmTaskCardItemBO.getAuxiliaryTenantName())
                        .engineHasWaittingData(ptmTaskCardItemBO.getEngineHasWaittingData()).build();

                // 设置逾期数据
                // 合并任务卡的逾期判断
                if (BooleanUtils.isTrue(ptmBacklog.getMerge())) {
                    if (ptmMergeBacklogBO != null) {
                        LocalDateTime planEndTimeMin = ptmMergeBacklogBO.getPlanEndTimeMin();

                        bl.setPlanEndTimeMin(planEndTimeMin);
                        bl.setPlanEndTimeMax(ptmMergeBacklogBO.getPlanEndTimeMax());
                        bl.setEndTime(planEndTimeMin);
                        // 未关闭的合并任务卡，比较最小预计完成时间和当前时间
                        if (Boolean.TRUE.equals(!ptmMergeBacklogBO.getClosed()) && planEndTimeMin.isBefore(LocalDateTime.now())) {
                            bl.setOverdue(true);
                            bl.setOverdueDays((int) Duration.between(planEndTimeMin, LocalDateTime.now()).toDays());
                            bl.setOverdueHours((int) Duration.between(planEndTimeMin, LocalDateTime.now()).toHours());
                            bl.setOverdueMinutes((int) Duration.between(planEndTimeMin, LocalDateTime.now()).toMinutes());
                        }
                        // 已关闭的合并任务卡，比较最小预计完成时间和关闭时间
                        else if (ptmTaskCardItemBO.getClosedTime() != null && ptmMergeBacklogBO.getClosed() && planEndTimeMin.isBefore(ptmTaskCardItemBO.getClosedTime())) {
                            bl.setOverdue(true);
                            bl.setOverdueDays((int) Duration.between(planEndTimeMin, ptmTaskCardItemBO.getClosedTime()).toDays());
                            bl.setOverdueHours((int) Duration.between(planEndTimeMin, ptmTaskCardItemBO.getClosedTime()).toHours());
                            bl.setOverdueMinutes((int) Duration.between(planEndTimeMin, ptmTaskCardItemBO.getClosedTime()).toMinutes());
                        }
                    }
                }
                // 非合并任务卡的逾期判断
                else {
                    if (Boolean.TRUE.equals(!ptmTaskCardItemBO.getClosed()) && ptmTaskRecord.getPlanEndTime().isBefore(LocalDateTime.now())) {
                        bl.setOverdue(true);
                        bl.setOverdueDays((int) Duration.between(ptmTaskRecord.getPlanEndTime(), LocalDateTime.now()).toDays());
                        bl.setOverdueHours((int) Duration.between(ptmTaskRecord.getPlanEndTime(), LocalDateTime.now()).toHours());
                        bl.setOverdueMinutes((int) Duration.between(ptmTaskRecord.getPlanEndTime(), LocalDateTime.now()).toMinutes());
                    } else if (ptmTaskCardItemBO.getClosedTime() != null && ptmTaskCardItemBO.getClosed() && ptmTaskRecord.getPlanEndTime().isBefore(ptmTaskCardItemBO.getClosedTime())) {
                        bl.setOverdue(true);
                        bl.setOverdueDays((int) Duration.between(ptmTaskRecord.getPlanEndTime(), ptmTaskCardItemBO.getClosedTime()).toDays());
                        bl.setOverdueHours((int) Duration.between(ptmTaskRecord.getPlanEndTime(), ptmTaskCardItemBO.getClosedTime()).toHours());
                        bl.setOverdueMinutes((int) Duration.between(ptmTaskRecord.getPlanEndTime(), ptmTaskCardItemBO.getClosedTime()).toMinutes());
                    }
                }

                //如果任务的逾时间就比当前时间小不足1分钟，会在画面显示逾时0分钟。
                if (Boolean.TRUE.equals(bl.getOverdue()) && bl.getOverdueMinutes() <= 0) {
                    bl.setOverdue(false);
                }

                result.add(bl);
            }
        }
        return result;
    }


    /**
     * 获取待处理和已完成的合并任务数据
     */
    private Map<Long, PtmMergeBacklogBO> getMergeBacklogData(List<PtmTaskCardItemBO> ptmTaskCardItemBOList) {
        Map<Long, PtmMergeBacklogBO> mergeBacklogMap = new HashMap<>();
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(ptmTaskCardItemBOList)) {
            return mergeBacklogMap;
        }
        //  筛选出合并任务
        List<PtmTaskCardItemBO> mergeCardList = ptmTaskCardItemBOList.stream().filter(x -> Objects.equals(x.getMerge(), true)).collect(Collectors.toList());
        // 没有合并任务不做任何处理
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(mergeCardList)) {
            return mergeBacklogMap;
        }
        // 获取未结束任务
        List<Long> pendingBacklogIds = mergeCardList.stream().filter(x -> !x.getClosed()).map(PtmTaskCardItemBO::getBacklogId).distinct().collect(Collectors.toList());
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(pendingBacklogIds)) {
            mergeBacklogMap.putAll(dealBacklogData(pendingBacklogIds, false));
        }
        // 获取已结束任务
        List<Long> closedBacklogIds = mergeCardList.stream().filter(PtmTaskCardItemBO::getClosed).map(PtmTaskCardItemBO::getBacklogId).distinct().collect(Collectors.toList());
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(closedBacklogIds)) {
            mergeBacklogMap.putAll(dealBacklogData(closedBacklogIds, true));
        }
        return mergeBacklogMap;
    }


    /**
     * 整理合并任务数据
     */
    public Map<Long, PtmMergeBacklogBO> dealBacklogData(List<Long> backlogIdList, Boolean history) {
        if (history == null) {
            history = false;
        }
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(backlogIdList)) {
            return new HashMap<>();
        }
        //获取任务详情
        List<PtmBacklogItemBO> ptmBacklogItemBOList = ptmService.getBacklogByBacklogIds(backlogIdList);
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(ptmBacklogItemBOList)) {
            return Collections.emptyMap();
        }
        //按backlogId分组
        Map<Long, List<PtmBacklogItemBO>> ptmBacklogItemMap = ptmBacklogItemBOList.stream().collect(Collectors.groupingBy(PtmBacklogItemBO::getBacklogId));
        //整理任务合并数据
        Map<Long, PtmMergeBacklogBO> resultMap = new HashMap<>();
        for (Map.Entry<Long, List<PtmBacklogItemBO>> entry : ptmBacklogItemMap.entrySet()) {
            Long backlogId = entry.getKey();
            List<PtmBacklogItemBO> itemDTOList = entry.getValue();
            //如果是已完成数据，则按照预计完成日升序排序；若是未完成数据，去除已完成的任务并且按照预计完成日升序排序
            List<LocalDateTime> planEndTimeList;
            if (history) {
                planEndTimeList = itemDTOList.stream().sorted(Comparator.comparing(PtmBacklogItemBO::getPlanEndTime))
                        .map(PtmBacklogItemBO::getPlanEndTime)
                        .collect(Collectors.toList());
            } else {
                planEndTimeList = itemDTOList.stream().filter(item -> !isWorkItemClosed(item.getState()))
                        .sorted(Comparator.comparing(PtmBacklogItemBO::getPlanEndTime))
                        .map(PtmBacklogItemBO::getPlanEndTime)
                        .collect(Collectors.toList());
            }

            //子任务是否存在未完成的
            List<PtmBacklogItemBO> pendingList = itemDTOList.stream().filter(item -> !isWorkItemClosed(item.getState()))
                    .collect(Collectors.toList());

            PtmMergeBacklogBO ptmMergeBacklogBO = new PtmMergeBacklogBO();
            ptmMergeBacklogBO.setBacklogId(backlogId);
            ptmMergeBacklogBO.setTaskName(itemDTOList.get(0).getTaskName());
            if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(planEndTimeList)) {
                ptmMergeBacklogBO.setPlanEndTimeMin(planEndTimeList.get(0));
                ptmMergeBacklogBO.setPlanEndTimeMax(planEndTimeList.get(planEndTimeList.size() - 1));
            }
            ptmMergeBacklogBO.setClosed(org.apache.commons.collections4.CollectionUtils.isEmpty(pendingList));
            resultMap.put(backlogId, ptmMergeBacklogBO);
        }
        return resultMap;
    }


    private boolean isWorkItemClosed(Integer state) {
        return Objects.nonNull(state) && state >= ActivityStateEnum.COMPLETED.getCode();
    }

    private Boolean checkByType(Integer type) {
        // 跨租户不控制权限,默认有权限
        return PtmBacklogTypeEnum.SPAN_TENANT.getValue().equals(type);
    }

    @Override
    public List<BacklogBO> queryPinningBacklogList(String tenantId, List<Long> backlogIdList) {
        List<BacklogBO> result = new ArrayList<>();
        // 是否配置了ptm url
        if (!enablePTM()) {
            return result;
        }
        // 获取任务列表
        List<PtmTaskCardItemBO> ptmTaskCardItemBOList = ptmService.queryTaskCardListByBacklogId(backlogIdList);
        // 普通任务卡
        result.addAll(dealBacklogList(ptmTaskCardItemBOList));
        // 逾时任务卡
        List<PtmBacklogBO> overdueList = ptmBacklogMapper.selectNoFinishedPiningOverdue(tenantId, backlogIdList);
        result.addAll(dealOverdueBacklog(overdueList));
        // 数据一致性异常排除任务卡
        List<PtmBacklogBO> exceptionList = ptmBacklogMapper.selectNoFinishedPinningException(tenantId, backlogIdList);
        result.addAll(dealExceptionData(exceptionList));
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(result)) {
            result.forEach(backLog -> backLog.setDataFrom(BpmConstant.DATA_FROM_PTM));
        }

        return result;
    }
}
