package com.digiwin.athena.atmc.http.restful.aglie.impl;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.HttpUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.constant.GlobalConstant;
import com.digiwin.athena.atmc.http.constant.UibotApiConstant;
import com.digiwin.athena.atmc.http.domain.ExecuteContext;
import com.digiwin.athena.atmc.http.domain.action.SubmitAction;
import com.digiwin.athena.atmc.http.domain.report.ExcelParamsDTO;
import com.digiwin.athena.atmc.http.domain.task.ActivityDataDTO;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.aglie.AglieService;
import com.digiwin.athena.atmc.http.restful.aglie.model.AglieDynamicFormDTO;
import com.digiwin.athena.atmc.http.restful.aglie.model.QueryActionAnalysisReq;
import com.digiwin.athena.atmc.http.util.AtmcStrUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import net.sf.json.JSONObject;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class AglieServiceImpl implements AglieService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    private MessageUtils messageUtils;

    /**
     * 获取项目卡CheckItems等数
     *
     * @param tmTaskId
     * @param tmActivityId
     * @param projectId
     * @return
     */
    public BaseResultDTO<Map<String, Object>> getProjectCardData(String tmTaskId, String tmActivityId, Long projectId, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.PROJECT_CARD_TM_TASK_ID_TM_ACTIVITY_ID_PROJECT_ID, tmTaskId, tmActivityId, projectId);
        return getCardData(url, activityDataDTO);
    }

    /**
     * 获取项目卡CheckItems等数,根据据项目自身的数据，不依赖任务
     *
     * @param tmTaskId
     * @param projectId
     * @return
     */
    public BaseResultDTO<Map<String, Object>> getProjectCardDataByProjectData(String tmTaskId, Long projectId, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.PROJECT_CARD_TM_TASK_ID_TM_TASK_ID, tmTaskId, projectId);
        return getCardData(url, activityDataDTO);
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public BaseResultDTO<HashMap> getTaskCardDataSize(String tmTaskId, String tmActivityId, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_TM_TASK_ID_TM_ACTIVITY_ID_DATA_SIZE, tmTaskId, tmActivityId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<HashMap>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<HashMap>>() {
                }, param);

        return responseEntity.getBody();
    }

    /**
     * 获取项目的数据大小
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    public BaseResultDTO<HashMap> getProjectCardDataSize(String tmTaskId, String tmActivityId, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.PROJECT_TM_TASK_ID_TM_ACTIVITY_ID_DATA_SIZE, tmTaskId, tmActivityId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<HashMap>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<HashMap>>() {
                }, param);

        return responseEntity.getBody();
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param tmTaskId
     * @param tmActivityId
     * @param backlogId
     * @return
     */
    public BaseResultDTO<AglieDynamicFormDTO> getTaskCardLayout(String tmTaskId, String tmActivityId, Long backlogId, Object activityDataDTO, String flag) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_CARD_TM_TASK_ID_TM_ACTIVITY_ID_BACKLOG_ID, tmTaskId, tmActivityId, backlogId);

        if (StringUtils.isNotEmpty(flag)) {
            url = url + "?flag=" + flag;
        }
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<AglieDynamicFormDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<AglieDynamicFormDTO>>() {
                }, param);

        return responseEntity.getBody();
    }

    /**
     * 获取任务卡 拼接字符等数
     *
     * @param tmTaskId
     * @param tmActivityId
     * @param backlogId
     * @return
     */
    public BaseResultDTO<Map> getTaskCardAbstractStr(String tmTaskId, String tmActivityId, Long backlogId, Object activityDataDTO) {
        return getTaskCardLayoutAbstract(tmTaskId, tmActivityId, backlogId, activityDataDTO, null);
    }

    /**
     * 项目卡摘要信息，不依赖任务（字符串）
     *
     * @Author：SYQ
     * @Date：2021/12/30 10:35
     */
    @Override
    public BaseResultDTO<Map<String, Object>> getProjectCardDataByProjectDataAbstract(String tmTaskId, Long projectId, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.PROJECT_CARD_ABSTRACT_TM_TASK_ID_PROJECT_ID, tmTaskId, projectId);

        return getCardData(url, activityDataDTO);
    }

    /**
     * 项目卡-任务摘要信息（字符串）
     *
     * @Author：SYQ
     * @Date：2021/12/30 10:36
     */
    @Override
    public BaseResultDTO<Map<String, Object>> getProjectCardDataAbstract(String tmTaskId, String tmActivityId, Long projectId,
                                                                         Object activityDataDTO) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.PROJECT_CARD_ABSTRACT_TM_TASK_ID_TM_ACTIVITY_ID_PROJECT_ID, tmTaskId, tmActivityId, projectId);

        return getCardData(url, activityDataDTO);
    }

    @Override
    public BaseResultDTO<Map> getTaskCardLayoutAbstract(String tmTaskId, String tmActivityId, Long backlogId,
                                                        Object activityDataDTO, String flag) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_CARD_ABSTRACT_TM_TASK_ID_TM_ACTIVITY_ID_BACKLOG_ID, tmTaskId, tmActivityId, backlogId);

        if (StringUtils.isNotEmpty(flag)) {
            url = url + "?flag=" + flag;
        }
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<Map>>() {
                }, param);
        return responseEntity.getBody();
    }

    @Override
    public BaseResultDTO<Map> getTaskCardLayoutAbstractUpToDate(String tmTaskId, String tmActivityId, Long backlogId,
                                                        Object activityDataDTO, String flag) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_CARD_ABSTRACT_PAGE_DATA, tmTaskId, tmActivityId, backlogId);

        if (StringUtils.isNotEmpty(flag)) {
            url = url + "?flag=" + flag;
        }
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<Map>>() {
                }, param);
        return responseEntity.getBody();
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param activityDataDTO
     * @return
     */
    @Override
    public BaseResultDTO<JSONObject> getData(ActivityDataDTO activityDataDTO) {
        return queryData(null, activityDataDTO);
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param activityDataDTO
     * @return
     */
    @Override
    public BaseResultDTO<JSONObject> queryData(String token, Object activityDataDTO) {
        String url = envProperties.getUibotUri() + UibotApiConstant.DATA_QUERY_BY_ACTIVITY;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        if (StringUtils.isNotEmpty(token)) {
            headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        }
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<Object>(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<JSONObject>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<JSONObject>>() {
                });
        // 直接返回了BaseResultDTO，因此这里不需要检查respEntity.isOK()
        return responseEntity.getBody();
    }

    /**
     * 查询报表信息
     *
     * @param excelParamsDTO
     * @return
     */
    @Override
    public List<Map<String, Object>> queryExcelInfo(ExcelParamsDTO excelParamsDTO) {
        String url = envProperties.getUibotUri() + UibotApiConstant.REPORT_QUERY_EXCEL_INFO;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity reqEntity = new HttpEntity<>(excelParamsDTO, headers);
        ResponseEntity<Object> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, reqEntity,
                new ParameterizedTypeReference<Object>() {
                });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<List<Map<String, Object>>>() {
        });
    }

    private BaseResultDTO<Map<String, Object>> getCardData(String uri, Object activityDataDTO) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        HttpEntity httpEntity = new HttpEntity(activityDataDTO, headers);
        ResponseEntity<BaseResultDTO<Map<String, Object>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map<String, Object>>>() {
        }, param);
        responseEntity.getBody().getResponseWithException("");
        return responseEntity.getBody();
    }

    private void addLang(HttpHeaders headers) {
        headers.add("locale", LocaleContextHolder.getLocale().toString());
    }

    @Override
    public BaseResultDTO<JSONObject> analysisQueryAction(ExecuteContext executeContext, Map tmAction) {
        String url = envProperties.getUibotUri() + UibotApiConstant.DATA_QUERY_ACTION;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        QueryActionAnalysisReq body = new QueryActionAnalysisReq(executeContext, tmAction);
        HttpEntity httpEntity = new HttpEntity<Object>(body, headers);
        ResponseEntity<BaseResultDTO<JSONObject>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<JSONObject>>() {});
        responseEntity.getBody().getResponseWithException("");
        return responseEntity.getBody();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map getReportTemplatePage(String reportCode) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TEMPLATE_STATEMENT_CONDITION_SHOW_REPORT_CODE, reportCode);

        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<String> reqEntity = new HttpEntity<>(null, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map getTaskPageData(String tmTaskId, String tmActivityId, Long workItemId) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_PAGE_DATA_TM_TASK_ID_TM_ACTIVITY_ID_WORK_ITEM_ID, tmTaskId, tmActivityId, workItemId);

        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<String> reqEntity = new HttpEntity<>(null, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }

    /**
     * 获取任务详情页面的submitActions
     *
     * @param tmTaskId
     * @param tmActivityId
     * @return
     */
    @Override
    public List<Map> getTaskPageSubmitActions(String tmTaskId, String tmActivityId, Map<String, Object> activityData) {
        String url = envProperties.getUibotUri() + AtmcStrUtil.format(UibotApiConstant.TASK_SUBMITACTIONS_TM_TASK_ID_TM_ACTIVITY_ID, tmTaskId, tmActivityId);

        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<Map<String, Object>> reqEntity = new HttpEntity<>(activityData, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<List<Map>>() {
        });
    }

    //region 转派 action

    /**
     * 获取 项目 转派的执行action
     *
     * @param queryAction
     * @return
     */
    @Override
    public List<SubmitAction> generateProjectReassignAction(Map queryAction) {
        String url = envProperties.getUibotUri() + UibotApiConstant.ACTION_PROJECT_REASSIGN;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(queryAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        }, param);
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<List<SubmitAction>>() {
        });
    }

    /**
     * 获取 任务 转派的执行action
     *
     * @param queryAction
     * @return
     */
    @Override
    public List<SubmitAction> generateTaskReassignAction(Map queryAction) {
        String url = envProperties.getUibotUri() + UibotApiConstant.ACTION_TASK_REASSIGN;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity httpEntity = new HttpEntity(queryAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        }, param);
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<List<SubmitAction>>() {
        });
    }

    //endregion

    /**
     * 清除缓存
     */
    @Override
    public void clearCache() {
        String url = envProperties.getUibotUri() + UibotApiConstant.CACHE_RESET;
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity<BaseResultDTO<Object>> reqEntity = new HttpEntity<>(null, headers);
        ResponseEntity<BaseResultDTO<Object>> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Object>>() {
        });
        // 请求失败，直接抛出异常
        if (HttpStatus.SC_OK != respEntity.getStatusCodeValue() || !respEntity.getBody().isOK()) {
            throw ErrorCodeEnum.ATMC_REMOVE_CACHE_FAIL.getBusinessException(JsonUtils.objectToString(respEntity.getBody()));
        }
    }

}
