package com.digiwin.athena.atmc.http.restful.aim.impl;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.constant.AimApiConstant;
import com.digiwin.athena.atmc.http.constant.GlobalConstant;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.aim.AimService;
import com.digiwin.athena.atmc.http.restful.emc.model.AppMessageDTO;
import com.digiwin.athena.atmc.http.util.AtmcStrUtil;
import com.digiwin.athena.atmc.http.domain.message.AimEventDO;
import com.digiwin.athena.atmc.http.domain.message.MessageBatchUserDTO;
import com.digiwin.athena.atmc.http.domain.message.MessageDO;
import com.digiwin.athena.atmc.http.restful.aim.model.OnlineClient;
import com.digiwin.athena.atmc.http.restful.aim.model.OnlineClientQueryResult;
import com.github.pagehelper.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.HttpStatus;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@Service
public class AimServiceImpl implements AimService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private MessageUtils messageUtils;

    public void message(AuthoredUser authoredUser, MessageDO message) {
        String url = envProperties.getAimUri() + AimApiConstant.MESSAGE;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());

        HttpEntity httpEntity = new HttpEntity(message, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO>() {
                    });
            if (respEntity.getStatusCode().value() != HttpStatus.SC_OK || !respEntity.getBody().isOK()) {
                log.error("请求aim存储新消息失败，statusCode: {}, responseBody: {}.", respEntity.getStatusCode().value(), respEntity.getBody());
            }
        } catch (Exception ex) {
            log.error("请求aim存储新消息失败，error: {}.", ex);
        }
    }

    @Override
    public void saveNewMessage(AuthoredUser authoredUser, List<MessageBatchUserDTO> messageList) {
        if (CollectionUtils.isEmpty(messageList)) {
            return;
        }

        messageList.forEach(message -> {
            if (null == message.getMessage().getState()) {
                message.getMessage().setState(0);
            }
            // 默认发送通知给移动端APP
            if (null == message.getMessage().getNoticeMobileApp()) {
                message.getMessage().setNoticeMobileApp(true);
            }
        });

        String url = envProperties.getAimUri() + AimApiConstant.BATCH_USER_MESSAGE;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());

        HttpEntity httpEntity = new HttpEntity(messageList, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO>() {
                    });
            if (respEntity.getStatusCode().value() != HttpStatus.SC_OK || !respEntity.getBody().isOK()) {
                log.error("请求aim批量保存新消息失败，statusCode: {}, responseBody: {}.", respEntity.getStatusCode().value(), respEntity.getBody());
            }
        } catch (Exception ex) {
            log.error("请求aim批量保存新消息失败，error: {}.", ex);
        }
    }

    @Override
    public Map<String, Integer> deleteByTenant(AuthoredUser authoredUser, String tenantId) {
        //结果集
        Map<String, Integer> resultMap = new HashMap<>();
        //租户id不能为空
        if (StringUtil.isEmpty(tenantId)) {
            return resultMap;
        }

        String uri = envProperties.getAimUri() + AtmcStrUtil.format(AimApiConstant.MISC_MESSAGE_DELETE_BY_TENANT, tenantId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());

        HttpEntity httpEntity = new HttpEntity(headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO>() {
                    });
            resultMap.put("code", 200);
            if (respEntity.getStatusCode().value() != HttpStatus.SC_OK) {
                resultMap.put("code", 500);
                log.error("根据租户删除消息提醒失败：{}", respEntity);
            }
        } catch (Exception e) {
            resultMap.put("code", 500);
            log.error("根据租户删除消息提醒失败：{}", e.getMessage());
        }

        return resultMap;
    }

    /**
     * 发送催办事件通知
     *
     * @param user
     * @param event
     */
    @Override
    public void sendUrageDealNotice(AuthoredUser user, AimEventDO event) {
        log.info("sendUrageDealNotice-user:{},event:{}", JsonUtils.objectToString(user), JsonUtils.objectToString(event));
        String url = envProperties.getAimUri() + AimApiConstant.EVENT_MQ_SEND_URGE_DEAL_EVENT;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());

        HttpEntity httpEntity = new HttpEntity(event, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            if (respEntity.getStatusCode().value() != HttpStatus.SC_OK || !respEntity.getBody().isOK()) {
                log.error("Failed to request aim to send reminder event ，statusCode: {}, responseBody: {}.", respEntity.getStatusCode().value(), respEntity.getBody());
            }
        } catch (Exception ex) {
            log.error("Failed to request aim to send reminder event，error: {}.", ex);
        }
    }

    /**
     * 根据制定用户来获取
     *
     * @param tenantId 租户id
     * @param userId   用户id
     * @return 在线用户
     */
    public List<OnlineClient> queryByUserId(String tenantId, String userId) {
        String url = envProperties.getAimUri() +
                AtmcStrUtil.format(AimApiConstant.CLIENTS_TENANTID_USERID, tenantId, userId);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        ResponseEntity<OnlineClientQueryResult> respEntity = restTemplate.getForEntity(url, OnlineClientQueryResult.class);
        return respEntity.getBody().getResponseWithException("");
    }

    /**
     * @param message 内容
     */
    @Override
    public void sendMessageToClient(String token, String tenantId, List<String> userIdList, MessageDO message) {
        MessageBatchUserDTO messageBatchUserDTO = new MessageBatchUserDTO();
        messageBatchUserDTO.setMessage(message);
        messageBatchUserDTO.setUserIdList(userIdList);
        messageBatchUserDTO.setTenantId(tenantId);

        String url = envProperties.getAimUri() + AimApiConstant.SEND_MESSAGE_TO_CLIENT;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);

        HttpEntity httpEntity = new HttpEntity(messageBatchUserDTO, headers);
        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    public void sendMsgRemindToClient(String token, String tenantId, String userId, AppMessageDTO appMessageDTO) {
        // 初始化消息创建时间
        initMsgCreateTime(appMessageDTO);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        HttpEntity<?> httpEntity = new HttpEntity<>(appMessageDTO, headers);
        String url = envProperties.getAimUri() + AimApiConstant.SEND_MESSAGE_REMIND_TO_CLIENT;
        log.info("【请求aim发送新消息入参开始】：url：{}，tenantId：{}，userId：{}，appMessageDTO：{}", url, tenantId, userId, appMessageDTO);
        ResponseEntity<BaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
        });
        log.info("【请求aim发送新消息入参结果：{}】", responseEntity);
        responseEntity.getBody().getResponseWithException("");
    }

    @Override
    public void batchSendMessageRemindToClient(String token, String tenantId, List<AppMessageDTO> appMessageList) {
        appMessageList.forEach(this::initMsgCreateTime);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        HttpEntity<?> httpEntity = new HttpEntity<>(appMessageList, headers);
        String url = envProperties.getAimUri() + AimApiConstant.BATCH_SEND_MESSAGE_REMIND_TO_CLIENT;
        log.info("【请求aim发送新消息入参开始】：url：{}，tenantId：{}，appMessageDTO：{}", url, tenantId, appMessageList);
        ResponseEntity<BaseResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
        });
        log.info("【请求aim发送新消息入参结果：{}】", responseEntity);
        responseEntity.getBody().getResponseWithException("");
    }

    /**
     * 初始化消息创建时间
     *
     * @param appMessageDTO
     * @author majianfu
     * @since [迭代13 消息通知弹窗增加已通知时间]
     */
    private void initMsgCreateTime(AppMessageDTO appMessageDTO) {
        if (null == appMessageDTO) {
            return;
        }

        Object msgContent = appMessageDTO.getContent();
        if (null == msgContent) {
            appMessageDTO.setContent(new HashMap<String, Object>());
            msgContent = appMessageDTO.getContent();
        }

        if (msgContent instanceof Map) {
            Map<String, Object> msgContentMap = (Map<String, Object>) msgContent;
            msgContentMap.put("msgCreateTime", System.currentTimeMillis());
        }
    }

    /**
     * 重置消息数据
     */
    public ResponseEntity<Map> resetAIMData() {
        ResponseEntity<Map> result = null;
        try {
            String uri = envProperties.getAimUri() + AimApiConstant.MISC_DEMO_RESET;
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);

            result = restTemplate.getForEntity(uri, Map.class);
            Map map = result.getBody();
            log.error("[demo reset] aim- {}", map);
        } catch (Exception ex) {
            log.error("[demo reset] aim- {}", ex);
        }
        return result;
    }

}
