package com.digiwin.athena.atmc.http.restful.atdm.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.HttpUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.http.constant.AtdmApiConstant;
import com.digiwin.athena.atmc.http.domain.action.SubmitAction;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.aglie.model.QueryDataByActionVO;
import com.digiwin.athena.atmc.http.domain.action.SubmitActionDTO;
import com.digiwin.athena.atmc.http.domain.action.SubmitExecuteContext;
import com.digiwin.athena.atmc.http.restful.atdm.AtdmService;
import com.digiwin.athena.atmc.http.restful.atdm.model.RecycleRecordStateChangeDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service
public class AtdmServiceImpl implements AtdmService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    MessageUtils messageUtils;

    /**
     * 提交
     *
     * @param submitAction
     * @return
     */
    public Map submitTask(SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }
        String url = envProperties.getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_TASK;
        url = appendTraceInfo(url, submitAction.getAction().getExecuteContext(), submitAction.getAction().getActionId());
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        // 实际执行操作的用户信息
        submitAction.setOperateAuthoredUser(AppAuthContextHolder.getContext().getAuthoredUser());

        HttpEntity httpEntity = new HttpEntity<SubmitActionDTO>(submitAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }


    /**
     * 执行
     *
     * @param submitAction
     * @return
     */
    public Map executeTask(SubmitActionDTO submitAction) {
        String url = envProperties.getAtdmUri() + AtdmApiConstant.ACTION_EXECUTE;
        url = appendTraceInfo(url, submitAction.getAction().getExecuteContext(), submitAction.getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<SubmitActionDTO>(submitAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }

    /**
     * 根据actionId查询
     *
     * @param action
     * @return
     */
    public BaseResultDTO queryByActionId(Map action) {
        String uri = envProperties.getAtdmUri() + AtdmApiConstant.DATA_QUERY_BY_ACTION_ID;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<Map>(action, headers);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<Map>>() {
                });

        return responseEntity.getBody();
    }

    /**
     * 提交合并的任务
     *
     * @param mergeActionList
     * @return
     */
    public List<Map> submitMergeTask(List<SubmitActionDTO> mergeActionList) {
        if (CollectionUtils.isEmpty(mergeActionList)) {
            return new ArrayList<>();
        }
        if (mergeActionList.get(0) == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        String uri = envProperties.getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_MERGE_TASK;

        uri = appendTraceInfo(uri, mergeActionList.get(0).getAction().getExecuteContext(), mergeActionList.get(0).getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        if (CollectionUtils.isNotEmpty(mergeActionList)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            mergeActionList.stream().forEach(action -> action.setOperateAuthoredUser(authoredUser));
        }

        HttpEntity httpEntity = new HttpEntity<List<SubmitActionDTO>>(mergeActionList, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
                });
        return responseEntity.getBody().getResponseWithException("");
    }

    private void addLang(HttpHeaders headers) {
        headers.add("locale", LocaleContextHolder.getLocale().toString());
    }

    static String appendTraceInfo(String url, SubmitExecuteContext executeContext, String actionId) {
        if (StringUtils.isEmpty(url)) {
            return url;
        }
        StringBuilder stringBuilder = new StringBuilder();
        if (executeContext != null) {
            if (executeContext.getTmActivityId() != null) {
                stringBuilder.append(String.format("&TmActivityId=%s", executeContext.getTmActivityId()));
            }
            if (executeContext.getBacklogId() != null) {
                stringBuilder.append(String.format("&BacklogId=%s", executeContext.getBacklogId()));
            }
            if (executeContext.getAuthoredUser() != null) {
                stringBuilder.append(String.format("&author=%s-%s", executeContext.getAuthoredUser().getTenantId(), executeContext.getAuthoredUser().getUserId()));
            }
            if (executeContext.getProcessSerialNumber() != null) {
                stringBuilder.append(String.format("&processSerialNumber=%s", executeContext.getProcessSerialNumber()));
            }
        }
        if (actionId != null) {
            stringBuilder.append(String.format("&actionId=%s", actionId));
        }

        if (stringBuilder.length() > 0) {
            if (url.contains("?")) {
                return url + stringBuilder.substring(1);
            } else {
                return url + "?" + stringBuilder.substring(1);
            }
        } else {
            return url;
        }
    }

    @Override
    public void markRecycleRecordExpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = envProperties.getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_EXPIRED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);

        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    @Override
    public void markRecycleRecordUnexpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = envProperties.getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_UNEXPIRED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);

        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    @Override
    public void markRecycleRecordDeleted(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = envProperties.getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_DELETED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);
        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param queryDataByActionVO
     * @return
     */
    @Override
    public BaseResultDTO<JSONObject> getDataByAction(QueryDataByActionVO queryDataByActionVO) {
//        String url = MessageFormat.format("{0}api/atdm/v1/data/query/by/action", envProperties.getAtdmUri());
        String url = envProperties.getAtdmUri() + AtdmApiConstant.DATA_QUERY_BY_ACTION;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);


        HttpEntity httpEntity = new HttpEntity<>(queryDataByActionVO, headers);
        ResponseEntity<BaseResultDTO<JSONObject>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<JSONObject>>() {
                });

        return responseEntity.getBody();
    }
}
