package com.digiwin.athena.atmc.http.restful.iam.impl;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.constant.GlobalConstant;
import com.digiwin.athena.atmc.http.constant.IamApiConstant;
import com.digiwin.athena.atmc.http.domain.usersetting.BaseUserSettingDto;
import com.digiwin.athena.atmc.http.domain.usersetting.PersonalizedDto;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.eoc.EocService;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import com.digiwin.athena.atmc.http.restful.iam.model.*;
import com.digiwin.athena.atmc.http.util.AtmcStrUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.*;

@Slf4j
@Service
public class UserServiceImpl implements UserService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    private EocService eocService;

    private static final Logger logger = LoggerFactory.getLogger(UserServiceImpl.class);

    public static final String CATALOG_ID = "basic";

    public static final String MESSAGE_NOTICE_TYPE_KEY = "MessageSendType";

    @Override
    public UserDTO query(String userId, String token) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        headers.add(GlobalConstant.IAM_APP_TOKEN, com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER;
        ResponseEntity<UserDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTO.class);
        return responseEntity.getBody();
    }

    @Override
    public UserDTO query(String userId) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_APP_TOKEN, com.digiwin.athena.appcore.auth.GlobalConstant.IAM_APP_TOKEN_ATHENA);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER;
        ResponseEntity<UserDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTO.class);
        UserDTO userDTO = responseEntity.getBody();
        if (userDTO == null || StringUtils.isEmpty(userDTO.getId())) {
            //如果没有获取到人员，则返回null
            log.error("[IAM] 查询 " + userId + " 的用户信息失败,  {}", userDTO);
            userDTO = null;
        }
        return userDTO;
    }

    /**
     * 根据用户email获取用户相关的信息
     *
     * @param email 用户email
     * @return 用户基本信息
     */
    @Override
    public UserDTO queryByEmail(String email) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("email", email);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER_EMAIL;
        ResponseEntity<UserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTOWithCode.class);
        if (responseEntity.getBody() != null && responseEntity.getBody().getCode() == 200) {
            return responseEntity.getBody().getData();
        }
        return null;
    }

    @Override
    public UserDTO queryByPhone(String phone) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("telephone", phone);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER_TELEPHONE;
        try {
            ResponseEntity<UserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTOWithCode.class);
            if (responseEntity.getBody() != null && responseEntity.getBody().getCode() == 200) {
                return responseEntity.getBody().getData();
            }
        } catch (Exception e) {
            logger.error("queryByPhone error params:{}", param, e);
            return null;
        }
        return null;
    }

    /**
     * 根据用户email和租户Id获取用户相关的信息
     *
     * @param email 用户email
     * @return 用户基本信息
     */
    @Override
    public UserDTO queryByEmail(String email,String tenantId){
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("email", email);
        param.put("tenantId", tenantId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER_EMAIL;
        ResponseEntity<UserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTOWithCode.class);
        if (responseEntity.getBody() != null && responseEntity.getBody().getCode() == 200) {
            return responseEntity.getBody().getData();
        }
        return null;
    }

    /**
     * 根据用户phone和租户Id获取用户相关的信息
     *
     * @param phone 用户phone
     * @return 用户基本信息
     */
    @Override
    public UserDTO queryByPhone(String phone, String tenantId) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("telephone", phone);
        param.put("tenantId", tenantId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.USER_TELEPHONE;
        try {
            ResponseEntity<UserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, UserDTOWithCode.class);
            if (responseEntity.getBody() != null && responseEntity.getBody().getCode() == 200) {
                return responseEntity.getBody().getData();
            }
        } catch (Exception e) {
            logger.error("queryByPhone error params:{}", param, e);
            return null;
        }
        return null;
    }

    @Override
    public UserDTO query(String mappingApp, String account, String verifyUserId, String token) {
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("identityId", mappingApp);
        param.put("verifyUserId", verifyUserId);
        if (!StringUtils.isEmpty(account)) {
            param.put("account", account);
        }
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getIamUri() + IamApiConstant.MAPPING_QUERY_QPPUSER;
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
        Map body = responseEntity.getBody();
        if (body == null) {
            return null;
        } else {
            String userId = body.get("userId").toString();
            return query(userId, token);
        }
    }

    /**
     * 通过用户ID 获取员工的多语言
     *
     * @param userId
     * @param token
     * @return
     */
    @Override
    public String getUserLangNameByUserId(String userId, String tenantId, String token) {
        return getUserLangMetadataAllTenant(userId,token);
    }

    /**
     * 通过用户ID 获取是否静默
     *
     * @param userId
     * @param token
     * @return
     */
    @Override
    public boolean getMessageDoNotDisturbUserId(String userId, String tenantId, String token) {
        try {
            //获取员工ID
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            headers.add(GlobalConstant.IAM_USER_TOKEN, token);
            //post body
            Map<String, String> param = new HashMap<>();
            param.put("tenantId", tenantId);
            param.put("userId", userId);
            param.put("catalogId", "basic");
            param.put("key", "doNotDisturb");
            HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
            String url = envProperties.getIamUri() + IamApiConstant.USER_METADATA_VALUE;
            ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
            Map result = responseEntity.getBody();
            if (result.containsKey("data")) {
                return "1".equals(result.get("data").toString());
            }
            return false;
        } catch (Exception ex) {
            logger.error(ex.getMessage(), ex);
            return false;
        }
    }


    @Override
    public UserDTO getTargetUser(String userId, String token) {
        UserDTO targetUser = null;
        ProxyUserDTO proxyUserDTO = eocService.getProxyUser(userId, token);
        if (proxyUserDTO != null && StringUtils.hasText(proxyUserDTO.getUserId())) {
            targetUser = this.query(proxyUserDTO.getUserId(), token);
        }

        if (targetUser == null) {
            targetUser = this.query(userId, token);
        }
        return targetUser;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getUserAcceptContacts(String userId, Long tenantSid, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);

        Map<String, Object> params = new HashMap<>();
        params.put("userId", userId);
        params.put("tenantSid", tenantSid);
        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_USER_ACCEPTCONTACTS_GET;
        ResponseEntity<UserAcceptContactsDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserAcceptContactsDTO.class);
        UserAcceptContactsDTO baseResult = responseEntity.getBody();
        if (baseResult.getCode() == 200) {
            return baseResult.getData();
        } else {
            return null;
        }
    }

    /**
     * 获取指定用户所有租户下的元数据信息
     *
     * @param userId
     * @return
     */
    @Override
    public UserMetadataDTO getUserMetadataAllTenant(String userId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> params = new HashMap<>();
        params.put("id", userId);
        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_CURRENT;
        ResponseEntity<UserMetadataDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserMetadataDTO.class);
        return responseEntity.getBody();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getUserLangMetadataAllTenant(String userId, String token) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);

        log.info("getUserLangMetadataAllTenant-userId:{}",userId);
        Map<String, Object> params = new HashMap<>();
        params.put("catalogId", "basic");
        params.put("key", "dapAcceptLanguage");
        params.put("userId", userId);
        params.put("tenantSid", 0);
        HttpEntity httpEntity = new HttpEntity(params, headers);

        log.info("{}.{}", token, JsonUtils.objectToString(params));

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_USER_GET;

        try{
            ResponseEntity<UserLangMetadataDTO> respEntity = restTemplate.postForEntity(url, httpEntity, UserLangMetadataDTO.class);
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                UserLangMetadataDTO baseResult = respEntity.getBody();
                if (null != baseResult && HttpStatus.OK.value() == baseResult.getCode()) {
                    Optional<Object> langOpt = Optional.ofNullable(baseResult.getData()).map(data -> data.get("value"));
                    //如果用户设置了语言别，则使用用户的语言别，如果没有使用语言别，则使用默认的语言别
                    return langOpt.isPresent() ? String.valueOf(langOpt.get()) : defaultLanguage();
                }
            }
            throw BusinessException.create(respEntity.getStatusCodeValue(), JsonUtils.objectToString(respEntity.getBody()));
        }catch (Exception ex) {
            log.error("{}, 请求body体：{}, 报错信息：", ErrorCodeEnum.IAM_GET_USERMETADATA_FAIL.getErrCode(), JsonUtils.objectToString(params), ex);
            throw ExceptionUtil.wrap(ErrorCodeEnum.IAM_GET_USERMETADATA_FAIL.getErrCode(), ex);
        }
    }

    /**
     * 获取用户默认的语言别
     * @return 语言别信息
     */
    private String defaultLanguage()
    {

            return org.apache.commons.lang.StringUtils.equals("SIMPLIFIED_CHINESE", envProperties.getDefaultLanguage()) ? Locale.SIMPLIFIED_CHINESE.toString() : Locale.TRADITIONAL_CHINESE.toString();

    }

    /**
     * 获取指定用户所有租户下的元数据信息(gai 该方法会增加缓存处理，请注意)
     *
     * @param userId
     * @return
     */
    @Override
    public UserMetadataDTO getUserMetadataInAllTenant(String userId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> params = new HashMap<>();
        params.put("id", userId);
        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_ALL;
        ResponseEntity<UserMetadataDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, UserMetadataDTO.class);
        return responseEntity.getBody();
    }

    /**
     * 企業邀請用戶加入【Athena】【4.15】
     * @param email
     * @param tenantId
     * @param appCode
     * @param locale
     */
    @Override
    public ReplyInviteResultDTO replyInvite(String email, String tenantId, String appCode, String locale){
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        if(locale!=null){
            headers.add("locale", locale);
        }
        Map<String, Object> params = new HashMap<>();
        params.put("userEmail", email);
        params.put("tenantId", tenantId);
        params.put("inviteApp", appCode);

        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.TENANT_USER_INVITE_NEW;
        ResponseEntity<ReplyInviteResultDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, ReplyInviteResultDTO.class);
        return responseEntity.getBody();

    }

    /**
     * 判断邮箱帐号是否已经注册,且是否为鼎捷用户，排除掉企业内部用户
     * @param email
     * @return
     */
    @Override
    public InnerExistDTO emailInnerExist(String email){
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> params = new HashMap<>();
        params.put("email", email);

        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USER_EMAIL_WITHOUT_INNER_EXIST;
        ResponseEntity<InnerExistDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, InnerExistDTO.class);
        return responseEntity.getBody();
    }

    /**
     * 判断手机号是否已经注册,且是否为鼎捷用户，排除掉企业内部用户
     * @param phone
     * @return
     */
    @Override
    public InnerExistDTO phoneInnerExist(String phone) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> params = new HashMap<>();
        params.put("telephone", phone);

        HttpEntity httpEntity = new HttpEntity(params, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USER_MOBILEPHONE_WITHOUT_INNER_EXIST;
        ResponseEntity<InnerExistDTO> responseEntity = restTemplate.postForEntity(url, httpEntity, InnerExistDTO.class);
        InnerExistDTO baseResult = responseEntity.getBody();

        try {
            restTemplate.postForEntity(url, httpEntity, Map.class);
        } catch (Exception e) {
            logger.error("getUserPhoneIsExit error params:{}", params, e);
            return baseResult;
        }
        return baseResult;
    }


    /**
     * 查询用户的个性化设置数据
     *
     * @param tenantSid 租户SID，注意非租户id
     * @param userId    用户id
     * @param key       需要查询的key
     * @param catalogId 类目id
     * @return PersonalizedDto
     */
    public PersonalizedDto queryPersonalizedInfo(Long tenantSid, String userId, String key, String catalogId) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> param = new HashMap<>();
        param.put("tenantSid", tenantSid);
        param.put("userId", userId);
        param.put("catalogId", catalogId);
        param.put("key", key);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        log.info("[UserServiceImpl.queryPersonalizedInfo],param :{},headers :{}",param,headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_USER_GET;
        ResponseEntity<BaseUserSettingDto> dataEntity = restTemplate.postForEntity(url, httpEntity, BaseUserSettingDto.class);
        if (dataEntity.getBody() == null) {
            logger.error("can not find user meta response body  with:{}", param);
            return null;
        }
        else if(dataEntity.getBody().getData() == null)
        {
            logger.error("can not find user meta response data  with:{}", param);
            return null;
        }
        else {
            BaseUserSettingDto baseDto = dataEntity.getBody();
            return JsonUtils.jsonToObject(JsonUtils.objectToString(baseDto.getData()), PersonalizedDto.class);
        }
    }


    /**
     * 查询用户的个性化设置数据
     *
     * @param userId   用户id
     * @param metaList 需要修改的参数列表
     * @return PersonalizedDto
     */
    public int updatePersonalizedInfo(String userId, List<Object> metaList) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> param = new HashMap<>();
        param.put("id", userId);
        param.put("metadata", metaList);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getIamUri() + IamApiConstant.USERMETADATA_UPDATE;
        try {
            restTemplate.postForEntity(url, httpEntity, Map.class);
        } catch (Exception e) {
            logger.error("update user meta data error with param:{}", param, e);
            return -1;
        }
        return 0;

    }


    /**
     * 根据用户id查询出用户的
     * @param userId 用户id
     * @return email
     */
    public String queryUserEmail(String userId)
    {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> param = new HashMap<>();
        param.put("id", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getIamUri() + AtmcStrUtil.format(IamApiConstant.USER_CONTACTS_USERID, userId);
        try {
            ResponseEntity<BaseUserSettingDto> dataEntity= restTemplate.postForEntity(url, httpEntity, BaseUserSettingDto.class);

            if(dataEntity.getBody()!=null)
            {
                BaseUserSettingDto baseDto = dataEntity.getBody();
                if(null == baseDto || null == baseDto.getData())
                {
                    logger.error("Query user email with no data return:{}", userId);
                    return null;
                }
                Map<String,Object> resultMap = JsonUtils.jsonToObject(JsonUtils.objectToString(baseDto.getData()), Map.class);
                return MapUtils.getString(resultMap,"email");
            }
        } catch (Exception e) {
            logger.error("Query User Email info Error error with param:{}", userId, e);
            return null;
        }
        return null;
    }

    /**
     * 根据邮箱获取任务的实际执行人
     * @Author：SYQ
     * @Date：2021/7/2 11:21
     */
    public String getActualPerformId(String tenantId, String emailAddress, AuthoredUser user){
        String actualPerformId = "";
        //判断邮箱是否存在
        if(StringUtils.isEmpty(emailAddress)){
            return actualPerformId;
        }
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        //接口地址
        String url = envProperties.getIamUri() + IamApiConstant.USER_EMAIL;
        //请求参数
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("tenantId",tenantId);
        paramMap.put("email",emailAddress);
        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap,headers);
        //调用接口
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url,httpEntity, Map.class);
        if (responseEntity.getStatusCode() == HttpStatus.OK) {
            Object response = responseEntity.getBody().get("data");
            if(response != null && response != ""){
                Map<String,String> resultMap = JsonUtils.jsonToObject(JsonUtils.objectToString(response),Map.class);
                actualPerformId = resultMap.get("id");
            }
        }
        return actualPerformId;
    }

    public ResponseEntity<Map> invoiceTaxno(String unifyNumber, AuthoredUser user) {
        //统编不能为空
        if (StringUtils.isEmpty(unifyNumber)) {
            return null;
        }
        //接口地址
        String url = envProperties.getIamUri() + AtmcStrUtil.format(IamApiConstant.INVOICE_TAXNO, unifyNumber);
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        //请求参数
        HttpEntity<?> httpEntity = new HttpEntity<>(new HashMap<>(), headers);
        //调用接口
        return restTemplate.postForEntity(url, httpEntity, Map.class);
    }

    /**
     * 获取内部员工
     * @Author：SYQ
     * @Date：2021/6/15 14:42
     */
    public List<String> getEmployee(List<String> userList,AuthoredUser user){
        List<String> memberList = new ArrayList<>();
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        //地址
        String url = envProperties.getIamUri() + IamApiConstant.IAM_V_2_USER_LIST;
        //参数
        HttpEntity<?> httpEntity = new HttpEntity<>(userList, headers);
        //获取用户信息
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url,httpEntity,Map.class);
        //判断调用API是否成功
        if(responseEntity.getStatusCode() == HttpStatus.OK && responseEntity.getBody().get("data") != null){
            //获取data数据
            JSONArray jsonArray = JSONArray.fromObject(responseEntity.getBody().get("data"));
            //循环判断用户是否为内部用户
            for(Object object : jsonArray){
                //用户数据
                Map map = JsonUtils.jsonToObject(JsonUtils.objectToString(object),Map.class);
                //type为0的是内部用户
                if("0".equals(map.get("type").toString())){
                    String id = eocService.getEocEmpInfo(map.get("id").toString(),user);
                    memberList.add(id);
                }
            }
        }
        return memberList;
    }
}
