package com.digiwin.athena.atmc.http.restful.dap.impl;

import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.digiwin.athena.atmc.http.constant.DapApiConstant;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.dap.SearchService;
import com.digiwin.athena.atmc.http.restful.dap.model.SearchSuggestDTO;
import com.digiwin.athena.atmc.http.domain.task.SearchReqDTO;
import com.google.common.collect.Sets;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.*;

@Service
public class SearchServiceImpl implements SearchService {
    private static final Logger logger = LoggerFactory.getLogger(SearchServiceImpl.class);

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    private static final String DEFAULT_SUGGEST_FIELD_NAME = "name";

    private static final String DEFAULT_SUGGEST_INDEX_NAME = "completion_index";

    private static final int DEFAULT_SUGGEST_QRY_SIZE = 8;

    /**
     * 获取搜索建议
     *
     * @param searchSuggestDTO
     */
    @Override
    public ResponseEntity<?> getSearchSuggest(SearchSuggestDTO searchSuggestDTO) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        JSONObject queryParams = new JSONObject();
        queryParams.put("prefix", searchSuggestDTO.getPrefix());
        queryParams.put("fieldName", DEFAULT_SUGGEST_FIELD_NAME);
        queryParams.put("indexName", DEFAULT_SUGGEST_INDEX_NAME);

        int size = null != searchSuggestDTO.getSize() && searchSuggestDTO.getSize() > 0 ? searchSuggestDTO.getSize() : DEFAULT_SUGGEST_QRY_SIZE;
        queryParams.put("size", size);

        HttpEntity<?> httpEntity = new HttpEntity<>(queryParams, headers);
        String url = envProperties.getSearchUri() + DapApiConstant.SEARCH_SUGGEST_URL;
        logger.debug("getSearchSuggest-> url:{} httpEntity: {}", url, httpEntity);
        JSONObject response = restTemplate.postForObject(url, httpEntity, JSONObject.class);
        logger.debug("getSearchSuggest-> response: {}", response);
        return convertSuggestQryRet(response);
    }

    private ResponseEntity<?> convertSuggestQryRet(JSONObject suggestResp) {
        int status = suggestResp.getInt("status");
        if (HttpStatus.SC_OK != status) {
            return ResponseEntityWrapper.wrapperFail(status, suggestResp.getString("statusDescription"));
        }

        Set<String> response = Sets.newHashSet();
        JSONArray respArray = suggestResp.getJSONArray("response");
        if (null != respArray && respArray.size() > 0) {
            for (int idx = 0; idx < respArray.size(); idx++) {
                JSONObject respObj = respArray.getJSONObject(idx);
                if (null != respObj) {
                    response.add(respObj.getString("text"));
                }
            }
        }

        return ResponseEntityWrapper.wrapperOk(response);
    }

    /**
     * 全文搜索
     *
     * @param reqDTO
     */
    @Override
    public List<Map<String, Object>> doSearch(SearchReqDTO reqDTO) {
        List<Map<String, Object>> resp = new ArrayList<>();
        // 记录搜索记录
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //post body
        JSONObject queryInfo = new JSONObject();
        queryInfo.put("text", reqDTO.getParam());
        queryInfo.put("indexName", "mariadb-athena-*");
        JSONArray fieldNames = new JSONArray();
        fieldNames.add("name");
//        fieldNames.add("start_time");
//        fieldNames.add("end_time");
        fieldNames.add("target_name");
        fieldNames.add("performer_name");
        fieldNames.add("person_in_charge_name");
        queryInfo.put("fieldNames", fieldNames);
        HttpEntity<?> httpEntity = new HttpEntity<>(queryInfo, headers);
        String url = envProperties.getSearchUri() + DapApiConstant.RESTFUL_STANDARD_FULLTEXT_SEARCH;
        logger.debug("doSearch-> url:{} httpEntity: {}", url, httpEntity);
        JSONObject response =  restTemplate.postForObject(url, httpEntity, JSONObject.class);
//        logger.info("doSearch-> response: {}", response);
        JSONArray hits = response.getJSONObject("response").getJSONArray("hits");
        for(int i = 0; i < hits.size(); i++) {
            Map<String, Object> data = new HashMap<>();
            JSONObject source = hits.getJSONObject(i).getJSONObject("_source");
            if (source != null) {
                String startTime = TimeUtils.format(TimeUtils.parse(source.getString("start_time"))
                        , TimeUtils.DEFAULT_FORMAT).substring(5, 10).replace("-", ".");
                String endTime = TimeUtils.format(TimeUtils.parse(source.getString("end_time"))
                        , TimeUtils.DEFAULT_FORMAT).substring(5, 10).replace("-", ".");
                if (source.get("task_id") == null) {
                    data.put("type", "task");
                    data.put("term", startTime + "-" + endTime + " " +
                            MessageUtils.getMessageByCurrentLanguage( source.getString("target_name")) + "-" +    MessageUtils.getMessageByCurrentLanguage( source.getString("name")));
                } else {
                    data.put("type", "backlog");
                    data.put("term", startTime + "-" + endTime + " " + MessageUtils.getMessageByCurrentLanguage( source.getString("name")));
                }
                data.put("key", source.getString("id"));
                data.put("searchData",hits.getJSONObject(i));
            }
            resp.add(data);
        }
        return resp;
    }
}
