package com.digiwin.athena.atmc.http.restful.iam.impl;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.http.constant.GlobalConstant;
import com.digiwin.athena.atmc.http.constant.IamApiConstant;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.iam.model.BossIamAddUserResultDTO;
import com.digiwin.athena.atmc.http.restful.iam.model.BossIamCheckUserResultDTO;
import com.digiwin.athena.atmc.http.restful.iam.model.BossIamTenantAuthDTO;
import com.digiwin.athena.atmc.http.restful.iam.model.BossIamTenantDTO;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocResultDTO;
import com.digiwin.athena.atmc.http.restful.iam.BossIamService;
import com.github.pagehelper.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 企业用户管理
 */
@Slf4j
@Service
public class BossIamServiceImpl implements BossIamService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    /**
     * 检查
     * @param tenantId
     * @param appId
     * @param email
     * @return
     */
    public BossIamCheckUserResultDTO checkTenantUser(String tenantId, String appId, String email){

        BossIamCheckUserResultDTO result = null;

        String uri = envProperties.getIamBossUri() + IamApiConstant.USERS_CHECK_USER_TENANT_AUTH;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, String> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("appId", appId);
        param.put("email", email);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<Map> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity,
                new ParameterizedTypeReference<Map>() {
                },
                param);

        Map body = responseEntity.getBody();
        if(body.containsKey("data")){
            result = JsonUtils.jsonToObject(JsonUtils.objectToString(body.get("data")), BossIamCheckUserResultDTO.class);
        }

        return result;
    }

    /**
     * 将用户添加到企业用户中
     * @param tenantId
     * @param email
     */
    public BossIamAddUserResultDTO addTenantUser(String tenantId, String email, String pwd, String appId){
/*
{
    "tenantId": "88880051",
    "id": "zhujg@digiwin.com",
    "name": "zhujg",
    "password": "mock",
    "app": [
        {
            "sid": 199246554387008,
            "id": "athena_task_MO",
            "name": "备料应用"
        }
    ],
    "email": "zhujg@digiwin.com",
    "telephone": "",
    "sendEmail": true
}
 */

        String uri = envProperties.getIamBossUri() + IamApiConstant.USERS_ENTERPRISE;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("id", email);
        param.put("name", email);
        param.put("password", pwd);
        param.put("email", email);
        param.put("telephone", "");
        param.put("sendEmail", true);

        Map<String, Object> appParam = new HashMap<>();
        appParam.put("sid", 199246554387008L);
        appParam.put("id", appId);
        appParam.put("name", appId);
        List<Map<String,Object>> apps = new ArrayList<>();
        apps.add(appParam);
        param.put("app", apps);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<BossIamAddUserResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BossIamAddUserResultDTO>() {
                }, new HashMap<>());

        return responseEntity.getBody();
    }

    /**
     * 根据邮箱获取个人租户
     * status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     * @Author：SYQ
     * @Date：2021/11/24 14:20
     */
    @Override
    public Map<String,Object> getTenantAuth(AuthoredUser authoredUser, String email, String telephone, String appId, Integer tenantType) {
        Map<String,Object> resultMap = new HashMap<>();
        //邮箱手机，不能全为空
        if(StringUtil.isEmpty(email) && StringUtil.isEmpty(telephone)){
            return null;
        }
        //默认值为Athena
        appId = StringUtil.isNotEmpty(appId) ? appId : "Athena";
        //默认值为0
        tenantType = tenantType != null ? tenantType : 0;

        //接口地址
        String url = envProperties.getIamUri() + IamApiConstant.USER_EMAIL_TENANTS_AUTH;
        //头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());
        //参数
        Map<String, Object> param = new HashMap<>();
        param.put("email",email);
        param.put("telephone",telephone);
        param.put("appId",appId);
        param.put("tenantType",tenantType);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        try {
            //调用接口
            ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
            //调用失败
            if(responseEntity == null || responseEntity.getStatusCode() != HttpStatus.OK){
                resultMap.put("status",-1);
                log.error("获取用户归属的租户以及指定应用授权信息失败");
                return resultMap;
            }
            //返回值
            EocResultDTO eocResultDTO = responseEntity.getBody();
            //返回值为空
            if(eocResultDTO == null){
                resultMap.put("status",-1);
                log.error("获取用户归属的租户以及指定应用授权信息失败");
                return resultMap;
            }

            //解析租户信息和用户信息
            BossIamTenantAuthDTO bossIamTenantAuthDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(eocResultDTO.getData()),BossIamTenantAuthDTO.class);
            //租户信息
            List<BossIamTenantDTO> bossIamTenantDTOS = bossIamTenantAuthDTO.getTenants();
            //没有个人租户
            if(CollectionUtils.isEmpty(bossIamTenantDTOS)){
                resultMap.put("status",0);
                log.error("获取用户归属的租户以及指定应用授权信息失败：没有个人租户");
                return resultMap;
            }
            //取第一条个人租户
            BossIamTenantDTO bossIamTenantDTO = bossIamTenantDTOS.get(0);
            //判断有没有授权
            if(!bossIamTenantDTO.getIsTenantAuth() || !bossIamTenantDTO.getIsUserAuth()){
                resultMap.put("status",1);
                resultMap.put("userId",bossIamTenantAuthDTO.getUserId());
                resultMap.put("tenantId",bossIamTenantDTO.getId());
                resultMap.put("isTenantAuth",bossIamTenantDTO.getIsTenantAuth());
                resultMap.put("isUserAuth",bossIamTenantDTO.getIsUserAuth());
                log.error("获取用户归属的租户以及指定应用授权信息失败（没有授权）：{}",bossIamTenantAuthDTO);
                return resultMap;
            }
            //获取用户id和租户id
            resultMap.put("status",2);
            resultMap.put("userId",bossIamTenantAuthDTO.getUserId());
            resultMap.put("tenantId",bossIamTenantDTO.getId());
            resultMap.put("isTenantAuth",bossIamTenantDTO.getIsTenantAuth());
            resultMap.put("isUserAuth",bossIamTenantDTO.getIsUserAuth());
        }catch (Exception e){
            resultMap.put("status",-1);
            log.error("获取用户归属的租户以及指定应用授权信息失败：{}",e);
        }
        return resultMap;
    }

    /**
     * 创建个人租户以及授权
     * @Author：SYQ
     * @Date：2021/11/24 15:43
     */
    @Override
    public Map<String, String> registerTenant(AuthoredUser authoredUser,String email, String telephone, String appId, String strategySid, String strategyCode) {
        Map<String,String> resultMap = new HashMap<>();
        //邮箱不能拿为空
        if(StringUtil.isEmpty(email)){
            return resultMap;
        }
        //默认值为Athena
        appId = StringUtil.isNotEmpty(appId) ? appId : "Athena";
        //默认值为athena_light_01
        strategyCode = StringUtil.isNotEmpty(strategyCode) ? strategyCode : "athena_light_01";

        //接口地址
        String url = envProperties.getIamUri() + IamApiConstant.USER_INDIVIDUAL_TENANT_REGISTER;

        //头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, authoredUser.getToken());
        //参数
        Map<String, Object> param = new HashMap<>();
        param.put("email",email);
        param.put("telephone",telephone);
        param.put("appId",appId);
        param.put("strategySid",strategySid);
        param.put("strategyCode",strategyCode);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //调用接口
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        if(responseEntity != null && responseEntity.getStatusCode() == HttpStatus.OK && responseEntity.getBody() != null){
            Object data = responseEntity.getBody().getData();
            resultMap = JsonUtils.jsonToObject(JsonUtils.objectToString(data),Map.class);
        }
        return resultMap;
    }

    /**
     * 给指定租户下的用户授权指定应用
     * @Author：SYQ
     * @Date：2021/11/24 15:50
     */
    @Override
    public Boolean authorizeUser(AuthoredUser authoredUser,String tenantId, String email, String telephone,
                                 String appId, String strategySid, String strategyCode,Boolean isTenantAuth) {
        Boolean result = true;
        //邮箱和租户id不能为空
        if(StringUtil.isEmpty(email) && StringUtil.isEmpty(tenantId)){
            return Boolean.FALSE;
        }
        //默认值为Athena
        appId = StringUtil.isNotEmpty(appId) ? appId : "Athena";
        //默认值为athena_light_01
        strategyCode = StringUtil.isNotEmpty(strategyCode) ? strategyCode : "athena_light_01";

        //接口地址
        String url = envProperties.getIamUri() + IamApiConstant.USER_TENANT_USER_AUTH;

        //头部数据
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //参数
        Map<String, Object> param = new HashMap<>();
        param.put("tenantId",tenantId);
        param.put("email",email);
        param.put("telephone",telephone);
        param.put("appId",appId);
        param.put("strategySid",strategySid);
        param.put("strategyCode",strategyCode);
        param.put("isTenantAuth",isTenantAuth);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //调用接口
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        if(responseEntity != null && responseEntity.getStatusCode() == HttpStatus.OK && responseEntity.getBody() != null){
            result = Boolean.valueOf(responseEntity.getBody().getData().toString());
        }
        return result;
    }

}
