package com.digiwin.athena.atmc.http.restful.bpm.impl;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.http.constant.TaskEngineApiConstant;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveActivityDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApprovePreviousActivityDTO;
import com.digiwin.athena.atmc.http.restful.bpm.TaskEngineService;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveRequestDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskPreApproveRequestDTO;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service
public class TaskEngineServiceImpl implements TaskEngineService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    private MessageUtils messageUtils;

    /**
     * 获取预测关卡
     * @param taskUid 任务UID
     * @return
     */
    public List<Map> getForeCastTask(String taskUid) {
        return getForeCastTaskAndPlanEndTime(taskUid, null);
    }

    /**
     * 获取预测关卡 及 预测结束时间
     * @param taskUid
     * @param newPlanEndTime
     * @return
     */
    public List<Map> getForeCastTaskAndPlanEndTime(String taskUid, String newPlanEndTime) {
        // http://athena-devops-doclever.digiwincloud.com.cn/html/web/controller/share/share.html#61020120e1305d000c0ea70f

        List<Map> result = null;
        //语言别
        String locale = "";
        if("SIMPLIFIED_CHINESE".equals(envProperties.getDefaultLanguage())){
            locale = "zh_CN";
        }else if("TRADITIONAL_CHINESE".equals(envProperties.getDefaultLanguage())){
            locale = "zh_TW";
        }
        //查询参数
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("taskUid",taskUid);
        paramMap.put("locale",locale);
        if(StringUtils.isNotEmpty(newPlanEndTime)){
            paramMap.put("planEndTime", newPlanEndTime);
        }
        //头部信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取appId接口地址
        String getAppIdUrl = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_FORECAST;
        //查询数据
        HttpEntity<?> httpEntity = new HttpEntity<Map>(paramMap, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> appIdResponse = restTemplate.exchange(getAppIdUrl, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>(){});
        //获取appId
        if(appIdResponse.getStatusCode() == HttpStatus.OK){
            result = appIdResponse.getBody().getResponse();
        }
        return result;
    }

    /**
     * 普通人工任务-转派
     * @param workitemId
     * @param oldPerformerId
     * @param newPerformerId
     * @param comment
     */
    public void reassignManualTask(String workitemId,String oldPerformerId, String newPerformerId, String comment){

        //查询参数
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("workitemId", workitemId);
        paramMap.put("performerId", oldPerformerId);
        paramMap.put("performerType", 0);
        paramMap.put("newPerformerId", newPerformerId);
        paramMap.put("comment", comment);

        //头部信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //获取appId接口地址
        String serviceUri = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_MANUAL_REASSIGN;

        //查询数据
        HttpEntity<?> httpEntity = new HttpEntity<Map>(paramMap, headers);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    /**
     * 签核任务-转派
     * @param workitemId
     * @param oldPerformerId
     * @param newPerformerId
     * @param comment
     */
    public void reassignApproveTask(String workitemId,String oldPerformerId, String newPerformerId, String comment){

        //查询参数
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("workitemId", workitemId);
        paramMap.put("performerId", oldPerformerId);
        paramMap.put("performerType", 0);
        paramMap.put("newPerformerId", newPerformerId);
        paramMap.put("comment", comment);

        //头部信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //获取appId接口地址
        String serviceUri = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_APPROVE_WORK_REASSIGN;

        //查询数据
        HttpEntity<?> httpEntity = new HttpEntity<Map>(paramMap, headers);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    /**
     * 普通人工任务-派送
     * @param workitemId
     * @param comment
     */
    public void dispatchTask(String workitemId, String performerId, String comment){
//查询参数
        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("workitemId", workitemId);
        paramMap.put("performerId", performerId);
        paramMap.put("performerType", 0);
        paramMap.put("comment", comment);

        //头部信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //获取appId接口地址
        String serviceUri = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_MANUAL_DISPATCH;

        //查询数据
        HttpEntity<?> httpEntity = new HttpEntity<Map>(paramMap, headers);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    /**
     * 任务引擎-新增工作事项
     * @param bpmActivityOID
     * @param performerId
     */
    public void addWorkitem(String bpmActivityOID, String performerId){

        //  http://athena-devops-doclever.digiwincloud.com.cn/html/web/controller/share/share.html#6128b5bba751d7000cf02065

        Map<String,Object> paramMap = new HashMap<>();
        paramMap.put("activityUid", bpmActivityOID);
        paramMap.put("performerId", performerId);

        //头部信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //获取appId接口地址
        String serviceUri = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_MANUAL_WORKITEM_ADD;

        //查询数据
        HttpEntity<?> httpEntity = new HttpEntity<Map>(paramMap, headers);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    /**
     * 查询指定任务的签核历程
     * @param requestDTO
     * @return 没有数据返回空集合
     */
    @Override
    public List<BpmTaskApproveActivityDTO> queryApproveByTaskUid(BpmTaskApproveRequestDTO requestDTO){
        //  http://athena-devops-doclever.digiwincloud.com.cn/html/web/controller/share/share.html#618a4467112f8e000b017864

        List<BpmTaskApproveActivityDTO> result = null;

        //请求地址
        String url = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_APPROVE_PROCESS;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String,Object> rootPara = new HashMap<>();
        rootPara.put("locale", requestDTO.getLocale());
        rootPara.put("taskUid", requestDTO.getTaskUid());

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<BpmTaskApproveActivityDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST,httpEntity,new ParameterizedTypeReference<BaseResultDTO<List<BpmTaskApproveActivityDTO>>>(){}, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new ArrayList<>() : result;

    }

    /**
     *查询指定任务可退回重签信息
     * @param requestDTO
     * @return
     */
    @Override
    public BpmTaskApprovePreviousActivityDTO queryPreApprove(BpmTaskPreApproveRequestDTO requestDTO) {
        BpmTaskApprovePreviousActivityDTO result = null;

        //请求地址 由taskEngineUrl修改为ptmUri
        String url = envProperties.getPtmUri() + TaskEngineApiConstant.TASK_APPROVE_WORK_PREVIOUS_ACTIVITIES;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String,Object> rootPara = new HashMap<>();
        rootPara.put("locale", requestDTO.getLocale());
        rootPara.put("workitemId", requestDTO.getWorkitemId());

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<BpmTaskApprovePreviousActivityDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST,httpEntity,new ParameterizedTypeReference<BaseResultDTO<BpmTaskApprovePreviousActivityDTO>>(){}, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new BpmTaskApprovePreviousActivityDTO() : result;
    }

    /**
     * 批次关闭任务
     */
    @Override
    public void batchTerminateTask(List<String> taskUids){

        //请求地址
        String url = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.TASK_TERMINATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String,Object> rootPara = new HashMap<>();
        rootPara.put("taskUids", taskUids);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST,httpEntity,new ParameterizedTypeReference<BaseResultDTO<Map>>(){}, rootPara);
        responseEntity.getBody().getResponseWithException("");
    }

    /**
     * 项目转派时，批量变更当责者，<br>
     * 只支持任务引擎
     * @param personInCharge
     * @param personInChargeName
     * @param processSerialNumberList
     */
    @Override
    public void batchChangePersonInCharge(String personInCharge, String personInChargeName, List<String> processSerialNumberList){
        //请求地址
        String url = envProperties.getTaskEngineUrl() + TaskEngineApiConstant.PROJECT_DETAIL_IN_CHARGE_BATCH;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String,Object> rootPara = new HashMap<>();
        rootPara.put("personInCharge", personInCharge);
        rootPara.put("personInChargeName", personInChargeName);
        rootPara.put("serialNumberList", processSerialNumberList);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST,httpEntity,new ParameterizedTypeReference<BaseResultDTO<Map>>(){}, rootPara);
        responseEntity.getBody().getResponseWithException("");
    }
}
