package com.digiwin.athena.atmc.http.restful.ptm.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.constant.ErrorTypeEnum;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.constant.PtmApiConstant;
import com.digiwin.athena.atmc.http.domain.backlog.TerminateTaskDataDTO;
import com.digiwin.athena.atmc.http.domain.task.TaskExceptionDTO;
import com.digiwin.athena.atmc.http.domain.task.mobile.MobileBacklogGroupDTO;
import com.digiwin.athena.atmc.http.domain.task.mobile.MobileProjectGroupDTO;
import com.digiwin.athena.atmc.http.domain.task.mobile.MobilePtmBacklogDTO;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.HttpAssistantServiceImpl;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveRequestDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApprovelDTO;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import com.digiwin.athena.atmc.http.restful.ptm.PtmService;
import com.digiwin.athena.atmc.http.restful.ptm.model.*;
import com.digiwin.athena.atmc.http.util.AtmcStrUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.time.LocalDateTime;
import java.util.*;

@Service
@Slf4j
public class PtmServiceImpl implements PtmService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    UserService userService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private HttpAssistantServiceImpl httpAssistantService;

    /**
     * 查询ptm项目卡列表
     *
     * @param userIds 要查询的userId
     * @return 没有值则返回空 List
     */
    @Override
    public List<PtmProjectCardItemDTO> queryProjectCardList(List<String> userIds) {
        return queryProjectCard(userIds, null, null);
    }

    @Override
    public List<PtmProjectCardItemDTO> queryProjectCardListByCodes(List<String> codes) {
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_PROCESSING_SIMPLIFY_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("codes", codes);
        return httpAssistantService.post(new TypeReference<List<PtmProjectCardItemDTO>>() {
        }, url, headers, rootPara);
    }

    /**
     * 根据 项目卡Id 查询ptm项目卡列表
     *
     * @param projectCardIdList 要查询的项目卡id
     * @return 没有值则返回空 List
     */
    @Override
    public List<PtmProjectCardItemDTO> queryProjectCardListByProjectCardId(List<Long> projectCardIdList, String autherUserId) {
        return queryProjectCard(null, projectCardIdList, autherUserId);
    }

    /**
     * 查询进行中的任务概要信息（项目id、项目名称，附属的任务id、任务名称等）
     *
     * @param userIds        用户id列表
     * @param projectCardIds 项目卡id列表，可选
     * @return
     */

    @Override
    public List<ProjectCardItemSimpleInfoDTO> queryProjectCardSimpleInfo(List<String> userIds, Collection<Long> projectCardIds, String authoredUserId) {
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_PROJECT_CARD_PROCESSING_LIST;

        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        String userId = StringUtils.isBlank(authoredUserId) ? authoredUser.getUserId() : authoredUserId;
//        headers.add("locale", userService.getUserLangMetadataAllTenant(userId, authoredUser.getToken()));

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        if (CollectionUtils.isNotEmpty(userIds)) {
            rootPara.put("userIds", userIds);
        }
        if (CollectionUtils.isNotEmpty(projectCardIds)) {
            rootPara.put("projectCardIds", projectCardIds);
        }

        List<ProjectCardItemSimpleInfoDTO> projectCards = httpAssistantService.post(new TypeReference<List<ProjectCardItemSimpleInfoDTO>>() {
        }, url, headers, rootPara);
        return Objects.nonNull(projectCards) ? projectCards : Lists.newArrayList();
    }

    /**
     * 查询进行中的任务概要信息（任务id、任务名称等）
     *
     * @param userIds        用户id列表
     * @param backlogIds     任务卡id列表，可选
     * @param authoredUserId 当前登录用户
     * @return
     */
    @Override
    public List<BacklogItemSimpleInfoDTO> queryBackCardSimpleInfo(List<String> userIds, List<Long> backlogIds, String authoredUserId) {
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_API_TASK_CARD_PROCESSING_LIST;

        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        String userId = StringUtils.isBlank(authoredUserId) ? authoredUser.getUserId() : authoredUserId;
//        headers.add("locale", userService.getUserLangMetadataAllTenant(userId, authoredUser.getToken()));

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        if (CollectionUtils.isNotEmpty(userIds)) {
            rootPara.put("userIds", userIds);
        }
        if (CollectionUtils.isNotEmpty(backlogIds)) {
            rootPara.put("backlogIds", backlogIds);
        }

        List<BacklogItemSimpleInfoDTO> backlogCards = httpAssistantService.post(new TypeReference<List<BacklogItemSimpleInfoDTO>>() {
        }, url, headers, rootPara);
        return Objects.nonNull(backlogCards) ? backlogCards : Lists.newArrayList();
    }

    /**
     * 查询ptm项目卡列表
     */
    private List<PtmProjectCardItemDTO> queryProjectCard(List<String> userIds, List<Long> projectCardIdList, String autherUserId) {
        List<PtmProjectCardItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_PROCESSING_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        log.info("queryProjectCard-authoredUser:{}", JsonUtils.objectToString(authoredUser));
        headers.add("token", authoredUser.getToken());
        String userId = StringUtils.isBlank(autherUserId) ? authoredUser.getUserId() : autherUserId;
//        headers.add("locale", userService.getUserLangMetadataAllTenant(userId, authoredUser.getToken()));

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        if (CollectionUtils.isNotEmpty(userIds)) {
            rootPara.put("userIds", userIds);
        }
        if (CollectionUtils.isNotEmpty(projectCardIdList)) {
            rootPara.put("projectCardIds", projectCardIdList);
        }

        result = httpAssistantService.post(new TypeReference<List<PtmProjectCardItemDTO>>() {
        }, url, headers, rootPara);

        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    /**
     * 获取 项目卡 详情
     *
     * @param projectCardId
     * @return
     */
    @Override
    public PtmProjectCardDetailDTO getProjectCardDetail(Long projectCardId) {
        PtmProjectCardDetailDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_DETAIL_ID_PROJECT_CARD_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectCardId", projectCardId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectCardDetailDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectCardDetailDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取项目卡中指定任务的所有任务信息
     *
     * @param projectCardId
     * @param taskDefCode
     * @return
     */
    @Override
    public List<PtmProjectCardTaskDetailDTO> getProjectCardDetail(Long projectCardId, String taskDefCode) {
        List<PtmProjectCardTaskDetailDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_DETAIL;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectCardId", projectCardId);
        rootPara.put("taskDefCode", taskDefCode);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardTaskDetailDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardTaskDetailDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    //region 路线

    /**
     * 获取 项目卡 路线
     *
     * @param projectCardId
     * @return
     */
    @Override
    public List<PtmProjectCardRouteDTO> getProjectCardRoutes(Long projectCardId) {

        List<PtmProjectCardRouteDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_ROUTES_PROJECT_CARD_ID_PROJECT_CARD_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectCardId", projectCardId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRouteDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRouteDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    /**
     * 获取 项目 的路线列表
     *
     * @param projectId
     * @return
     */
    @Override
    public PtmProjectCardRouteDTO getProjectRoute(Long projectId) {
        PtmProjectCardRouteDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_ROUTES_ID_PROJECT_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectId", projectId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectCardRouteDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectCardRouteDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取 指定项目ID 指定任务路线 的任务数据
     *
     * @param projectIds
     * @param routeTasks
     * @return
     */
    @Override
    public List<PtmProjectCardRouteTaskDTO> getProjectCardRouteTasks(List<Long> projectIds, String routeTasks) {
        List<PtmProjectCardRouteTaskDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_ROUTE_TASKS;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectIds", projectIds);
        rootPara.put("routeTasks", routeTasks);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRouteTaskDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRouteTaskDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    /**
     * 设置默认路线
     *
     * @param defaultRouteId 要设置为默认路线的id
     * @return
     */
    @Override
    public Boolean setDefaultRoute(Long defaultRouteId) {

        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_ROUTE_SELECTED_ID_DEFAULT_ROUTE_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("defaultRouteId", defaultRouteId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    //endregion

    /**
     * 查询ptm任务卡列表
     *
     * @param userIds 要查询的userId
     * @return 没有值则返回空 List
     */
    @Override
    public List<PtmTaskCardItemDTO> queryTaskCardList(List<String> userIds) {
        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("userIds", userIds);
        return reqPtmTaskCard(rootPara,PtmApiConstant.TASK_CARD_PROCESSING_LIST);
    }

    private List<PtmTaskCardItemDTO> reqPtmTaskCard(Map<String, Object> params,String reqUrl){
        List<PtmTaskCardItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + reqUrl;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());



        result = httpAssistantService.post(new TypeReference<List<PtmTaskCardItemDTO>>() {
        }, url, headers, params);

        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    @Override
    public List<PtmTaskCardItemDTO> queryTaskCardListByCodes(List<String> codes) {
        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("codes", codes);
        return reqPtmTaskCard(rootPara,PtmApiConstant.TASK_CARD_PROCESSING_SIMPLIFY_LIST);
    }

    /**
     * 查询ptm任务卡列表
     *
     * @param backlogIdList 要查询的待办ID列表
     * @return 没有值则返回空 List
     */
    @Override
    public List<PtmTaskCardItemDTO> queryTaskCardListByBacklogId(List<Long> backlogIdList) {
        List<PtmTaskCardItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_PROCESSING_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("backlogIds", backlogIdList);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskCardItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskCardItemDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    /**
     * 根据项目任务信息获取进行中的任务列表
     *
     * @param projectId   ptm项目id
     * @param taskDefCode km任务code
     * @return data取的是ptm_task.data
     */
    @Override
    public List<PtmBacklogItemDTO> getBacklogByProjectIdAndTaskCode(Long projectId, String taskDefCode) {
        List<PtmBacklogItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_TASK_BY_CONDITION_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectId", projectId);
        rootPara.put("taskDefCode", taskDefCode);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogItemDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据项目卡任务信息获取进行中的任务列表
     *
     * @param projectCardId ptm项目卡id
     * @param taskDefCode   km任务code
     * @return data取的是ptm_task.data
     */
    @Override
    public List<PtmBacklogItemDTO> getBacklogByProjectCardIdAndTaskCode(Long projectCardId, String taskDefCode) {
        List<PtmBacklogItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_TASK_BY_CONDITION_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectCardId", projectCardId);
        rootPara.put("taskDefCode", taskDefCode);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogItemDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }


    /**
     * 根据backlogId获取全部的任务列表
     *
     * @param backlogId 待办id
     * @return
     */
    @Override
    @Cacheable(cacheManager = "requestScopedCacheManager",
            key = "'_' + #backlogId",
            value = "getBacklogByBacklogId")
    public List<PtmBacklogItemDTO> getBacklogByBacklogId(Long backlogId) {
        List<PtmBacklogItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_TASK_LIST_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogItemDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据backlogId获取全部的任务列表
     *
     * @param backlogIdList 待办id
     * @return
     */
    @Override
    public List<PtmBacklogItemDTO> getBacklogByBacklogIds(List<Long> backlogIdList) {
        List<PtmBacklogItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_TASK_LIST_BATCH;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogIds", backlogIdList);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogItemDTO>>>() {
        }, param);

        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }


    //region 转派

    /**
     * 转派 项目卡
     *
     * @param requestDTO
     * @return
     */
    @Override
    public Boolean reassignProjectCard(PtmProjectCardReassignRequestDTO requestDTO) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_REASSIGN;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectCardId", requestDTO.getProjectCardId());
        rootPara.put("newPersonInCharge", requestDTO.getNewPersonInCharge());
        rootPara.put("newPersonInChargeName", requestDTO.getNewPersonInChargeName());
        rootPara.put("comment", requestDTO.getComment());
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 转派 待办
     *
     * @param requestDTO
     * @return
     */
    @Override
    public Boolean reassignBacklog(PtmBacklogReassignRequestDTO requestDTO) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_REASSIGN;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("backlogId", requestDTO.getBacklogId());
        rootPara.put("newPerformerId", requestDTO.getNewPerformerId());
        rootPara.put("newPerformerName", requestDTO.getNewPerformerName());
        rootPara.put("comment", requestDTO.getComment());

        log.info("reassignBacklog，data：{}", JsonUtils.objectToString(rootPara));

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 记录详情

    /**
     * 获取项目卡记录详情（ptm_project_card表）
     *
     * @param projectCardId
     * @return
     */
    @Override
    public PtmProjectCardRecordDTO getProjectCardRecord(Long projectCardId) {
        PtmProjectCardRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_CARD_ID_PROJECT_CARD_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectCardId", projectCardId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectCardRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectCardRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取项目卡 记录详情（ptm_project_card表）
     *
     * @param projectCardIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmProjectCardRecordDTO> getProjectCardRecordByProjectCardIds(List<Long> projectCardIds) {
        List<PtmProjectCardRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_CARD_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("idList", projectCardIds);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRecordDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }


    /**
     * 批量 获取项目卡 记录详情（ptm_project_card表）
     *
     * @param condition 传入任何条件
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmProjectCardRecordDTO> getProjectCardByCondition(Map condition) {
        List<PtmProjectCardRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_DATA_SUB_PROJECT_CARD_LIST_BY_CONDITION;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(condition, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRecordDTO>>>() {
        }, condition);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量-获取指定项目卡中最近一次项目记录<br>
     * 不含data数据
     *
     * @param projectCardIds
     * @return
     */
    @Override
    public List<PtmProjectRecordDTO> getLastProjectRecordByProjectCardIds(List<Long> projectCardIds) {
        List<PtmProjectRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_PROCESSING_LATEST_PROJECT_BATCH;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectCardIds", projectCardIds);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectRecordDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取项目记录详情(ptm_project表)
     *
     * @param projectId
     * @return
     */
    @Override
    @Cacheable(cacheManager = "requestScopedCacheManager",
            key = "'_' + #projectId",
            value = "getProjectRecord")
    public PtmProjectRecordDTO getProjectRecord(Long projectId) {
        PtmProjectRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_ID_PROJECT_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectId", projectId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量-获取项目 记录详情(ptm_project表)
     * 警告：使用次方法前请确认是否需要项目列表中的data，如果不需要，请使用getProjectRecordByProjectIdsV2替代，需要data会消耗ptm较多内存
     *
     * @param projectIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmProjectRecordDTO> getProjectRecordByProjectIds(List<Long> projectIds) {
        return getProjectRecordByProjectIdsV2(projectIds, true);
    }

    /**
     * 批量-获取项目 记录详情(ptm_project表)
     *
     * @param projectIds
     * @param needData   是否需要data数据，默认为需要
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmProjectRecordDTO> getProjectRecordByProjectIdsV2(List<Long> projectIds, boolean needData) {
        List<PtmProjectRecordDTO> result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("idList", projectIds);
        //如果不需要data，传needData为false，目的优化ptm内存消耗
        //needData不传或者传true，ptm都是默认查data
        rootPara.put("needData", needData);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectRecordDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取任务记录详情(ptm_task表)
     *
     * @param taskId
     * @return
     */
    @Override
    @Cacheable(cacheManager = "requestScopedCacheManager",
            key = "'_' + #taskId",
            value = "getTaskRecord")
    public PtmTaskRecordDTO getTaskRecord(Long taskId) {
        PtmTaskRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ID_TASK_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("taskId", taskId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmTaskRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmTaskRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取任务记录详情(ptm_task表)
     *
     * @param taskIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmTaskRecordDTO> getTaskRecordByTaskIds(List<Long> taskIds) {
        List<PtmTaskRecordDTO> result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("idList", taskIds);
        //不需要data，传needData为false，目的优化ptm内存消耗
        //needData不传或者传true，ptm都是默认查data，atmc中查此接口的业务均不需要data
        rootPara.put("needData", false);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskRecordDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取任务记录详情(ptm_task表)
     *
     * @param taskIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmTaskRecordDTO> getTaskRecordByTaskIds(List<Long> taskIds, boolean needData) {
        List<PtmTaskRecordDTO> result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("idList", taskIds);
        //不需要data，传needData为false，目的优化ptm内存消耗
        //needData不传或者传true，ptm都是默认查data，atmc中查此接口的业务均不需要data
        rootPara.put("needData", needData);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskRecordDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取活动记录详情(ptm_activity表)
     *
     * @param activityId
     * @return
     */
    @Override
    public PtmActivityRecordDTO getActivityRecord(Long activityId) {
        PtmActivityRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_ID_ACTIVITY_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("activityId", activityId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmActivityRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmActivityRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取活动记录详情(ptm_activity表)
     *
     * @param activityIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmActivityRecordDTO> getActivityRecordByActivityIds(List<Long> activityIds) {
        List<PtmActivityRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("idList", activityIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmActivityRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmActivityRecordDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取事项的记录详情(ptm_task_activity_item 表记录)
     *
     * @param workItemId
     * @return
     */
    @Override
    @Cacheable(cacheManager = "requestScopedCacheManager",
            key = "'_' + #workItemId",
            value = "getWorkItemRecord")
    public PtmWorkItemRecordDTO getWorkItemRecord(Long workItemId) {
        PtmWorkItemRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_ITEM_ID_WORK_ITEM_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("workItemId", workItemId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmWorkItemRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmWorkItemRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取事项 的记录详情(ptm_task_activity_item 表记录)
     *
     * @param workItemIds
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmWorkItemRecordDTO> getWorkItemRecordByWorkItemIds(List<Long> workItemIds) {
        List<PtmWorkItemRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_ITEM_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("idList", workItemIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmWorkItemRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmWorkItemRecordDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取指定工作项 被发起退回重签或重办的 工作项，可能为null
     *
     * @param type             0 或 null，同一个任务，指签核任务内；1-不是同一个任务
     * @param workItemId
     * @param approvalTaskCode 要查询的签核任务code
     * @return 可能为null
     */
    @Override
    public PtmWorkItemRecordDTO getReExecuteWorkItem(Integer type, Long workItemId, String approvalTaskCode) {
        PtmWorkItemRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_PREVIOUS_ITEM;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("id", workItemId);
        rootPara.put("type", type);
        if (StringUtils.isNotBlank(approvalTaskCode)) {
            rootPara.put("taskDefCode", approvalTaskCode);
        }
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmWorkItemRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmWorkItemRecordDTO>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取待办的记录详情(ptm_backlog 表记录)
     *
     * @param backlogId
     * @return
     */
    @Override
    @Cacheable(cacheManager = "requestScopedCacheManager",
            key = "'_' + #backlogId",
            value = "getBacklogRecord")
    public PtmBacklogRecordDTO getBacklogRecord(Long backlogId) {
        PtmBacklogRecordDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_ITEM_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmBacklogRecordDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmBacklogRecordDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量 获取待办的记录详情(ptm_backlog 表记录)
     *
     * @param backlogIds
     * @return
     */
    @Override
    public List<PtmBacklogRecordDTO> getBacklogRecordByBacklogIds(List<Long> backlogIds) {
        List<PtmBacklogRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_ACTIVITY_ITEM_BACKLOG_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("idList", backlogIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogRecordDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogRecordDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region data、stateData

    /**
     * 根据待办id获取任务卡的data、stateData
     *
     * @param backlogId ptm_backlog.id
     * @param type      非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return data 取的是ptm.activity 的 data
     */
    @Override
    public List<PtmTaskCardDataDTO> getTaskDataByBacklogId(Long backlogId, String type) {
        return getTaskData(backlogId, null, null, type);
    }

    public List<PtmTaskCardDataDTO> getTaskData(Long backlogId, Long activityId, Long taskId, String type) {
        List<PtmTaskCardDataDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_DATA;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        if (Objects.nonNull(backlogId)) {
            rootPara.put("backlogId", backlogId);
        }
        if (Objects.nonNull(activityId)) {
            rootPara.put("activityId", activityId);
        }
        if (Objects.nonNull(taskId)) {
            rootPara.put("taskId", taskId);
        }
        rootPara.put("type", type);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskCardDataDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskCardDataDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    /**
     * 根据 ptm_task_activity.id 获取任务卡的data、stateData
     *
     * @param activityId ptm_task_activity.id
     * @param type       非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return
     */
    @Override
    public PtmTaskCardDataDTO getTaskDataByActivityId(Long activityId, String type) {
        List<PtmTaskCardDataDTO> result = getTaskData(null, activityId, null, type);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return null;
    }

    /**
     * 根据 ptm_task.id 获取任务卡的data、stateData
     *
     * @param taskId ptm_task.id
     * @param type   非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return
     */
    @Override
    public PtmTaskCardDataDTO getTaskDataByTaskId(Long taskId, String type) {
        List<PtmTaskCardDataDTO> result = getTaskData(null, null, taskId, type);
        if (CollectionUtils.isNotEmpty(result)) {
            return result.get(0);
        }
        return null;
    }

    /**
     * 更新 stateData 数据
     *
     * @param activityId ptm_task_activity.id
     * @param stateData
     * @return
     */
    @Override
    public Boolean updateStateData(Long activityId, Object stateData) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_DATA_STATE_UPDATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("activityId", activityId);
        rootPara.put("dataState", stateData);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 更新 data 数据
     *
     * @param activityId ptm_task_activity.id
     * @param data
     * @return
     */
    @Override
    public Boolean updateData(Long activityId, Object data) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_DATA_UPDATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("activityId", activityId);
        rootPara.put("data", data);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 数据终止-更新 bpmData
     *
     * @param data
     */
    @Override
    public void resetBpmDataForTerminateTaskData(Map data) {
        resetBpmData("TerminateTaskData", data);
    }

    /**
     * 数据转派-更新 bpmData
     *
     * @param data
     */
    @Override
    public void resetBpmDataForReassignTaskData(Map data) {
        resetBpmData("ReassignTaskData", data);
    }

    /**
     * 数据转派-更新 bpmData
     *
     * @param data
     */
    private void resetBpmData(String action, Map data) {

        //请求地址
        String url = envProperties.getPtmUri() + AtmcStrUtil.format(PtmApiConstant.BUSINESS_EXECUTE_ACTION, action);
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity<?> httpEntity = new HttpEntity<>(data, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Object>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Object>>() {
        }, data);
        responseEntity.getBody().getResponseWithException("");

    }

    //endregion

    //region 最近更新时间

    /**
     * 获取指人员的项目卡的最后一次更新时间
     *
     * @param userId
     * @return
     */
    @Override
    public LocalDateTime getProjectCardLastModifyTime(String userId) {
        LocalDateTime result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_PROCESSING_LATEST_TIME_USER_ID_USER_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<LocalDateTime>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<LocalDateTime>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取指定人员的待办的最后一次更新时间
     *
     * @param userId
     * @return
     */
    @Override
    public LocalDateTime getBacklogLastModifyTime(String userId) {
        LocalDateTime result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_PROCESSING_LATEST_TIME_USER_ID_USER_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<LocalDateTime>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<LocalDateTime>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 创建待办

    /**
     * 多元交互-创建外部厂商个人租户待办
     *
     * @param emailRequest
     * @return
     */
    public Boolean addCrossTenantBacklog(PtmEmailBacklogRequestDTO emailRequest) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_CROSS_TENANT_ADD;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("workItemId", emailRequest.getWorkItemId());
        rootPara.put("performerId", emailRequest.getPerformerId());
        rootPara.put("performerName", emailRequest.getPerformerName());
        rootPara.put("tenantId", emailRequest.getTenantId());
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 创建辅助执行者-待办
     *
     * @param assistRequest
     * @return
     */
    @Override
    public Boolean addAssistBacklog(PtmAssistBacklogRequestDTO assistRequest) {

        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_ASSIST_ADD;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("workItemId", assistRequest.getWorkItemId());
        rootPara.put("performerId", assistRequest.getPerformerId());
        rootPara.put("performerName", assistRequest.getPerformerName());
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    //endregion

    //region 更新项目、任务信息

    /**
     * 批量更新项目基本信息
     *
     * @param param
     * @return
     */
    @Override
    public Boolean batchUpdateProjectInfo(Map param) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_UPDATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, param);

        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 批量更新任务基本信息
     *
     * @param param
     * @return
     */
    @Override
    public Boolean batchUpdateBacklogInfo(Map param) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_UPDATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, param);

        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 项目

    /**
     * 根据流程序号，关闭其所在的主项目。提前结束项目
     *
     * @param map
     * @return
     */
    @Override
    public Boolean closeProjectCard(Map<String, Object> map) {

        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_CLOSE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.putAll(map);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    //endregion

    //region 任务、待办

    /**
     * 批量终止任务
     *
     * @param param
     * @return
     */
    @Override
    public Boolean batchTerminateTask(Map param) {
        Boolean result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_TERMINATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmTerminateTaskResultDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmTerminateTaskResultDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("").getSuccess();
        return result;
    }

    /**
     * 批量根据ptm_task.id查询所属的待办部分信息
     *
     * @param taskIds
     * @return
     */
    public List<PtmBacklogPartialDTO> queryBacklogPartialByTaskIds(List<Long> taskIds) {
        List<PtmBacklogPartialDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_BACKLOG_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map param = new HashMap();
        param.put("taskIds", taskIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogPartialDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogPartialDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 报表相关

    /**
     * PTM 项目型报表
     *
     * @param param
     * @return
     */
    @Override
    public List<PtmProjectReportItemDTO> queryProjectReport(Map<String, Object> param) {
        List<PtmProjectReportItemDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_ITEM_REPORT;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectReportItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectReportItemDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 查询项目下任务的业务数据
     *
     * @param param
     * @return
     */
    @Override
    public List<PtmTaskReportItemDTO> queryProjectTaskReport(Map<String, Object> param) {
        List<PtmTaskReportItemDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_PROJECT_REPORT;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskReportItemDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskReportItemDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 异常排除

    /**
     * 根据异常排除待办id获取新发起项目的 所有的任务列表
     *
     * @param originalBacklogIds
     * @return
     */
    @Override
    public List<Map> getSolveTaskList(List<Long> originalBacklogIds) {
        List<Map> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_SOLVE_TASK_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("originalBacklogIds", originalBacklogIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    //region 知会相关

    /**
     * 更新签核通知信息
     *
     * @param activityId
     * @param informer
     */
    @Override
    public void updateSignInformNotice(Long activityId, Map informer) {

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_APPROVE_INFORMER_UPDATE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map param = new HashMap();
        param.put("activityId", activityId);
        param.put("informer", informer);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<Boolean>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, param);
        responseEntity.getBody().getResponseWithException("");
    }

    /**
     * 获取 知会通知信息
     *
     * @param bpmActivityUids
     * @return
     */
    @Override
    public List<PtmActivitySignInformerDTO> getSignInformer(List<String> bpmActivityUids) {
        List<PtmActivitySignInformerDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_APPROVE_INFORMER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map param = new HashMap();
        param.put("bpmActivityUids", bpmActivityUids);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmActivitySignInformerDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmActivitySignInformerDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    //endregion

    /**
     * 获取待办的签核历程
     *
     * @param backlogId
     * @return 没有内容返回空集合
     */
    @Override
    public List<PtmBacklogApprovalDTO> queryApproval(Long backlogId) {
        List<PtmBacklogApprovalDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_SIGN_PROCESS_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogApprovalDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogApprovalDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new ArrayList<>() : result;
    }


    /**
     * 获取待办的签核历程
     *
     * @param backlogId
     * @return 没有内容返回空集合
     */
    @Override
    public List<PtmBacklogApprovalDTO> queryApprovalNew(Long backlogId) {
        List<PtmBacklogApprovalDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_BACKLOG_SIGN_PROCESS_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogApprovalDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogApprovalDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new ArrayList<>() : result;
    }

    @Override
    public List<PtmBacklogApprovalDTO> queryApprovalByWorkItemId(Long workItemId) {

        List<PtmBacklogApprovalDTO> result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORK_ITEM_SIGN_PROCESS_WORK_ITEM_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("workItemId", workItemId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogApprovalDTO>>> responseEntity;
        try {
            responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogApprovalDTO>>>() {
            }, param);
        } catch (Exception e) {
            log.error("[atmc访问PTM获取 签核历程] error:{}, stack:{}, workItemId:{}", e, e.getStackTrace(), workItemId);
            String message = MessageFormat.format(messageUtils.getMessage("exception.ptm.find.sign.process.by.workitemid"), workItemId);
            throw BusinessException.create(599, "599", message, e.getMessage(), messageUtils.getMessage("exception.ptm.find.sign.process.by.workitemid"), e);
        }

        if (responseEntity.getBody().isOK() && responseEntity.getBody().getResponse() != null) {
            result = responseEntity.getBody().getResponse();
        } else {
            result = new ArrayList<>();
        }
        return result;
    }

    @Override
    public List<PtmBacklogApprovalDTO> queryApprovalByData(ApprovalWorkItemByDataDTO workItemDataDTO) {
        List<PtmBacklogApprovalDTO> result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORK_ITEM_SIGN_PROCESS_DATA;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity<?> httpEntity = new HttpEntity<>(workItemDataDTO, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogApprovalDTO>>> responseEntity;
        try {
            responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogApprovalDTO>>>() {
            }, workItemDataDTO);
        } catch (Exception e) {
            log.error("[atmc访问PTM获取人工任务签核历程] error:{}, stack:{}, workItemDataDTO:{}", e, e.getStackTrace(), JsonUtils.objectToString(workItemDataDTO));
            String message = MessageFormat.format(messageUtils.getMessage("exception.ptm.find.sign.process.by.data"), JsonUtils.objectToString(workItemDataDTO));
            throw BusinessException.create(599, "599", message, e.getMessage(), messageUtils.getMessage("exception.ptm.find.sign.process.by.data"), e);
        }

        if (responseEntity.getBody().isOK() && responseEntity.getBody().getResponse() != null) {
            result = responseEntity.getBody().getResponse();
        } else {
            result = new ArrayList<>();
        }
        return result;
    }

    /**
     * 根据项目id和任务code来获取发起的子项目
     *
     * @param projectId
     * @param taskCodes
     * @return
     */
    @Override
    public List<PtmSubProjectResponseDTO> getSubProject(Long projectId, List<String> taskCodes) {
        List<PtmSubProjectResponseDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.BUSINESS_EXECUTE_QUERY_SUB_PROJECT_BY_PROJECT_CARD_ID_TASK_CODE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> p = new HashMap<>();
        p.put("projectCardId", projectId);
        p.put("taskCodes", taskCodes);
        Map<String, Object> param = new HashMap<>();
        param.put("data", p);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmSubProjectResponseDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmSubProjectResponseDTO>>>() {
        }, new HashMap<>());
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new ArrayList<>() : result;
    }

    /**
     * 获取已完成的代理任务id
     *
     * @Author：SYQ
     * @Date：2022/6/29 13:45
     */
    @Override
    public List<Long> getCompleteAgentBacklog(String agentPerformerId) {
        List<Long> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_AGENT_BACKLOG_AGENT_PERFORM_ID_AGENT_PERFORM_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("agentPerformId", agentPerformerId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<Long>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Long>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result == null ? new ArrayList<>() : result;
    }

    /**
     * 获取项目卡的可视化面板数据
     *
     * @param projectCardId
     * @return
     */
    @Override
    public PtmProjectCardPanelDTO getProjectCardPanel(Long projectCardId) {
        PtmProjectCardPanelDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_PANEL_ID_PROJECT_CARD_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectCardId", projectCardId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectCardPanelDTO>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectCardPanelDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 是否允许撤回
     *
     * @param allowRetrieveReqDTO
     * @return 返回码：10:允许撤回 20:不支持撤回 30:已撤回 40:任务不存在
     */
    @Override
    public void checkTaskSupportWithdraw(PtmCheckSupportWithdrawReqDTO allowRetrieveReqDTO) {
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_IS_TASK_WITHDRAW;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity<?> httpEntity = new HttpEntity<>(allowRetrieveReqDTO, headers);
        try {
            ResponseEntity<BaseResultDTO<Integer>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Integer>>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && null != respEntity.getBody() && respEntity.getBody().isOK()) {

            } else {
                log.error("[ptm-task-withdraw-{}] atmc访问PTM判断任务是否支持撤回出现异常, data: {}, 响应内容: {}",
                        allowRetrieveReqDTO.getBacklogId(),
                        JsonUtils.objectToString(allowRetrieveReqDTO),
                        JsonUtils.objectToString(respEntity.getBody()));

                String message = messageUtils.getMessage("exception.ptm.is.allow.withdraw");
                BusinessException ex = BusinessException.create(respEntity.getStatusCodeValue(), ErrorCodeEnum.CHECK_TASK_WITHDRAW_ERROR.getErrCode(), message);
                ex.setDescription(JsonUtils.objectToString(respEntity.getBody()));
                throw ex;
            }
        } catch (HttpServerErrorException ex) {
            // 500
            if (HttpStatus.INTERNAL_SERVER_ERROR.value() == ex.getStatusCode().value()) {
                // 尝试反序列化
                BaseResultDTO baseResultDTO = null;
                try {
                    baseResultDTO = JsonUtils.jsonToObject(ex.getResponseBodyAsString(), new TypeReference<BaseResultDTO>() {
                    });
                } catch (Exception exception) {
                    baseResultDTO = null;
                }

                if (null != baseResultDTO) {
                    // PTM返回的errorType为Application，会导致正式区错误提示框展示为“友好的、通用的”提示信息。
                    // 这边显式将其修改为Business
                    throw BusinessException.create(baseResultDTO.getStatus(),
                            baseResultDTO.getErrorCode(),
                            baseResultDTO.getErrorMessage(),
                            baseResultDTO.getDescription(),
                            baseResultDTO.getStatusDescription(),
                            ErrorTypeEnum.BUSINESS.getValue(),
                            ex);
                }
            }

            // 非500 && 非正常的PTM异常业务响应
            throw ex;
        }
    }

    /**
     * 是否允许跨BK撤回
     *
     * @param allowRetrieveReqDTOList
     * @return 本次撤回可能涉及到的taskCode
     */
    @Override
    public PtmCheckSupportCrossBkWithdrawRespDTO checkTaskSupportCrossBkWithdraw(List<PtmCheckSupportCrossBkWithdrawReqDTO> allowRetrieveReqDTOList) {
        PtmCheckSupportCrossBkWithdrawRespDTO result = new PtmCheckSupportCrossBkWithdrawRespDTO();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_IS_TASK_WITHDRAW_CROSS_BK;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map paramMap = new HashMap();
        paramMap.put("params", allowRetrieveReqDTOList);

        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO<PtmCheckSupportCrossBkWithdrawRespDTO>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmCheckSupportCrossBkWithdrawRespDTO>>(){});
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && null != respEntity.getBody() && respEntity.getBody().isOK()) {
                result = respEntity.getBody().getResponseWithException("");
                return result;
            } else {
                log.error("[ptm-task-crossbk-withdraw] atmc访问PTM判断任务是否支持跨BK撤回出现异常, data: {}, 响应内容: {}",
                        JsonUtils.objectToString(allowRetrieveReqDTOList),
                        JsonUtils.objectToString(respEntity.getBody()));

                String message = messageUtils.getMessage("exception.ptm.is.allow.withdraw");
                BusinessException ex = BusinessException.create(respEntity.getStatusCodeValue(), ErrorCodeEnum.CHECK_TASK_WITHDRAW_ERROR.getErrCode(), message);
                ex.setDescription(JsonUtils.objectToString(respEntity.getBody()));
                throw ex;
            }
        } catch (HttpServerErrorException ex) {
            // 500
            if (HttpStatus.INTERNAL_SERVER_ERROR.value() == ex.getStatusCode().value()) {
                // 尝试反序列化
                BaseResultDTO baseResultDTO = null;
                try {
                    baseResultDTO = JsonUtils.jsonToObject(ex.getResponseBodyAsString(), new TypeReference<BaseResultDTO>() {
                    });
                } catch (Exception exception) {
                    baseResultDTO = null;
                }

                if (null != baseResultDTO) {
                    // PTM返回的errorType为Application，会导致正式区错误提示框展示为“友好的、通用的”提示信息。
                    // 这边显式将其修改为Business
                    throw BusinessException.create(baseResultDTO.getStatus(),
                            baseResultDTO.getErrorCode(),
                            baseResultDTO.getErrorMessage(),
                            baseResultDTO.getDescription(),
                            baseResultDTO.getStatusDescription(),
                            ErrorTypeEnum.BUSINESS.getValue(),
                            ex);
                }
            }

            // 非500 && 非正常的PTM异常业务响应
            throw ex;
        }
    }

    /**
     * 批量 获取项目卡 记录详情（ptm_project_card表）
     *
     * @param condition 传入任何条件
     * @return 没有值慢返回空集合
     */
    @Override
    public List<PtmProjectCardRecordDTO> getProjectDataByCondition(Map condition) {
        List<PtmProjectCardRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_LIST_BY_CONDITION;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(condition, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRecordDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRecordDTO>>>() {
        }, condition);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 通过流程序号获取项目数据
     *
     * @param processSerialNumber
     * @return
     */
    @Override
    public List<PtmProjectCardRecordDTO> getProjectDataByProcessSerialNumber(String processSerialNumber) {
        List<PtmProjectCardRecordDTO> result = new ArrayList<>();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_PROJECT_LIST_BY_CONDITION;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> condition = Maps.newHashMap();
        condition.put("process_serial_number", processSerialNumber);
        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(condition, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmProjectCardRecordDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardRecordDTO>>>() {
        }, condition);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }


    /**
     * 根据项目卡Id查询所有待办
     *
     * @param projectCardId
     * @return
     */
    @Override
    public List<PtmBacklogForProjectCardDTO> queryBacklogListForProjectCardId(Long projectCardId) {
        List<PtmBacklogForProjectCardDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_TASK_BY_PROJECT_CARD_PROJECT_CARD_ID_PROJECT_CARD_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("projectCardId", projectCardId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmBacklogForProjectCardDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmBacklogForProjectCardDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取团队项目分组（用于移动端）
     *
     * @Author：SYQ
     * @Date：2022/12/29 17:37
     */
    @Override
    public List<MobileProjectGroupDTO> getMobileDepartProjectGroup(List<String> userIds, Integer type) {
        List<MobileProjectGroupDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_PROJECT_CARD_GROUP;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("userIds", userIds);
        param.put("type", type);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<MobileProjectGroupDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<MobileProjectGroupDTO>>>() {
        });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取团队任务分组（用于移动端）
     *
     * @Author：SYQ
     * @Date：2022/12/30 09:42
     */
    @Override
    public List<MobileBacklogGroupDTO> getMobileDepartBacklogGroup(List<String> userIds, Integer type) {
        List<MobileBacklogGroupDTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_TASK_CARD_GROUP;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("userIds", userIds);
        param.put("type", type);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<MobileBacklogGroupDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<MobileBacklogGroupDTO>>>() {
        });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据分组查询项目卡列表（用于移动端）
     *
     * @Author：SYQ
     * @Date：2023/1/4 15:03
     */
    @Override
    public List<PtmProjectCardItemDTO> findMobileProjectCardGroupList(List<String> userIds, String projectDefCode) {
        List<PtmProjectCardItemDTO> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_PROJECT_CARD_GROUP_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        // 防止存在代理token的情况
        // appcore中会用proxyToken代替token，此时拿到的列表不是当前登录用户本身的项目列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());

        //入参
        Map<String, Object> paramMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(userIds)) {
            paramMap.put("userIds", userIds);
        }
        if (StringUtils.isNotBlank(projectDefCode)) {
            paramMap.put("projectDefCode", projectDefCode);
        }
        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);

        ResponseEntity<BaseResultDTO<List<PtmProjectCardItemDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<PtmProjectCardItemDTO>>>() {
                });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据分组查询任务卡列表（用于移动端）
     *
     * @Author：SYQ
     * @Date：2023/1/5 13:50
     */
    @Override
    public List<PtmTaskCardItemDTO> findMobileBacklogCardGroupList(List<String> userIds, String taskDefCode) {
        List<PtmTaskCardItemDTO> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_TASK_CARD_GROUP_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        // 防止存在代理token的情况
        // appcore中会用proxyToken代替token，此时拿到的列表不是当前登录用户本身的项目列表
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());

        //入参
        Map<String, Object> paramMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(userIds)) {
            paramMap.put("userIds", userIds);
        }
        if (StringUtils.isNotBlank(taskDefCode)) {
            paramMap.put("taskDefCode", taskDefCode);
        }

        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);

        ResponseEntity<BaseResultDTO<List<PtmTaskCardItemDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskCardItemDTO>>>() {
                });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据人员及查询关键字查询符合条件进行中的任务
     *
     * @Author：SYQ
     * @Date：2023/1/9 14:55
     */
    @Override
    public List<MobilePtmBacklogDTO> searchBacklogByParam(String queryParam, List<String> userIds) {
        List<MobilePtmBacklogDTO> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_PROCESSING_TASK_BY_CONDITION;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("userIds", userIds);
        paramMap.put("queryParam", queryParam);

        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);

        ResponseEntity<BaseResultDTO<List<MobilePtmBacklogDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<MobilePtmBacklogDTO>>>() {
                });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据人员及任务id查询进行中的任务信息
     *
     * @Author：SYQ
     * @Date：2023/1/9 15:15
     */
    @Override
    public List<MobilePtmBacklogDTO> searchBacklogByIdsAndUserIds(List<String> userIds, List<Long> backlogIds) {
        List<MobilePtmBacklogDTO> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_MOBILE_PROCESSING_TASK_BY_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("userIds", userIds);
        paramMap.put("backlogIds", backlogIds);

        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);

        ResponseEntity<BaseResultDTO<List<MobilePtmBacklogDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<MobilePtmBacklogDTO>>>() {
                });
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据任务信息获取执行人
     *
     * @param backlogIds
     * @return
     */
    @Override
    public List<PtmAgentPerformerDTO> queryAgentPerformerList(List<Long> backlogIds) {
        List<PtmAgentPerformerDTO> result = Lists.newArrayList();

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORKITEM_PERFORMER_INFO_BY_BACKLOG_IDS;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        headers.add("routerKey", authoredUser.getTenantId());
//        headers.add("locale", userService.getUserLangMetadataAllTenant(authoredUser.getUserId(), authoredUser.getToken()));

        //入参
        Map<String, Object> param = Maps.newHashMap();
        param.put("backlogIds", backlogIds);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmAgentPerformerDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmAgentPerformerDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据活动id和任务id获取bpmdata
     *
     * @param activityId
     * @param taskId
     * @return
     */
    @Override
    public Map<String, Object> getBpmDataByActivityIdAndTaskId(Long activityId, Long taskId) {
        Map<String, Object> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.DATA_TASK_VARIABLES;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("taskId", taskId);
        rootPara.put("activityId", activityId);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map<String, Object>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map<String, Object>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取项目下异常任务信息
     *
     * @param projectId 项目ID
     * @param taskCode  任务编码
     * @return
     */
    @Override
    public List<TaskExceptionDTO> getExceptionTaskInfo(long projectId, String taskCode) {
        List<TaskExceptionDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_TASK_EXCEPTION_ACTIVITY_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        headers.add("routerKey", authoredUser.getTenantId());
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        //入参
        Map<String, Object> param = Maps.newHashMap();
        param.put("projectCardId", projectId);
        param.put("taskDefCode", taskCode);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        //请求接口
        ResponseEntity<BaseResultDTO<List<TaskExceptionDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<TaskExceptionDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 交接任务
     *
     * @param newPerformerId   交接人
     * @param newPerformerName 交接姓名
     * @param backlogIds       任务ID
     * @return
     */
    @Override
    public Map<String, Object> handoverTask(String newPerformerId, String newPerformerName, List<Long> backlogIds) {
        Map<String, Object> result = new HashMap<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_TASK_HANDOVER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("backlogIds", backlogIds);
        rootPara.put("newPerformerId", newPerformerId);
        rootPara.put("newPerformerName", newPerformerName);

        log.info("reassignBacklog，data：{}", JsonUtils.objectToString(rootPara));

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }


    /**
     * 交接任务
     *
     * @param newPerformerId   交接人
     * @param newPerformerName 交接姓名
     * @param projectCardIds   项目卡id
     * @return
     */
    @Override
    public Map<String, Object> handoverProject(String newPerformerId, String newPerformerName, List<Long> projectCardIds) {
        Map<String, Object> result;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_PROJECT_CARD_HANDOVER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("projectCardIds", projectCardIds);
        rootPara.put("replacerUserId", newPerformerId);
        rootPara.put("replacerUserName", newPerformerName);

        log.info("reassignBacklog，data：{}", JsonUtils.objectToString(rootPara));

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据用户id列表， 起始的待办的计划完成时间 和 截止的待办的计划完成时间 查询待办项列表
     *
     * @param userIds   用户id列表
     * @param beginDate 起始的待办的计划完成时间，时间格式：yyyy-MM-dd
     * @param endDate   截止的待办的计划完成时间，时间格式：yyyy-MM-dd
     * @return
     */
    @Override
    public List<PtmWorkItemV2DTO> queryUserWorkItemListByTimeRange(Set<String> userIds, String beginDate, String endDate) {
        List<PtmWorkItemV2DTO> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORKITEM_LIST_BY_PERFORMER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        headers.add("token", authoredUser.getToken());
        headers.add("routerKey", authoredUser.getTenantId());
        headers.add("locale", LocaleContextHolder.getLocale().toString());

        Map<String, Object> reqBody = new HashMap<>();
        reqBody.put("userIds", userIds);
        reqBody.put("beginDate", beginDate);
        reqBody.put("endDate", endDate);
        //入参
        HttpEntity<?> httpEntity = new HttpEntity<>(reqBody, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmWorkItemV2DTO>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<PtmWorkItemV2DTO>>>() {
                });
        result = respEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 判断任务是否存在转派行为（因转派产生/因转派关闭）
     *
     * @param backlogId 待办id
     * @return
     */
    @Override
    public Boolean reassignOrNot(Long backlogId) {
        Boolean result = false;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORKITEM_REASSIGN_OR_NOT_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取转派历程
     *
     * @param backlogId 待办id
     * @return
     */
    @Override
    public List<PtmRessignTrace> getReassignTrace(Long backlogId) {
        List<PtmRessignTrace> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORKITEM_REASSIGN_TRACE_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmRessignTrace>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmRessignTrace>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据异常排除任务id查询发起的签核任务
     *
     * @param backlogId
     * @return
     */
    @Override
    public List<PtmSolveTask> getSolveTaskListV2(Long backlogId) {
        List<PtmSolveTask> result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_BACKLOG_SOLVE_TASK_LIST_BACKLOG_ID_BACKLOG_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("backlogId", backlogId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmSolveTask>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmSolveTask>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 查询ptm任务卡列表
     *
     * @param userIds 要查询的userId
     * @return 没有值则返回空 List
     */
    @Override
    public List<PtmTaskCardItemDTO> queryAllTaskCardList(List<String> userIds, String startPlanEndTime, String endPlanEndTime, List<Long> backlogIds, Boolean completed) {
        List<PtmTaskCardItemDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_BACKLOG_LIST;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //获取列表时，传当前token，如果使用代理token，获取的就是代理token的列表
        headers.add("token", AppAuthContextHolder.getContext().getAuthoredUser().getToken());

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("userIds", userIds);
        rootPara.put("startPlanEndTime", startPlanEndTime);
        rootPara.put("endPlanEndTime", endPlanEndTime);
        rootPara.put("backlogIds", backlogIds);
        rootPara.put("closed", completed);

        result = httpAssistantService.post(new TypeReference<List<PtmTaskCardItemDTO>>() {
        }, url, headers, rootPara);

        if (result == null) {
            result = new ArrayList<>();
        }
        return result;
    }

    /**
     * 根据taskId获取ptmWorkItem列表
     *
     * @param taskId
     * @return
     */
    @Override
    public List<PtmTaskWorkItemDTO> getWorkItemByTaskId(Long taskId) {
        List<PtmTaskWorkItemDTO> result;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORKITEM_BY_TASK_ID_TASK_ID_TASK_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("taskId", taskId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskWorkItemDTO>>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskWorkItemDTO>>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 根据taskId等信息删除bpmdata
     *
     * @param terminateTaskDataDTO
     * @return
     */
    @Override
    public Boolean removeBpmDataByTaskId(TerminateTaskDataDTO terminateTaskDataDTO) {
        Boolean result;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_TASK_ACTIVITY_REMOVE_DATA;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("taskId", terminateTaskDataDTO.getTaskId());
        rootPara.put("dataVariableKey", terminateTaskDataDTO.getDataVariableKey());
        rootPara.put("data", terminateTaskDataDTO.getData());
        rootPara.put("uniKeys", terminateTaskDataDTO.getUniKeys());

        log.info("remove bpmdata by taskId ，data：{}", JsonUtils.objectToString(rootPara));

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Boolean>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    /**
     * 获取任务记录详情(ptm_task表)
     *
     * @param workItemId
     * @return
     */
    @Override
    public PtmActWithTaskAndProjectDTO getActivityByWorkItemId(Long workItemId) {
        PtmActWithTaskAndProjectDTO result = null;
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_PROJECT_TASK_ACTIVITY_BY_WORK_ITEM_ID_WORK_ITEM_ID_WORK_ITEM_ID;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> param = new HashMap<>();
        param.put("workItemId", workItemId);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmActWithTaskAndProjectDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmActWithTaskAndProjectDTO>>() {
        }, param);
        result = responseEntity.getBody().getResponseWithException("");
        return result;
    }

    @Override
    public PtmProjectBusinessDTO getProjectBusinessByProcessSerNum(String processSerialNumber) {

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_API_PROJECT_BY_SERIAL_NUMBER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("processSerialNumber", processSerialNumber);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<PtmProjectBusinessDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectBusinessDTO>>() {
        }, rootPara);
        return Objects.requireNonNull(responseEntity.getBody()).getResponseWithException("");
    }

    @Override
    public void deleteProjectByProcessSerNum(String processSerialNumber) {

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_API_PROJECT_DELETE_BY_SERIAL_NUMBER;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("processSerialNumber", processSerialNumber);

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<PtmProjectBusinessDTO>>() {
        }, rootPara);

    }

    /**
     * 根据任务id、活动code查询待办id(需求:9733)
     *
     * @param taskId
     * @param bpmActivityId
     * @return
     */
    @Override
    public List<Long> getBacklogByCondotion(Long taskId, String bpmActivityId) {
        List<Long> result = new ArrayList<>();
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_BACKLOG_IDS_BY_CONDITION;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("taskId", taskId);
        rootPara.put("activityCode", bpmActivityId);
        log.info("根据任务id、活动code查询待办id ，data：{}", JsonUtils.objectToString(rootPara));

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<Long>>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Long>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return responseEntity.getBody().getResponse();
    }

    /**
     * 查询指定任务的签核历程
     *
     * @param requestDTO
     * @return 没有数据返回空集合
     */
    @Override
    public BpmTaskApprovelDTO queryApproveByTaskUid(BpmTaskApproveRequestDTO requestDTO) {
        BpmTaskApprovelDTO result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_TASK_APPROVE_PROCESS;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("locale", requestDTO.getLocale());
        rootPara.put("taskUid", requestDTO.getTaskUid());

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<BpmTaskApprovelDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<BpmTaskApprovelDTO>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    /**
     * 获取需发送line消息的任务
     *
     * @param ptmLineParamDTO
     * @return
     */
    @Override
    public List<PtmLineBacklogDTO.BacklogData> getLineBacklogs(PtmLineParamDTO ptmLineParamDTO) {
        // 请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_DECISION_CONSULT;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("data", ptmLineParamDTO.getData());
        if (ptmLineParamDTO.getBacklogId() != null) {
            rootPara.put("backlogId", ptmLineParamDTO.getBacklogId());
        }
        if (ptmLineParamDTO.getTaskUid() != null) {
            rootPara.put("taskUid", ptmLineParamDTO.getTaskUid());
        }

        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        // 请求接口
        ResponseEntity<BaseResultDTO<PtmLineBacklogDTO>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<PtmLineBacklogDTO>>() {
                }, rootPara);
        PtmLineBacklogDTO lineBacklogDTO = responseEntity.getBody().getResponseWithException("");
        if (null == lineBacklogDTO) {
            return new ArrayList<>();
        }
        return lineBacklogDTO.getData();
    }

    public Map taskcardForceclose(Map<String, Object> map) {
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_FORCECLOSE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.putAll(map);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
        }, rootPara);
        return Objects.requireNonNull(responseEntity.getBody()).getResponseWithException("");
    }

    public Map projectcardForceclose(Map<String, Object> map) {
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.PROJECT_CARD_FORCECLOSE;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.putAll(map);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
        }, rootPara);
        return Objects.requireNonNull(responseEntity.getBody()).getResponseWithException("");
    }

    /**
     * 是否允许退回
     *
     * @param allowRetrieveReqDTO
     * @return 返回码：10:允许撤回 20:不支持撤回 30:已撤回 40:任务不存在
     */
    @Override
    public void checkTaskSupportReturn(List<PtmCheckSupportReturnReqDTO> allowRetrieveReqDTO) {
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_IS_TASK_RETURN;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        Map paramMap = new HashMap();
        paramMap.put("params", allowRetrieveReqDTO);

        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO<Integer>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Integer>>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue() && null != respEntity.getBody()
                    && respEntity.getBody().isOK() && 10 == respEntity.getBody().getResponse()) {

            } else {
                log.error("[ptm-task-return] atmc访问PTM判断任务是否支持退回出现异常, data: {}, 响应内容: {}",
                        JsonUtils.objectToString(allowRetrieveReqDTO),
                        JsonUtils.objectToString(respEntity.getBody()));

                String message = messageUtils.getMessage("exception.ptm.is.allow.return");
                BusinessException ex = BusinessException.create(respEntity.getStatusCodeValue(), ErrorCodeEnum.CHECK_TASK_RETURN_ERROR.getErrCode(), message);
                ex.setDescription(JsonUtils.objectToString(respEntity.getBody()));
                throw ex;
            }
        } catch (HttpServerErrorException ex) {
            // 500
            if (HttpStatus.INTERNAL_SERVER_ERROR.value() == ex.getStatusCode().value()) {
                // 尝试反序列化
                BaseResultDTO baseResultDTO = null;
                try {
                    baseResultDTO = JsonUtils.jsonToObject(ex.getResponseBodyAsString(), new TypeReference<BaseResultDTO>() {
                    });
                } catch (Exception exception) {
                    baseResultDTO = null;
                }

                if (null != baseResultDTO) {
                    // PTM返回的errorType为Application，会导致正式区错误提示框展示为“友好的、通用的”提示信息。
                    // 这边显式将其修改为Business
                    throw BusinessException.create(baseResultDTO.getStatus(),
                            baseResultDTO.getErrorCode(),
                            baseResultDTO.getErrorMessage(),
                            baseResultDTO.getDescription(),
                            baseResultDTO.getStatusDescription(),
                            ErrorTypeEnum.BUSINESS.getValue(),
                            ex);
                }
            }

            // 非500 && 非正常的PTM异常业务响应
            throw ex;
        }
    }


    @Override
    public List<PtmTaskCardDataDTO> getTaskDataByBacklogIds(List<Map<String,Object>> backlogIds, String type, String isAllQuery) {
        return getTaskData(backlogIds,null,null,type,isAllQuery);
    }

    public List<PtmTaskCardDataDTO> getTaskData(List<Map<String,Object>> backlogIds, Long activityId, Long taskId, String type,String isAllQuery) {
        List<PtmTaskCardDataDTO> result = null;

        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.TASK_CARD_DATA_BATCH;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        //入参
        Map<String, Object> rootPara = new HashMap<>();
        if (Objects.nonNull(backlogIds)){
            rootPara.put("taskDataRequireKeyModels", backlogIds);
        }
        if (Objects.nonNull(activityId)){
            rootPara.put("activityId", activityId);
        }
        if (Objects.nonNull(taskId)){
            rootPara.put("taskId", taskId);
        }
        rootPara.put("type", type);
        rootPara.put("isAllQuery",isAllQuery);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmTaskCardDataDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmTaskCardDataDTO>>>() {
        }, rootPara);
        result = responseEntity.getBody().getResponseWithException("");
        return result;

    }

    @Override
    public List<PtmSpecialBacklogDTO> getSpecialBacklog(List<Long> workItemIds) {

        if (CollectionUtils.isEmpty(workItemIds)){
            return null;
        }
        //请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_DERIVE_CARDS;
        //头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, Object> rootPara = new HashMap<>();
        rootPara.put("workItemIds",workItemIds);
        HttpEntity<?> httpEntity = new HttpEntity<>(rootPara, headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<PtmSpecialBacklogDTO>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<PtmSpecialBacklogDTO>>>() {
        }, rootPara);
        return Objects.requireNonNull(responseEntity.getBody()).getResponseWithException("");
    }

    @Override
    public List<ReassignTrace> selectReassignOrHandoverTrace(String workItemId, Integer type) {

        if (StringUtils.isBlank(workItemId)){
            return new ArrayList<>();
        }
        // 请求地址
        String url = envProperties.getPtmUri() + PtmApiConstant.V_2_WORK_ITEM_REASSIGN_HANDOVER_TRACE;
        // 头部
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        // 参数
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("fromWorkItemId",workItemId);
        paramMap.put("traceType",type);
        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        //请求接口
        ResponseEntity<BaseResultDTO<List<ReassignTrace>>> responseEntity = null;
        responseEntity = restTemplate.exchange(url, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<ReassignTrace>>>() {
        }, paramMap);
        return Objects.requireNonNull(responseEntity.getBody()).getResponseWithException("");
    }
}
