package com.digiwin.athena.atmc.http.restful.bpm.impl;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.application.configuration.EnvProperties;
import com.digiwin.athena.atmc.http.constant.BpmApiConstant;
import com.digiwin.athena.atmc.http.restful.bpm.model.ForecatedUserTaskDTO;
import com.digiwin.athena.atmc.http.domain.misc.DemoCreateDTO;
import com.digiwin.athena.atmc.http.domain.misc.DemoItemResultDTO;
import com.digiwin.athena.atmc.http.restful.bpm.BpmService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.util.*;

@Slf4j
@Service
public class BpmServiceImpl implements BpmService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Override
    public void dispatch(String processSerialNumber, String workitemId, String performerId) {
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");

        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_ENGINE_TASKS_DISPATCH_USER_TASK;
        Map<String, Object> map = new HashMap<>();
/*
            "processSerialNumber":"Process0000100000016",                //类型：String  必有字段  备注：流程序號
            "workitemId":"fe6a6d25e4da10048db7e6174d1d9fee",                //类型：String  必有字段  备注：工作項目代號
            "performerId":"02782",                //类型：String  必有字段  备注：處理者
            "agentPerformerId":"emp001",                //类型：String  可有字段  备注：代理人IAM账号
            "comment":"mock",                //类型：String  可有字段  备注：簽核意見
            "locale":"zh_TW",                //类型：String  可有字段  备注：語系
            "processVariable": {
                "key1": "value1"
            }
 */
        map.put("processSerialNumber", processSerialNumber);
        map.put("workitemId", workitemId);
        map.put("performerId", performerId);
        map.put("agentPerformerId", performerId);
        map.put("comment", "");
        map.put("locale", LocaleContextHolder.getLocale().toString());
        HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    @Override
    public void terminateProcess(String processSerialNumber, String workitemId, String performerId) {
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");

        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_ENGINE_PROCESS_INSTANCES_TERMINATE_PROCESS;
        Map<String, Object> map = new HashMap<>();
        /*
                 {
            "processSerialNumber":"Process00100000001",                //类型：String  必有字段  备注：流程序號
            "performerId":"emp001",                //类型：String  必有字段  备注：*處理者
            "workitemId":"fe6a6d25e4da10048db7e6174d1d9fee",                //类型：String  必有字段  备注：*当前事项代號
            "comment":"流程终止",                //类型：String  必有字段  备注：终止原因
            "locale":"zh_CN"                //类型：String  可有字段  备注：语言
        }
         */
        map.put("processSerialNumber", processSerialNumber);
        map.put("workitemId", workitemId);
        map.put("performerId", performerId);
        map.put("comment", "因为所有异常排除都已经结束了，终止流程");
        map.put("locale", LocaleContextHolder.getLocale().toString());
        HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
        try {
            restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
        }catch (HttpServerErrorException.InternalServerError internalServerError){
            String message = internalServerError.getResponseBodyAsString();
            if(StringUtils.isNotBlank(message)) {
                Map response = JsonUtils.jsonToObject(message, Map.class);
                if (response != null && response.containsKey("errorCode") && Objects.equals(response.get("errorCode"), "70930")) {
                    //70930表示流程已经关关闭
                    return;
                }
            }
            throw internalServerError;
        }
    }

    /**
     * 获取预测关卡
     * @param processSerialNumber
     * @param tmActivityId
     */
    public BaseResultDTO<List<ForecatedUserTaskDTO>> getForecastedUsertask(String processSerialNumber, String tmActivityId){

        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");

        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.ATHENA_PROCESS_ENGINE_PROCESS_INSTANCES_GET_FORECASTED_USERTASK;
        Map<String, Object> map = new HashMap<>();
        /*
            {
                "processSerialNumber":"mock",                //类型：String  必有字段  备注：流程序号
                "tmActivityIds": - [                //类型：Array  必有字段  备注：集合
                    "mock"                //类型：String  必有字段  备注：任务ID
                ],
                "locale":"mock"                //类型：String  必有字段  备注：语言
            }
         */
        map.put("processSerialNumber", processSerialNumber);
        List<String> list = new ArrayList<>();
        list.add(tmActivityId);
        map.put("tmActivityIds", list);
        map.put("locale", LocaleContextHolder.getLocale().toString());

        HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
        ResponseEntity<BaseResultDTO<List<ForecatedUserTaskDTO>>> responseEntity = restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<ForecatedUserTaskDTO>>>(){});
        if(responseEntity.getBody().isOK()){
            return responseEntity.getBody();
        }

        return null;
    }


    @Override
    public void reassignWorkitem(String processSerialNumber, String bpmActivityId, String workitemId, String oldPerformerId, String newPerformerId, String comment) {
        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");
/*
{
    "processSerialNumber":"Process00100000001",                //类型：String  必有字段  备注：流程序號
    "workitemId":"fe6a6d25e4da10048db7e6174d1d9fee",                //类型：String  必有字段  备注：*預轉派的工作代號
    "performerId":"emp001",                //类型：String  必有字段  备注：*原處理者
    "newPerformerId":"emp002",                //类型：String  必有字段  备注：*新處理者
    "agentPerformerId":"emp001",                //类型：String  可有字段  备注：代理人IAM账号
    "comment":"非我的工作，故轉由emp002處理",                //类型：String  可有字段  备注：轉派原因
    "locale":"zh_CN"                //类型：String  可有字段  备注：语言
}
*/
        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_ENGINE_WORKITEMS_REASSIGN_WORKITEM;
        Map<String, Object> map = new HashMap<>();

        map.put("processSerialNumber", processSerialNumber);
        map.put("taskId", bpmActivityId);
        map.put("workitemId", workitemId);
        map.put("performerId", oldPerformerId);
        map.put("newPerformerId", newPerformerId);
        map.put("comment", comment);
        map.put("locale", LocaleContextHolder.getLocale().toString());
        HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
        restTemplate.postForEntity(serviceUri, httpEntity, Map.class);
    }

    /**
     * 重置指定租户的数据
     * @param tenantId
     * @return
     */
    public BaseResultDTO resetTenantData(String tenantId){

        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_DEMO_DEMO_DATA_IMPORT;

        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");

        Map<String, Object> map = new HashMap<>();
        map.put("importTenantId", tenantId);

        HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>(){});
        return responseEntity.getBody();
    }

    /**
     * 直接更换工作项目的执行人，目前用于 合并后的任务转派
     * @param workitems
     */
    public BaseResultDTO<Map> updatePerformer(Map workitems){
        //http://athena-devops-doclever.digiwincloud.com.cn/html/web/controller/share/share.html#6177cb03112f8e000b017843

        String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_ENGINE_WORKITEMS_UPDATE_WORKITEMS;

        HttpHeaders headerMap = new HttpHeaders();
        headerMap.add("Content-Type", "application/json;charset=UTF-8");

        HttpEntity httpEntity = new HttpEntity<Map>(workitems, headerMap);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>(){});
        return responseEntity.getBody();
    }

    /**
     * bpm 录制数量
     * @param tenantId
     * @return
     */
    @Override
    public DemoItemResultDTO bpmRecordData(String tenantId){

        DemoItemResultDTO result = new DemoItemResultDTO();
        result.setSuccess(true);
        result.setItem("Bpm 录制数据");
        result.setStartTime(LocalDateTime.now());
        result.setMessage(new ArrayList<>());
        List<String> msgList = result.getMessage();

        try {
            String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_DEMO_DEMO_DATA_EXPORT;

            HttpHeaders headerMap = new HttpHeaders();
            headerMap.add("Content-Type", "application/json;charset=UTF-8");

            Map<String, Object> map = new HashMap<>();
            map.put("exportTenantId", tenantId);

            HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
            ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });

            if(!responseEntity.getBody().isOK()) {
                result.setSuccess(false);
                msgList.add("----录制 Bpm 的数据出现异常,详情查看日志");
                log.error("[录制 Bpm]", responseEntity.getBody());
            }
        }catch (Exception ex){
            result.setSuccess(false);
            msgList.add("Bpm 录制数据 出现异常，详情查看日志");
            msgList.add(ex.getMessage());
            log.error("[Bpm 录制数据]", ex);
        }

        result.setEndTime(LocalDateTime.now());
        return result;
    }

    /**
     * bpm 清除数据
     * @param tenantId
     * @return
     */
    @Override
    public DemoItemResultDTO bpmDeleteData(String tenantId){
        DemoItemResultDTO result = new DemoItemResultDTO();
        result.setSuccess(true);
        result.setItem("Bpm 清除数据");
        result.setStartTime(LocalDateTime.now());
        result.setMessage(new ArrayList<>());
        List<String> msgList = result.getMessage();

        try {
            String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_DEMO_DEMO_DATA_CLEAR_INSTANCE;

            HttpHeaders headerMap = new HttpHeaders();
            headerMap.add("Content-Type", "application/json;charset=UTF-8");

            Map<String, Object> map = new HashMap<>();
            map.put("delTenantId", tenantId);

            HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
            restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {});

        }catch (Exception ex){
            result.setSuccess(false);
            msgList.add("Bpm 清除数据 出现异常，详情查看日志");
            msgList.add(ex.getMessage());
            log.error("[Bpm 清除数据]", ex);
        }

        result.setEndTime(LocalDateTime.now());
        return result;
    }

    /**
     * bpm 录制数量
     * @param dto
     * @return
     */
    @Override
    public DemoItemResultDTO bpmCreateData(DemoCreateDTO dto){

        DemoItemResultDTO result = new DemoItemResultDTO();
        result.setSuccess(true);
        result.setItem("Bpm 重置数据");
        result.setStartTime(LocalDateTime.now());
        result.setMessage(new ArrayList<>());
        List<String> msgList = result.getMessage();

        try {
            String serviceUri = envProperties.getBpmUri() + BpmApiConstant.PROCESS_DEMO_DEMO_DATA_IMPORT;

            HttpHeaders headerMap = new HttpHeaders();
            headerMap.add("Content-Type", "application/json;charset=UTF-8");

            Map<String, Object> map = new HashMap<>();
            map.put("fromDemoTenantId", dto.getSourceTenantId());
            map.put("importTenantId", dto.getTargetTenantId());

            HttpEntity httpEntity = new HttpEntity<Map>(map, headerMap);
            restTemplate.exchange(serviceUri, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });
        }catch (Exception ex){
            result.setSuccess(false);
            msgList.add("Bpm 重置数据 出现异常，详情查看日志");
            msgList.add(ex.getMessage());
            log.error("[Bpm 重置数据]", ex);
        }

        result.setEndTime(LocalDateTime.now());
        return result;

    }


}
