package com.digiwin.athena.atmc.http.restful.ptm;

import com.digiwin.athena.atmc.http.domain.backlog.TerminateTaskDataDTO;
import com.digiwin.athena.atmc.http.domain.task.TaskExceptionDTO;
import com.digiwin.athena.atmc.http.domain.task.mobile.MobileBacklogGroupDTO;
import com.digiwin.athena.atmc.http.domain.task.mobile.MobileProjectGroupDTO;
//import com.digiwin.athena.atmc.http.domain.task.mobile.MobilePtmBacklogDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveRequestDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApprovelDTO;
import com.digiwin.athena.atmc.http.restful.ptm.model.*;
import com.digiwin.athena.atmc.infrastructure.pojo.bo.migration.*;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * PTM 服务
 */
public interface PtmService {

    /**
     * 查询ptm项目卡列表
     * @param userIds 要查询的userId
     * @return 没有值则返回空 List
     */
    List<PtmProjectCardItemDTO> queryProjectCardList(List<String> userIds);

    /**
     * 查询ptm项目卡列表返回code和id
     * @param codes 项目编码
     * @return List
     */
    List<PtmProjectCardItemDTO> queryProjectCardListByCodes(List<String> codes);

    /**
     * 根据 项目卡Id 查询ptm项目卡列表
     * @param projectCardIdList 要查询的项目卡id
     * @return 没有值则返回空 List
     */
    List<PtmProjectCardItemDTO> queryProjectCardListByProjectCardId(List<Long> projectCardIdList,String userId);

    /**
     * 查询进行中的项目卡概要信息（项目id、项目名称，附属的任务id、任务名称等）
     *
     * @param userIds        用户id列表
     * @param projectCardIds 项目卡id列表，可选
     * @param authoredUserId 当前登录用户
     * @return
     */
    List<ProjectCardItemSimpleInfoDTO> queryProjectCardSimpleInfo(List<String> userIds, Collection<Long> projectCardIds, String authoredUserId);

    /**
     * 查询进行中的任务概要信息（任务id、任务名称等）
     *
     * @param userIds        用户id列表
     * @param authoredUserId 当前登录用户
     * @return
     */
    List<BacklogItemSimpleInfoDTO> queryBackCardSimpleInfo(List<String> userIds, List<Long> backlogIds, String authoredUserId);

    /**
     * 查询ptm任务卡列表
     * @param userIds 要查询的userId
     * @return 没有值则返回空 List
     */
    List<PtmTaskCardItemBO> queryTaskCardList(List<String> userIds);

    /**
     * 根据任务编码查询ptm待处理任务卡列表
     * @param codes
     * @return
     */
    List<PtmTaskCardItemBO> queryTaskCardListByCodes(List<String> codes);

    /**
     * 查询ptm任务卡列表
     * @param backlogIdList 要查询的待办ID列表
     * @return 没有值则返回空 List
     */
    List<PtmTaskCardItemBO> queryTaskCardListByBacklogId(List<Long> backlogIdList);

    /**
     * 获取 项目卡 详情
     * @param projectCardId
     * @return
     */
    PtmProjectCardDetailDTO getProjectCardDetail(Long projectCardId);

    /**
     * 获取项目卡中指定任务的所有任务信息
     * @param projectCardId
     * @param taskDefCode
     * @return
     */
    List<PtmProjectCardTaskDetailDTO> getProjectCardDetail(Long projectCardId, String taskDefCode);

    //region 路线

    /**
     * 获取 项目卡 路线
     * @param projectCardId
     * @return
     */
    List<PtmProjectCardRouteDTO> getProjectCardRoutes(Long projectCardId);

    /**
     * 获取 项目 的路线列表
     * @param projectId
     * @return
     */
    PtmProjectCardRouteDTO getProjectRoute(Long projectId);

    /**
     * 获取 指定项目ID 指定任务路线 的任务数据
     * @param projectIds
     * @param routeTasks
     * @return
     */
    List<PtmProjectCardRouteTaskDTO> getProjectCardRouteTasks(List<Long> projectIds, String routeTasks);

    /**
     * 设置默认路线
     * @param defaultRouteId 要设置为默认路线的id
     * @return
     */
    Boolean setDefaultRoute(Long defaultRouteId);

    //endregion

    /**
     * 根据项目任务信息获取进行中的任务列表
     * @param projectId ptm项目id
     * @param taskDefCode km任务code
     * @return  data取的是ptm_task.data
     */
    List<PtmBacklogItemBO> getBacklogByProjectIdAndTaskCode(Long projectId, String taskDefCode);

    /**
     * 根据项目卡任务信息获取进行中的任务列表
     * @param projectCardId ptm项目卡id
     * @param taskDefCode km任务code
     * @return  data取的是ptm_task.data
     */
    List<PtmBacklogItemBO> getBacklogByProjectCardIdAndTaskCode(Long projectCardId, String taskDefCode);

    /**
     * 根据backlogId获取全部中的任务列表
     * @param backlogId 待办id
     * @return
     */
    List<PtmBacklogItemBO> getBacklogByBacklogId(Long backlogId);

    /**
     * 根据backlogId获取全部中的任务列表（批量）
     * @Author：SYQ
     * @Date：2022/6/16 15:52
     */
    List<PtmBacklogItemBO> getBacklogByBacklogIds(List<Long> backlogIdList);


    //region 转派

    /**
     * 转派 项目卡
     * @param requestDTO
     * @return
     */
    Boolean reassignProjectCard(PtmProjectCardReassignRequestDTO requestDTO);

    /**
     * 转派 待办
     * @param requestDTO
     * @return
     */
    Boolean reassignBacklog(PtmBacklogReassignRequestDTO requestDTO);

    //endregion

    //region 记录详情

    /**
     * 获取项目卡记录详情（ptm_project_card表）
     * @param projectCardId
     * @return
     */
    PtmProjectCardRecordDTO getProjectCardRecord(Long projectCardId);

    /**
     * 批量 获取项目卡 记录详情（ptm_project_card表）
     * @param projectCardIds
     * @return 没有值慢返回空集合
     */
    List<PtmProjectCardRecordDTO> getProjectCardRecordByProjectCardIds(List<Long> projectCardIds);

    List<PtmProjectCardRecordDTO> getProjectCardByCondition(Map condition);

    /**
     * 批量-获取指定项目卡中最近一次项目记录<br>
     * 不含data数据
     * @param projectCardIds
     * @return
     */
    List<PtmProjectRecordBO> getLastProjectRecordByProjectCardIds(List<Long> projectCardIds);

    /**
     * 获取项目记录详情(ptm_project表)
     * @param projectId
     * @return
     */
    PtmProjectRecordBO getProjectRecord(Long projectId);

    /**
     * 批量-获取项目 记录详情(ptm_project表)
     * 警告：使用次方法前请确认是否需要项目列表中的data，如果不需要，请使用getProjectRecordByProjectIdsV2替代，需要data会消耗ptm较多内存
     * @param projectIds
     * @return 没有值慢返回空集合
     */
    List<PtmProjectRecordBO> getProjectRecordByProjectIds(List<Long> projectIds);

    /**
     * 批量-获取项目 记录详情(ptm_project表)
     * @param projectIds
     * @param needData 是否需要data数据，默认为需要
     *
     * @return 没有值慢返回空集合
     */
    List<PtmProjectRecordBO> getProjectRecordByProjectIdsV2(List<Long> projectIds, boolean needData);

    /**
     * 获取任务记录详情(ptm_task表)
     * @param taskId
     * @return
     */
    PtmTaskRecordBO getTaskRecord(Long taskId);

    /**
     * 获取任务记录详情(ptm_task表)
     * @param taskIds
     * @return 没有值慢返回空集合
     */
    List<PtmTaskRecordBO> getTaskRecordByTaskIds(List<Long> taskIds);

    /**
     * 获取任务记录详情(ptm_task表)
     * @param taskIds
     * @return 没有值慢返回空集合
     */
    List<PtmTaskRecordBO> getTaskRecordByTaskIds(List<Long> taskIds, boolean needData);

    /**
     * 获取活动记录详情(ptm_activity表)
     * @param activityId
     * @return
     */
    PtmActivityRecordDTO getActivityRecord(Long activityId);

    /**
     * 批量 获取活动记录详情(ptm_activity表)
     * @param activityIds
     * @return 没有值慢返回空集合
     */
    List<PtmActivityRecordDTO> getActivityRecordByActivityIds(List<Long> activityIds);

    /**
     * 获取事项的记录详情(ptm_task_activity_item 表记录)
     * @param workItemId
     * @return
     */
    PtmWorkItemRecordBO getWorkItemRecord(Long workItemId);

    /**
     * 批量 获取事项 的记录详情(ptm_task_activity_item 表记录)
     * @param workItemIds
     * @return 没有值慢返回空集合
     */
    List<PtmWorkItemRecordBO> getWorkItemRecordByWorkItemIds(List<Long> workItemIds);

    /**
     * 获取指定工作项 被发起退回重签或重办的 工作项，可能为null
     * @param type 0 或 null，同一个任务，指签核任务内；1-不是同一个任务
     * @param workItemId
     * @param approvalTaskCode 要查询的签核任务code
     * @return 可能为null
     */
    PtmWorkItemRecordBO getReExecuteWorkItem(Integer type, Long workItemId, String approvalTaskCode);

    /**
     * 获取待办的记录详情(ptm_backlog 表记录)
     * @param backlogId
     * @return
     */
    PtmBacklogRecordDTO getBacklogRecord(Long backlogId);

    /**
     * 批量 获取待办的记录详情(ptm_backlog 表记录)
     * @param backlogIds
     * @return
     */
    List<PtmBacklogRecordDTO> getBacklogRecordByBacklogIds(List<Long> backlogIds);

    //endregion

    //region data、stateData

    /**
     * 根据待办id获取任务卡的data、stateData
     * @param backlogId ptm_backlog.id
     * @param type 非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return data 取的是ptm.activity 的 data
     */
    List<PtmTaskCardDataDTO> getTaskDataByBacklogId(Long backlogId, String type);

    /**
     * 根据 ptm_task_activity.id 获取任务卡的data、stateData
     * @param activityId ptm_task_activity.id
     * @param type 非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return
     */
    PtmTaskCardDataDTO getTaskDataByActivityId(Long activityId, String type);

    /**
     * 根据 ptm_task.id 获取任务卡的data、stateData
     * @param taskId ptm_task.id
     * @param type 非必填，传data，只返回data节点；传dataState，只返回dataState节点；不传都返回，后续异常排除任务卡查询数据可复用
     * @return
     */
    PtmTaskCardDataDTO getTaskDataByTaskId(Long taskId, String type);

    /**
     * 更新 stateData 数据
     * @param activityId ptm_task_activity.id
     * @param stateData
     * @return
     */
    Boolean updateStateData(Long activityId, Object stateData);

    /**
     * 更新 data 数据
     * @param activityId ptm_task_activity.id
     * @param data
     * @return
     */
    Boolean updateData(Long activityId, Object data);

    /**
     * 数据终止-更新 bpmData
     * @param data
     */
    void resetBpmDataForTerminateTaskData(Map data);

    /**
     * 数据转派-更新 bpmData
     * @param data
     */
    void resetBpmDataForReassignTaskData(Map data);

    //endregion

    //region 最近更新时间

    /**
     * 获取指人员的项目卡的最后一次更新时间
     * @param userId
     * @return
     */
    LocalDateTime getProjectCardLastModifyTime(String userId);

    /**
     * 获取指定人员的待办的最后一次更新时间
     * @param userId
     * @return
     */
    LocalDateTime getBacklogLastModifyTime(String userId);

    //endregion

    //region 创建待办

    /**
     * 多元交互-创建外部厂商个人租户待办
     * @param emailRequest
     * @return
     */
    Boolean addCrossTenantBacklog(PtmEmailBacklogRequestDTO emailRequest);

    /**
     * 创建辅助执行者-待办
     * @param assistRequest
     * @return
     */
    Boolean addAssistBacklog(PtmAssistBacklogRequestDTO assistRequest);

    //endregion

    //region 更新项目、任务信息

    /**
     * 批量更新项目基本信息
     * @param param
     * @return
     */
    Boolean batchUpdateProjectInfo(Map param);

    /**
     * 批量更新任务基本信息
     * @param param
     * @return
     */
    Boolean batchUpdateBacklogInfo(Map param);

    //endregion

    //region 项目

    /**
     * 根据流程序号，关闭其所在的主项目。提前结束项目
     * @param map
     * @return
     */
    Boolean closeProjectCard(Map<String,Object> map);

    //endregion

    //region 任务、待办

    /**
     * 批量终止任务
     * @param paraMap
     * @return
     */
    Boolean batchTerminateTask(Map paraMap);

    /**
     * 批量根据ptm_task.id查询所有任务卡信息
     * @param taskIds
     * @return
     */
    List<PtmBacklogPartialDTO> queryBacklogPartialByTaskIds(List<Long> taskIds);

    //endregion

    //region 报表相关

    /**
     * PTM 项目型报表
     * @param param
     * @return
     */
    List<PtmProjectReportItemDTO> queryProjectReport(Map<String, Object> param);

    /**
     * 查询项目下任务的业务数据
     * @param param
     * @return
     */
    List<PtmTaskReportItemDTO> queryProjectTaskReport(Map<String,Object> param);

    //endregion

    //region 异常排除

    /**
     * 根据异常排除待办id获取新发起项目的 所有的任务列表
     * @param originalBacklogIds
     * @return
     */
    List<Map> getSolveTaskList(List<Long> originalBacklogIds);

    //endregion

    //region 知会相关

    /**
     * 更新签核通知信息
     * @param activityId
     * @param informer
     */
    void updateSignInformNotice(Long activityId, Map informer);

    /**
     * 获取 知会通知信息
     * @param bpmActivityUids
     * @return
     */
    List<PtmActivitySignInformerDTO> getSignInformer(List<String> bpmActivityUids);

    //endregion

    /**
     * 获取待办的签核历程
     * @param backlogId
     * @return 没有内容返回空集合
     */
    List<PtmBacklogApprovalDTO> queryApproval(Long backlogId);

    /**
     * 根据代办id 获取待办的签核历程
     * @param backlogId
     * @return 没有内容返回空集合
     */
    List<PtmBacklogApprovalDTO> queryApprovalNew(Long backlogId);

    /**
     * 根据workItemId获取待办的签核历程
     * @param workItemId
     * @return 没有内容返回空集合
     */
    List<PtmBacklogApprovalDTO> queryApprovalByWorkItemId(Long workItemId);

    /**
     * 根据数据获取待办的签核历程
     * @param workItemDataDTO
     * @return
     */
    List<PtmBacklogApprovalDTO> queryApprovalByData(ApprovalWorkItemByDataDTO workItemDataDTO);

    /**
     * 根据项目id和任务code来获取发起的子项目
     * @param projectId
     * @param taskCodes
     * @return
     */
    List<PtmSubProjectResponseDTO> getSubProject(Long projectId, List<String> taskCodes);

    /**
     * 获取已完成的代理任务id
     * @Author：SYQ
     * @Date：2022/6/29 13:45
     */
    List<Long> getCompleteAgentBacklog(String agentPerformerId);

    /**
     * 获取项目卡的可视化面板数据
     * @param projectCardId
     * @return
     */
    PtmProjectCardPanelDTO getProjectCardPanel(Long projectCardId);

    /**
     * 是否允许撤回
     * @param allowRetrieveReqDTO
     * @return 返回码：10:允许撤回 20:不支持撤回 30:已撤回 40:任务不存在
     */
    void checkTaskSupportWithdraw(PtmCheckSupportWithdrawReqDTO allowRetrieveReqDTO);

    /**
     * 是否允许跨BK撤回
     * @param allowRetrieveReqDTOList
     * @return 本次撤回可能涉及到的taskCode
     */
    PtmCheckSupportCrossBkWithdrawRespDTO checkTaskSupportCrossBkWithdraw(List<PtmCheckSupportCrossBkWithdrawReqDTO> allowRetrieveReqDTOList);

    /**
     * 获取项目数据
     * @Author：SYQ
     * @Date：2022/10/11 10:03
     */
    List<PtmProjectCardRecordDTO> getProjectDataByCondition(Map condition);


    /**
     * 通过流程序号获取项目数据
     * @param processSerialNumber
     * @return
     */
    List<PtmProjectCardRecordDTO> getProjectDataByProcessSerialNumber(String processSerialNumber);

    /**
     * 根据项目卡Id查询所有待办
     * @param projectCardId
     * @return
     */
    List<PtmBacklogForProjectCardDTO> queryBacklogListForProjectCardId(Long projectCardId);

    /**
     * 获取团队项目分组（用于移动端）
     * @Author：SYQ
     * @Date：2022/12/29 17:37
     */
    List<MobileProjectGroupDTO> getMobileDepartProjectGroup(List<String> userIds, Integer type);

    /**
     * 获取团队任务分组（用于移动端）
     * @Author：SYQ
     * @Date：2022/12/30 09:42
     */
    List<MobileBacklogGroupDTO> getMobileDepartBacklogGroup(List<String> userIds, Integer type);

    /**
     * 根据分组查询项目卡列表（用于移动端）
     * @Author：SYQ
     * @Date：2023/1/4 15:03
     */
    List<PtmProjectCardItemDTO> findMobileProjectCardGroupList(List<String> userIds,String projectDefCode);

    /**
     * 根据分组查询任务卡列表（用于移动端）
     * @Author：SYQ
     * @Date：2023/1/5 13:50
     */
    List<PtmTaskCardItemBO> findMobileBacklogCardGroupList(List<String> userIds, String taskDefCode);
    /**
     * 根据人员及查询关键字查询符合条件进行中的任务
     * @Author：SYQ
     * @Date：2023/1/9 14:55
     */
    List<MobilePtmBacklogBO> searchBacklogByParam(String queryParam, List<String> userIds);
    /**
     * 根据人员及任务id查询进行中的任务信息
     * @Author：SYQ
     * @Date：2023/1/9 15:15
     */
    List<MobilePtmBacklogBO> searchBacklogByIdsAndUserIds(List<String> userIds, List<Long> backlogIds);

    /**
     * 查询任务真实执行人
     * @backlogIds 任务ID集合
     * @return
     */
    List<PtmAgentPerformerDTO> queryAgentPerformerList(List<Long> backlogIds);

    /**
     * 根据用户id列表， 起始的待办的计划完成时间 和 截止的待办的计划完成时间 查询待办项列表
     *
     * @param userIds 用户id列表
     * @param beginDate 起始的待办的计划完成时间，时间格式：yyyy-MM-dd
     * @param endDate 截止的待办的计划完成时间，时间格式：yyyy-MM-dd
     * @return
     */
    List<PtmWorkItemV2DTO> queryUserWorkItemListByTimeRange(Set<String> userIds, String beginDate, String endDate);



    /**
     * 根据活动id和任务id获取bpmdata
     *
     * @param  activityId
     * @param  taskId
     * @return
     */
    Map<String, Object> getBpmDataByActivityIdAndTaskId(Long activityId, Long taskId);

    /**
     * 获取项目下异常任务信息
     * @param projectId
     * @param taskCode
     * @return
     */
    List<TaskExceptionDTO> getExceptionTaskInfo(long projectId, String taskCode);

    /**
     * 交接任务
     * @param newPerformerId 交接人
     * @param newPerformerName 交接姓名
     * @param backlogIds 任务ID
     * @return
     */
    Map<String, Object> handoverTask(String newPerformerId, String newPerformerName, List<Long> backlogIds);


    /**
     * 交接项目
     * @param newPerformerId 交接人
     * @param newPerformerName 交接姓名
     * @param projectCardIds 项目卡id
     * @return
     */
    Map<String, Object> handoverProject(String newPerformerId, String newPerformerName, List<Long> projectCardIds);

    /**
     * 判断任务是否存在转派行为（因转派产生/因转派关闭）
     *
     * @param backlogId 待办id
     * @return
     */
    Boolean reassignOrNot(Long backlogId);

    /**
     * 获取转派历程
     *
     * @param backlogId 待办id
     * @return
     */
    List<PtmRessignTrace> getReassignTrace(Long backlogId);

    /**
     * 获取条件范围内的所有工时
     * @param userIds
     * @param startPlanEndTime
     * @param endPlanEndTime
     * @param backlogIds
     * @return
     */
    List<PtmTaskCardItemBO> queryAllTaskCardList(List<String> userIds, String startPlanEndTime, String endPlanEndTime, List<Long> backlogIds, Boolean completed);

    /**
     * 根据异常排除待办id获取新发起项目的 所有的任务列表
     * @param backlogId 待办id
     * @return
     */
    List<PtmSolveTask> getSolveTaskListV2(Long backlogId);

    /**
     * 根据taskId获取ptm workItem列表
     * @param taskId 数据足迹的actId，ptm的task表的id
     * @return
     */
    List<PtmTaskWorkItemDTO> getWorkItemByTaskId(Long taskId);

    /**
     * 根据taskId获取ptm workItem列表
     * @param terminateTaskDataDTO 数据足迹的actId，ptm的task表的id
     * @return
     */
    Boolean removeBpmDataByTaskId(TerminateTaskDataDTO terminateTaskDataDTO);

    /**
     * 根据任务id、活动code查询待办id(需求:9733)
     * @param taskId
     * @param bpmActivityId
     * @return
     */
    List<Long> getBacklogByCondotion(Long taskId, String bpmActivityId);



    /**
     * 查询指定任务的签核历程
     * @param requestDTO
     * @return 没有数据返回空集合
     */
    BpmTaskApprovelDTO queryApproveByTaskUid(BpmTaskApproveRequestDTO requestDTO);

    /**
     * @description 根据workItemId查询项目任务活动信息
     * @param workItemId Long
     * @return PtmActWithTaskAndProjectDTO
    */
    PtmActWithTaskAndProjectDTO getActivityByWorkItemId(Long workItemId);

    /**
     * 根据流程序号擦查询项目相关业务字段
     * @param processSerialNumber
     * @return
     */
    PtmProjectBusinessDTO getProjectBusinessByProcessSerNum(String processSerialNumber);


    /**
     * 根据流程序号，删除项目相关信息
     * @param processSerialNumber
     */
    void deleteProjectByProcessSerNum(String processSerialNumber);

    /**
     * 获取需发送line消息的任务
     * @param ptmLineParamDTO
     * @return
     */
    List<PtmLineBacklogDTO.BacklogData> getLineBacklogs(PtmLineParamDTO ptmLineParamDTO);


    /**
     * 强制关闭任务卡
     * @param map
     * @return
     */
    Map taskcardForceclose(Map<String,Object> map);

    /**
     * 强制关闭项目卡
     * @param map
     * @return
     */
    Map projectcardForceclose(Map<String,Object> map);

    /**
     * 是否允许退回
     * @param allowRetrieveReqDTO
     * @return 返回码：10:允许撤回 20:不支持撤回 30:已撤回 40:任务不存在
     */
    void checkTaskSupportReturn(List<PtmCheckSupportReturnReqDTO> allowRetrieveReqDTO);


    List<PtmTaskCardDataDTO> getTaskDataByBacklogIds(List<Map<String,Object>> backlogIds, String type, String isAllQuery);

    /**
     * 查询常规卡片所对应的辅助执行人，跨租户的卡片backlogId
     */
    List<PtmSpecialBacklogDTO> getSpecialBacklog(List<Long> workItemIds);

    /**
     * 查询事项的转派或者交接记录
     * @return
     */
    List<ReassignTrace> selectReassignOrHandoverTrace(String workItemId, Integer type);
}
