package com.digiwin.athena.atmc.http.restful.atdm.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.HttpUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.application.configuration.EnvProperties;
import com.digiwin.athena.atmc.http.constant.AtdmApiConstant;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.domain.action.MergeSubmitActionDTO;
import com.digiwin.athena.atmc.http.domain.action.SubmitActionDTO;
import com.digiwin.athena.atmc.http.domain.action.SubmitExecuteContext;
import com.digiwin.athena.atmc.http.restful.aglie.model.QueryDataByActionVO;
import com.digiwin.athena.atmc.http.restful.atdm.AtdmService;
import com.digiwin.athena.atmc.http.restful.atdm.model.QueryResult;
import com.digiwin.athena.atmc.http.restful.atdm.model.QueryResultSet;
import com.digiwin.athena.atmc.http.restful.atdm.model.RecycleRecordStateChangeDTO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.jugg.agile.biz.digiwin.config.biz.DwBizConfig;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class AtdmServiceImpl implements AtdmService {

    @Autowired
    EnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    MessageUtils messageUtils;

    /**
     * 提交
     *
     * @param submitAction
     * @return
     */
    public Map submitTask(SubmitActionDTO submitAction) {
        if (submitAction.getAction() == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }
        String url = getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_TASK;
        url = appendTraceInfo(url, submitAction.getAction().getExecuteContext(), submitAction.getAction().getActionId());
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        // 实际执行操作的用户信息
        submitAction.setOperateAuthoredUser(AppAuthContextHolder.getContext().getAuthoredUser());

        HttpEntity httpEntity = new HttpEntity<SubmitActionDTO>(submitAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }

    private String getAtdmUri() {
        return DwBizConfig.AaskDomainConfig.Sai.getAtdmUri();

    }



    /**
     * 执行
     *
     * @param submitAction
     * @return
     */
    public Map executeTask(SubmitActionDTO submitAction) {
        String url = getAtdmUri() + AtdmApiConstant.ACTION_EXECUTE;
        url = appendTraceInfo(url, submitAction.getAction().getExecuteContext(), submitAction.getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<SubmitActionDTO>(submitAction, headers);
        ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Object>() {
        });
        return HttpUtils.descResponseBody(url, null, respEntity, new TypeReference<Map>() {
        });
    }

    /**
     * 根据actionId查询
     *
     * @param action
     * @return
     */
    public BaseResultDTO queryByActionId(Map action) {
        String uri = getAtdmUri() + AtdmApiConstant.DATA_QUERY_BY_ACTION_ID;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<Map>(action, headers);
        ResponseEntity<BaseResultDTO<Map>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<Map>>() {
                });

        return responseEntity.getBody();
    }

    /**
     * 提交合并的任务
     *
     * @param mergeActionList
     * @return
     */
    public List<Map> submitMergeTask(List<SubmitActionDTO> mergeActionList) {
        if (CollectionUtils.isEmpty(mergeActionList)) {
            return new ArrayList<>();
        }
        if (mergeActionList.get(0) == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        String uri = getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_MERGE_TASK;

        uri = appendTraceInfo(uri, mergeActionList.get(0).getAction().getExecuteContext(), mergeActionList.get(0).getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        if (CollectionUtils.isNotEmpty(mergeActionList)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            mergeActionList.stream().forEach(action -> action.setOperateAuthoredUser(authoredUser));
        }

        HttpEntity httpEntity = new HttpEntity<List<SubmitActionDTO>>(mergeActionList, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
                });
        return responseEntity.getBody().getResponseWithException("");
    }

    /**
     * 提交合并的任务_支持跨BK
     *
     * @param mergeActionList
     * @return
     */
    public List<Map> submitMergeTaskCrossBk(List<SubmitActionDTO> mergeActionList) {
        if (CollectionUtils.isEmpty(mergeActionList)) {
            return new ArrayList<>();
        }
        if (mergeActionList.get(0) == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        String uri = getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_MERGE_TASK_CROSS_BK;

        uri = appendTraceInfo(uri, mergeActionList.get(0).getAction().getExecuteContext(), mergeActionList.get(0).getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        if (CollectionUtils.isNotEmpty(mergeActionList)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            mergeActionList.stream().forEach(action -> action.setOperateAuthoredUser(authoredUser));
        }

        HttpEntity httpEntity = new HttpEntity<List<SubmitActionDTO>>(mergeActionList, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
                });
        return responseEntity.getBody().getResponseWithException("");
    }

    private void addLang(HttpHeaders headers) {
        headers.add("locale", LocaleContextHolder.getLocale().toString());
    }

    static String appendTraceInfo(String url, SubmitExecuteContext executeContext, String actionId) {
        if (StringUtils.isEmpty(url)) {
            return url;
        }
        StringBuilder stringBuilder = new StringBuilder();
        if (executeContext != null) {
            if (executeContext.getTmActivityId() != null) {
                stringBuilder.append(String.format("&TmActivityId=%s", executeContext.getTmActivityId()));
            }
            if (executeContext.getBacklogId() != null) {
                stringBuilder.append(String.format("&BacklogId=%s", executeContext.getBacklogId()));
            }
            if (executeContext.getAuthoredUser() != null) {
                stringBuilder.append(String.format("&author=%s-%s", executeContext.getAuthoredUser().getTenantId(), executeContext.getAuthoredUser().getUserId()));
            }
            if (executeContext.getProcessSerialNumber() != null) {
                stringBuilder.append(String.format("&processSerialNumber=%s", executeContext.getProcessSerialNumber()));
            }
        }
        if (actionId != null) {
            stringBuilder.append(String.format("&actionId=%s", actionId));
        }

        if (stringBuilder.length() > 0) {
            if (url.contains("?")) {
                return url + stringBuilder.substring(1);
            } else {
                return url + "?" + stringBuilder.substring(1);
            }
        } else {
            return url;
        }
    }

    @Override
    public void markRecycleRecordExpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_EXPIRED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);

        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    @Override
    public void markRecycleRecordUnexpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_UNEXPIRED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);

        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    @Override
    public void markRecycleRecordDeleted(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        String uri = getAtdmUri() + AtdmApiConstant.RECYCLE_STATE_DELETED;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        HttpEntity httpEntity = new HttpEntity<>(recycleRecordStateChangeDTO, headers);
        ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO>() {
                });
        respEntity.getBody().getResponseWithException("");
    }

    /**
     * 获取任务卡CheckItems等数
     *
     * @param queryDataByActionVO
     * @return
     */
    @Override
    public BaseResultDTO<JSONObject> getDataByAction(QueryDataByActionVO queryDataByActionVO) {
//        String url = MessageFormat.format("{0}api/atdm/v1/data/query/by/action", envProperties.getAtdmUri());
        String url = getAtdmUri() + AtdmApiConstant.DATA_QUERY_BY_ACTION;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);


        HttpEntity httpEntity = new HttpEntity<>(queryDataByActionVO, headers);
        ResponseEntity<BaseResultDTO<JSONObject>> responseEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<JSONObject>>() {
                });

        return responseEntity.getBody();
    }


    /**
     * 退回
     *
     * @param mergeActionList
     * @return
     */
    public List<Map> submitReturnTask(List<SubmitActionDTO> mergeActionList) {
        if (CollectionUtils.isEmpty(mergeActionList)) {
            return new ArrayList<>();
        }
        if (mergeActionList.get(0) == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        String uri = getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_RETURN_MERGE_TASK;

        uri = appendTraceInfo(uri, mergeActionList.get(0).getAction().getExecuteContext(), mergeActionList.get(0).getAction().getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        if (CollectionUtils.isNotEmpty(mergeActionList)) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            mergeActionList.stream().forEach(action -> action.setOperateAuthoredUser(authoredUser));
        }

        HttpEntity httpEntity = new HttpEntity<List<SubmitActionDTO>>(mergeActionList, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
                });
        return responseEntity.getBody().getResponseWithException("");
    }


    @Override
    public QueryResultSet getResultSet(Map map, AuthoredUser authoredUser) {

//        if (dataSourceSet == null) {
//            return QueryResultSet.empty();
//        }
        Map<String, Object> executeContext = (Map<String, Object>) map.get("executeContext");
        String actionId = null;
        String tmActivityId = executeContext.get("tmActivityId").toString();
        Long backlogId = Long.parseLong(executeContext.get("backlogId").toString());
        Map<String, Object> taskWithBacklogData = (Map<String, Object>) executeContext.get("taskWithBacklogData");
        String processSerialNumber = taskWithBacklogData.get("processSerialNumber").toString();
        Map<String, Object> dataSourceSet = (Map<String, Object>) map.get("dataSourceSet");
        List<Map<String, Object>> dataSourceList = (List<Map<String, Object>>) dataSourceSet.get("dataSourceList");
        if (!dataSourceList.isEmpty()) {
            actionId = dataSourceList.get(0).get("actionId").toString();
        }


        SubmitExecuteContext submitExecuteContext = new SubmitExecuteContext();
        submitExecuteContext.setAuthoredUser(authoredUser);
        submitExecuteContext.setTmActivityId(tmActivityId);
        submitExecuteContext.setBacklogId(backlogId);
        submitExecuteContext.setProcessSerialNumber(processSerialNumber);

        String url = getAtdmUri() + AtdmApiConstant.WITH_META_DATA;
        url = appendTraceInfo(url, submitExecuteContext, actionId);
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());
        Map<String, Object> body = new HashMap<>();
        body.put("dataSourceSetDTO", map.get("dataSourceSet"));
        body.put("executeContext", map.get("executeContext"));
        body.put("parameter", map.get("parameter"));
        body.put("dataViewQuery", null);

        HttpEntity requestEntity = new HttpEntity<>(body, headers);

        ResponseEntity<BaseResultDTO<QueryResultSet>> apiResultDTO = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                new ParameterizedTypeReference<BaseResultDTO<QueryResultSet>>() {
                });

        QueryResultSet queryResultSet = apiResultDTO.getBody().getResponseWithException("");

        Map<String, List<QueryResult>> group = queryResultSet.getQueryResults().stream().collect(Collectors.groupingBy(QueryResult::getDataSourceName));
        for (Map.Entry<String, List<QueryResult>> groupItem : group.entrySet()) {
            // add for 50243:全局参数没有dataKey等信息，也不需要设置重新设置主数据源
            if ("".equals(groupItem.getKey())) {
                continue;
            }
            List<QueryResult> resultList = groupItem.getValue();
            if (resultList.size() > 1) {
                QueryResult qr = resultList.get(0);
                for (int i = 1; i < resultList.size(); i++) {
                    qr.getData().addAll(resultList.get(i).getData());
                    qr.getDataKeyIndex().putAll(resultList.get(i).getDataKeyIndex());
                    //有可能第一个 QueryResult 没有查到数据，所以需要根据条件更新 DataKeys 和 ApiMetadataCollection
                    if (org.apache.commons.collections4.CollectionUtils.isEmpty(qr.getDataKeys()) && org.apache.commons.collections4.CollectionUtils.isNotEmpty(resultList.get(i).getDataKeys())) {
                        qr.setDataKeys(resultList.get(i).getDataKeys());
                    }
                    queryResultSet.getQueryResults().remove(resultList.get(i));
                }
            }
        }
        return queryResultSet;
    }

    @Override
    public List<Map> mergeTaskRetry(MergeSubmitActionDTO mergeSubmitAction) {
        if (Objects.isNull(mergeSubmitAction.getActionMerge())) {
            return new ArrayList<>();
        }
        if (mergeSubmitAction.getActionMerge().getActionList() == null) {
            throw new IllegalArgumentException("submitAction.getAction()");
        }

        String uri = getAtdmUri() + AtdmApiConstant.ACTION_SUBMIT_MERGE_TASK_RETRY;

        uri = appendTraceInfo(uri, mergeSubmitAction.getActionMerge().getActionList().get(0).getExecuteContext(), mergeSubmitAction.getActionMerge().getActionList().get(0).getActionId());

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        addLang(headers);

        HttpEntity httpEntity = new HttpEntity<>(mergeSubmitAction, headers);
        ResponseEntity<BaseResultDTO<List<Map>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
                });
        return responseEntity.getBody().getResponseWithException("");
    }

    /**
     * 清除缓存
     */
    @Override
    public void clearCache(Long size) {
        String url = getAtdmUri() + AtdmApiConstant.CACHE_RESET;
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", LocaleContextHolder.getLocale().toString());
        Map<String, Object> param = new HashMap<>();
        param.put("size", size);
        HttpEntity<BaseResultDTO<Object>> reqEntity = new HttpEntity<>(null, headers);
        ResponseEntity<BaseResultDTO<Object>> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Object>>() {
        }, param);
        // 请求失败，直接抛出异常
        if (HttpStatus.SC_OK != respEntity.getStatusCodeValue() || !respEntity.getBody().isOK()) {
            throw ErrorCodeEnum.ATMC_REMOVE_CACHE_FAIL.getBusinessException(JsonUtils.objectToString(respEntity.getBody()));
        }
    }
}
