package com.digiwin.athena.km_deployer_service.config.mongodb;

import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import lombok.Data;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.config.MongoConfigurationSupport;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoClientDatabaseFactory;
import org.springframework.data.mongodb.core.convert.DbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultDbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultMongoTypeMapper;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;
import org.springframework.data.mongodb.core.mapping.MongoMappingContext;
import org.springframework.data.mongodb.repository.config.EnableMongoRepositories;

import java.util.concurrent.TimeUnit;

/***
 * @Author wangxiao
 * @Date 2021/7/1 9:20
 * @Description mongod配置
 **/
@Data
@Configuration
@EnableMongoRepositories(basePackages = DataMapMongodbConfig.BASE_PACKAGE, repositoryBaseClass = ExtendSimpleMongoRepository.class)
@ConditionalOnProperty(prefix = "spring.data.mongo-datamap", name = "database")
public class DataMapMongodbConfig extends MongoConfigurationSupport {
    public static final String BASE_PACKAGE = "com.digiwin.athena.mongodb.datamap.repository";
    public static final String MONGO_TEMPLATE_REF = "mongoDatamapTemplate";

    @Value("${spring.data.mongodb.uri}")
    private String uri;

    @Value("${spring.data.mongo-datamap.database}")
    private String databaseDatamap;

    @Bean("datamapMongoClient")
    public MongoClient mongoClient() {
        MongoClientSettings mongoClientSettings = MongoClientSettings.builder()
                .applyConnectionString(new ConnectionString(uri))
                .applyToConnectionPoolSettings(builder -> builder.maxSize(30)
                        .minSize(10)
                        .maxConnectionIdleTime(10, TimeUnit.MINUTES)
                        .maxConnectionLifeTime(10, TimeUnit.MINUTES)
                        .maxWaitTime(15, TimeUnit.MINUTES))
                .build();
        return MongoClients.create(mongoClientSettings);
    }

    @Bean("datamapMongoMappingContext")
    public MongoMappingContext mongoMappingContext() {
        MongoMappingContext mappingContext = new MongoMappingContext();
        return mappingContext;
    }

    @Bean("datamapMongoDbFactory")
    public MongoDatabaseFactory mongoDbFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), getDatabaseName());
    }

    @Bean(name = DataMapMongodbConfig.MONGO_TEMPLATE_REF)
    public MongoTemplate mongoTemplate(@Qualifier("datamapMongoDbFactory") MongoDatabaseFactory mongoDbFactory,
                                       @Qualifier("datamapMongoMappingContext") MongoMappingContext mongoMappingContext) throws Exception {
        return new MongoTemplate(mongoDbFactory(), mappingMongoConverter());
    }

    @Bean("datamapMappingMongoConverter")
    public MappingMongoConverter mappingMongoConverter() throws Exception {
        DbRefResolver dbRefResolver = new DefaultDbRefResolver(mongoDbFactory());
        MappingMongoConverter converter = new MappingMongoConverter(dbRefResolver, mongoMappingContext());
        //不保存 _class 属性到mongo
        converter.setTypeMapper(new DefaultMongoTypeMapper(null));
        converter.setCustomConversions(customConversions());

        return converter;
    }

    @Override
    protected String getDatabaseName() {
        return databaseDatamap;
    }
}
