package com.digiwin.athena.km_deployer_service.config.mongodb;

import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import lombok.Data;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.mongo.MongoProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.config.MongoConfigurationSupport;
import org.springframework.data.mongodb.core.*;
import org.springframework.data.mongodb.core.convert.DbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultDbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultMongoTypeMapper;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;

import java.util.concurrent.TimeUnit;

/**
 * @Auther: lixinyuan
 * @Date: 2022/2/9 9:48
 * @Description:
 */
@Data
@Configuration
@ConfigurationProperties(prefix = "spring.data.mongodb")
public class MongoConfig extends MongoConfigurationSupport {

    private String uri;

    private String database;

    @Value("${mongo.suffix}")
    private String mongoSuffix;

    @Bean
    @Primary
    MongoTemplate mongoTemplate() throws Exception {
        return new MongoTemplate(mongoDbFactory(), mappingMongoConverter());
    }

    @Primary
    @Bean
    public MongoDatabaseFactory mongoDbFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), getDatabaseName());
    }


    @Bean
    /*  @Scope("prototype")*/
    @Primary
    public com.mongodb.client.MongoClient mongoClient() {
        MongoClientSettings mongoClientSettings = MongoClientSettings.builder()
                .applyConnectionString(new ConnectionString(uri))
                .applyToConnectionPoolSettings(builder -> builder.maxSize(30)
                        .maxSize(30)
                        .minSize(10)
                        .maxConnectionIdleTime(10, TimeUnit.MINUTES)
                        .maxConnectionLifeTime(10, TimeUnit.MINUTES)
                        .maxWaitTime(15, TimeUnit.MINUTES))
                .build();
        return MongoClients.create(mongoClientSettings);
    }

    @Bean
    @Primary
    public MappingMongoConverter mappingMongoConverter() throws Exception {
        DbRefResolver dbRefResolver = new DefaultDbRefResolver(mongoDbFactory());
        MappingMongoConverter converter = new MappingMongoConverter(dbRefResolver, mongoMappingContext(customConversions()));
        //不保存 _class 属性到mongo
        converter.setTypeMapper(new DefaultMongoTypeMapper(null));
        converter.setCustomConversions(customConversions());

        return converter;
    }

//    @Bean
//    public MongoClientOptions mongoOptions() {
//        //将最大闲置超时时间设置3000ms
//        return MongoClientOptions.builder()
//                .connectTimeout(10000)
//                .maxConnectionIdleTime(4*60*1000)
//                .build();
//    }


    @Bean
    @Primary
    public MongoClient mongo() {
        MongoClientSettings mongoClientSettings = MongoClientSettings.builder()
                .applyConnectionString(new ConnectionString(uri))
                .applyToConnectionPoolSettings(builder -> builder
                        .maxSize(30)
                        .minSize(10)
                        .maxConnectionIdleTime(10, TimeUnit.MINUTES)
                        .maxConnectionLifeTime(10, TimeUnit.MINUTES)
                        .maxWaitTime(15, TimeUnit.MINUTES))
                .build();
        return MongoClients.create(mongoClientSettings);
    }


    @Bean
    MongoDatabaseFactorySupport<?> mongoDbFactory(ObjectProvider<MongoClient> mongo,
                                                  ObjectProvider<com.mongodb.client.MongoClient> mongoClient, MongoProperties properties) {
        MongoClient preferredClient = mongo.getIfAvailable();
        if (preferredClient != null) {
            return new MySimpleMongoDbFactory(preferredClient, properties.getMongoClientDatabase(), mongoSuffix);
        }
        com.mongodb.client.MongoClient fallbackClient = mongoClient.getIfAvailable();
        if (fallbackClient != null) {
            return new MySimpleMongoClientDbFactory(fallbackClient, properties.getMongoClientDatabase(), mongoSuffix);
        }
        throw new IllegalStateException("Expected to find at least one MongoDB client.");
    }

    @Override
    protected String getDatabaseName() {
        return database;
    }
}
