package com.digiwin.athena.km_deployer_service.config.mongodb;

import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import lombok.Data;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.config.MongoConfigurationSupport;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoClientDatabaseFactory;
import org.springframework.data.mongodb.core.convert.DbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultDbRefResolver;
import org.springframework.data.mongodb.core.convert.DefaultMongoTypeMapper;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;
import org.springframework.data.mongodb.core.mapping.MongoMappingContext;
import org.springframework.data.mongodb.repository.config.EnableMongoRepositories;

import java.util.concurrent.TimeUnit;

/***
 * @Author caichao
 * @Date 2024/09/12 9:20
 * @Description mongod配置
 **/
@Data
@Configuration
@EnableMongoRepositories(basePackages = PresetMongodbConfig.BASE_PACKAGE, repositoryBaseClass = ExtendSimpleMongoRepository.class)
@ConditionalOnProperty(prefix = "spring.data.mongo-preset", name = "database")
public class PresetMongodbConfig extends MongoConfigurationSupport {
    public static final String BASE_PACKAGE = "com.digiwin.athena.mongodb.preset.repository";
    public static final String MONGO_TEMPLATE_REF = "mongoPresetTemplate";

    @Value("${spring.data.mongodb.uri}")
    private String uri;

    @Value("${spring.data.mongo-preset.database}")
    private String databasePreset;

    @Bean("presetMongoClient")
    public MongoClient mongoClient() {
        MongoClientSettings mongoClientSettings = MongoClientSettings.builder()
                .applyConnectionString(new ConnectionString(uri))
                .applyToConnectionPoolSettings(builder -> builder.maxSize(30)
                        .minSize(10)
                        .maxConnectionIdleTime(10, TimeUnit.MINUTES)
                        .maxConnectionLifeTime(10, TimeUnit.MINUTES)
                        .maxWaitTime(15, TimeUnit.MINUTES))
                .build();
        return MongoClients.create(mongoClientSettings);
    }

    @Bean("presetMongoMappingContext")
    public MongoMappingContext mongoMappingContext() {
        MongoMappingContext mappingContext = new MongoMappingContext();
        return mappingContext;
    }

    @Bean("presetMongoDbFactory")
    public MongoDatabaseFactory mongoDbFactory() {
        return new SimpleMongoClientDatabaseFactory(mongoClient(), getDatabaseName());
    }

    @Bean(name = PresetMongodbConfig.MONGO_TEMPLATE_REF)
    public MongoTemplate mongoTemplate(@Qualifier("presetMongoDbFactory") MongoDatabaseFactory mongoDbFactory,
                                       @Qualifier("presetMongoMappingContext") MongoMappingContext mongoMappingContext) throws Exception {
        return new MongoTemplate(mongoDbFactory(), presetMappingMongoConverter());
    }

    @Bean("presetMappingMongoConverter")
    public MappingMongoConverter presetMappingMongoConverter() {
        DbRefResolver dbRefResolver = new DefaultDbRefResolver(mongoDbFactory());
        MappingMongoConverter converter = new MappingMongoConverter(dbRefResolver, mongoMappingContext());
        //不保存 _class 属性到mongo
        converter.setTypeMapper(new DefaultMongoTypeMapper(null));
        converter.setCustomConversions(customConversions());

        return converter;
    }

    @Override
    protected String getDatabaseName() {
        return databasePreset;
    }
}
