package com.digiwin.athena.km_deployer_service.config.neo4j;

import lombok.extern.slf4j.Slf4j;
import org.neo4j.driver.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.neo4j.Neo4jProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.neo4j.core.Neo4jClient;
import org.springframework.data.neo4j.core.Neo4jTemplate;
import org.springframework.data.neo4j.core.transaction.Neo4jTransactionManager;
import org.springframework.data.neo4j.repository.config.EnableNeo4jRepositories;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import java.util.concurrent.TimeUnit;

/**
 * @Author wangxiao
 * @Date 2021/7/1 9:21
 * @Description neo4j配置
 **/
//@Configuration
@EnableTransactionManagement
@EnableNeo4jRepositories(
        basePackages = BackupNeo4jConfig.BASE_PACKAGE,
        transactionManagerRef = BackupNeo4jConfig.TRANSACTION_MANAGER
)
@ConditionalOnProperty(prefix = "spring.neo4j.domain2", name = "enable", havingValue = "true")
@Slf4j
public class BackupNeo4jConfig {
    public static final String TRANSACTION_MANAGER = "domain2TransactionManager";
    public static final String NEO4J_DRIVER = "domain2Driver";
    public static final String NEO4J_TEMPLATE = "domain2Neo4jTemplate";
    static final String BASE_PACKAGE = "com.digiwin.athena.km_deployer_service.neo4jbasepkg.backup";

    @Bean(name = "domain2Neo4jProperties")
    @ConfigurationProperties("spring.neo4j.domain2")
    public Neo4jProperties neo4jProperties() {
        return new Neo4jProperties();
    }

    @Bean(name = NEO4J_DRIVER)
    public Driver driver() {
        log.info("neo4j domain2 driver初始化开始， username: {}, password: {}", neo4jProperties().getAuthentication().getUsername(), neo4jProperties().getAuthentication().getPassword());
        Config config = Config.builder()
                .withConnectionTimeout(10, TimeUnit.SECONDS)
                .withConnectionLivenessCheckTimeout(2, TimeUnit.MINUTES)
                .withMaxConnectionLifetime(4, TimeUnit.MINUTES)
                .withMaxConnectionPoolSize(50)
                .withConnectionAcquisitionTimeout(1, TimeUnit.MINUTES)
                .build();

        Driver driver = GraphDatabase.driver(neo4jProperties().getUri(),
                AuthTokens.basic(
                        neo4jProperties().getAuthentication().getUsername(),
                        neo4jProperties().getAuthentication().getPassword()),
                config);
        try {
            driver.verifyConnectivity();
        } catch (Exception e) {
            log.error("back库初始化失败");
            driver = null;
        }
        return driver;
    }

    @Bean
    public Neo4jClient neo4jClient() {
        return Neo4jClient.create(driver());
    }

    @Bean(name = TRANSACTION_MANAGER)
    public Neo4jTransactionManager neo4jTransactionManager() {
        return new Neo4jTransactionManager(driver());
    }


    @Bean(name = NEO4J_TEMPLATE)
    public Neo4jTemplate neo4jTemplate(Driver driver) {
        return new Neo4jTemplate(Neo4jClient.create(driver));
    }
}
