package com.digiwin.athena.km_deployer_service.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * @author cjbi
 */
@Controller
@Slf4j
@RequestMapping("/jacoco")
public class JacocoController {

    @Value("${jacoco.workingDir:/usr/local/km_deployer_service}")
    private String workingDir = "";

    @Value("${jacoco.jarDir:/usr/local/km_deployer_service/jacoco}")
    private String jacocoJarDir;

    private void dumpJacocoExec() {
        try {
            Class<?> rt = Class.forName("org.jacoco.agent.rt.RT");
            Method getAgent = rt.getMethod("getAgent");
            Method dump = getAgent.getReturnType().getMethod("dump", boolean.class);
            Object agent = getAgent.invoke(null);
            //  if true, the current execution data is cleared
            dump.invoke(agent, false);
        } catch (ClassNotFoundException e) {
            log.error("Jacoco not enabled", e);
            throw new RuntimeException("Jacoco not enabled");
        } catch (Exception e) {
            log.error("Dump exec failed", e);
            throw new RuntimeException("Dump exec failed");
        }
    }

    /**
     * 将目录下所有文件打包为 ZIP，写入 outputStream。
     */
    public void zipDirectory(String name, Path sourceDir, OutputStream outputStream) throws IOException {
        try (ZipOutputStream zipOut = new ZipOutputStream(outputStream);
             Stream<Path> walkStream = Files.walk(sourceDir)) { // 同时关闭流
            walkStream.forEach(path -> {
                Path rel = sourceDir.relativize(path);
                String entryName = (name + "/" + rel.toString().replace(File.separatorChar, '/')
                        + (Files.isDirectory(path) ? "/" : "")).replace("//", "/");
                try {
                    zipOut.putNextEntry(new ZipEntry(entryName));
                    if (Files.isRegularFile(path)) {
                        Files.copy(path, zipOut);
                    }
                    zipOut.closeEntry();
                } catch (IOException e) {
                    log.error("An error occurred in packaging file : {}", path, e);
                }
            });
        }
    }

    private void extractSourceJar(String workingDir) {
        File sourceDirectory = new File(workingDir, "src/main/java");
        if (!sourceDirectory.exists()) {
            try {
                extractSourceJar(workingDir + "/km_deployer_service-1.0.0-sources.jar", workingDir + "/src/main/java");
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void extractSourceJar(String jarPath, String destDir) throws IOException {
        JarFile jarFile = new JarFile(jarPath);
        Enumeration<JarEntry> entries = jarFile.entries();

        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            File entryFile = new File(destDir, entry.getName());

            if (entry.isDirectory()) {
                entryFile.mkdirs();
            } else {
                // 确保父目录存在
                entryFile.getParentFile().mkdirs();

                try (InputStream in = jarFile.getInputStream(entry);
                     OutputStream out = new FileOutputStream(entryFile)) {

                    byte[] buffer = new byte[4096];
                    int bytesRead;

                    while ((bytesRead = in.read(buffer)) != -1) {
                        out.write(buffer, 0, bytesRead);
                    }
                }
            }
        }

        jarFile.close();
        log.info("Decompression completed: " + destDir);
    }

    private void execCmd(String coverageReportDir) throws InterruptedException, IOException {
        ProcessBuilder processBuilder = new ProcessBuilder()
            .command("java", "-jar",
                jacocoJarDir + "/jacococli.jar", "report", workingDir + "/platform/jacoco/jacoco.exec",
                "--classfiles", workingDir + "/km_deployer_service-classes/BOOT-INF/classes/com/digiwin/athena",
                "--sourcefiles", workingDir + "/src/main/java",
                "--html", workingDir + coverageReportDir)
            .redirectErrorStream(true);
        Process process = processBuilder.start();
        int exitCode = process.waitFor();
        if (exitCode == 0) {
            log.info("HTML report generation successfully!");
        } else {
            log.error("Report generation failed. exitCode: {}", exitCode);
        }
    }

    /**
     * 获取Jacoco exec 文件
     *
     * @return
     */
    @GetMapping("/jacoco.exec")
    public void getJacocoExec(HttpServletResponse response) throws IOException {
        dumpJacocoExec();

        Path execPath = Paths.get(workingDir, "platform", "jacoco", "jacoco.exec");
        // 使用缓冲流传输大文件
        try (OutputStream os = response.getOutputStream();
             InputStream is = Files.newInputStream(execPath)) {
            byte[] buffer = new byte[4096];
            int bytesRead;
            while ((bytesRead = is.read(buffer)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
            os.flush();
        }
    }

    /**
     * 获取Jacoco HTML报告
     * @param response
     * @throws Exception
     */
    @GetMapping("/jacoco-report.zip")
    public void getReport(HttpServletResponse response) throws Exception {
        dumpJacocoExec();
        extractSourceJar(workingDir);
        String coverageReportDir = "/coveragereport";
        execCmd(coverageReportDir);
        String filename = "km_deployer_service-jacoco-report_" + LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMddHHmmss")) + ".zip";
        // 设置响应头
        response.setContentType("application/octet-stream");
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        try (OutputStream os = response.getOutputStream()) {
            // 写入你的数据
            zipDirectory(filename, Paths.get(workingDir + coverageReportDir), os);
            os.flush();
        }
    }

}
