package com.digiwin.athena.km_deployer_service.domain.asa.model.mongodb;

import cn.hutool.core.collection.CollUtil;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Description 模型表
 * @Author cai chao
 * @Date 2024/8/2 10:25
 */
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
@Document(collection = "asa_assistant_model")
public class AssistantProjectModel {

    /**
     * 主键编号
     */
    @Id
    private String objectId;

    /**
     * 模型编号 【模型代号 助理代号_版本号】
     */
    private String modelCode;

    /**
     * 模型名称
     */
    private String modelName;

    /**
     * 项目编号
     */
    private String projectCode;

    /**
     * 模型语言 zh-hans|zh-hant|en，默认zh_hans
     */
    private String language = "zh_hans";

    /**
     * 模型类型 1：大模型；2：小模型；3：ES辅助
     */
    private List<Integer> modelTypes;

    /**
     * ChatGPT|GPT4_turbo|GPT4o
     */
    private String llmModel;

    /**
     * 每个意图对应的测试语料数量，默认20
     */
    private Integer testDataNum = 20;

    /**
     * 每个意图对应的训练语料数量，默认150
     */
    private Integer trainDataNum = 150;

    /**
     * 每个意图对应的None语料数量，默认150
     */
    private Integer noneDataNum = 150;

    /**
     * 每个NG语料对应的优化语料数量，默认3
     */
    private Integer trainOptDataNum = 3;

    /**
     * 基础版本 代表版本前置
     */
    private String baseVersion;

    /**
     * 评测版本
     */
    private String trainVersion;

    /**
     * 部署状态  0：未部署；1：已部署
     */
    private Integer deployStatus;

    /**
     * 应用状态  0：非应用；1：应用
     */
    private Integer applyStatus;

    /**
     * 模型与前一版本的差异，0：无变化，1:有变化
     */
    private Integer comparePre;

    /**
     * 模型状态 0-草稿中 1-不可用 2-评估中 3-可用
     */
    private Integer status;

    /**
     * 意图
     */
    private List<ModelIntent> modelIntents;

    /**
     * 提示词编号
     */
    private String llmPromptId;

    /**
     * 提示词内容
     */
    private String llmPrompt;

    /**
     * 准确率
     */
    private Estimate estimate;

    /**
     * 创建时间
     */
    private Date createTime;

    /**
     * 原来版本
     */
    private String originalVersion;

    /**
     * 原来版本
     */
    private String originalModelCode;

    /**
     * 助理介绍
     */
    private String introduction;

    @Data
    public static class ModelIntent {
        /**
         * 场景名称
         */
        private String intent;

        /**
         * 意图，初始意图对应场景名称
         */
        private RelaScene relaScene;

        /**
         * 场景描述
         */
        private String definition;

        /**
         * 意图类型  entry：入口意图
         */
        private String type;

        @Data
        public static class RelaScene {
            private String code;
            private String name;
        }
    }

    @Data
    public static class Estimate {
        /**
         * 模型准确率
         */
        private String accuracyRate;

        // 模型场景准确率列表
        private List<IntentAccuracyRate> intentAccuracyRate;

        /**
         * 大模型准确率
         */
        private String llmAccuracyRate;

        /**
         * 大模型场景准确率列表
         */
        private List<IntentAccuracyRate> llmIntentAccuracyRate;

        /**
         * 小模型准确率
         */
        private String nluAccuracyRate;

        /**
         * model_1准确率
         */
        private String model1AccuracyRate;

        /**
         * model_2准确率
         */
        private String model2AccuracyRate;

        /**
         * 小模型场景准确率列表
         */
        private List<IntentAccuracyRate> nluIntentAccuracyRate;

        /**
         * 大模型测试语料数
         */
        private int llmTestCount;

        /**
         * 小模型测试语料数
         */
        private int nluTestCount;

        @Data
        public static class IntentAccuracyRate {
            /**
             * 场景名称
             */
            private String intent;

            /**
             * 场景准确率
             */
            private String accuracyRate;
        }
    }

    /**
     * 提示词入参
     *
     * @return
     */
    public List<Map> constructIntentDefinition() {
        List<ModelIntent> modelIntents = this.getModelIntents();
        return CollUtil.emptyIfNull(modelIntents).stream().map(modelIntent -> {
            Map map = new HashMap();
            map.put("intent", modelIntent.getIntent());
            map.put("definition", modelIntent.getDefinition());
            map.put("create", "O");
            map.put("read", "X");
            map.put("update", "X");
            map.put("delete", "X");
            return map;
        }).collect(Collectors.toList());
    }
}
