package com.digiwin.athena.km_deployer_service.domain.neo4j;

import cn.hutool.core.collection.ListUtil;
import com.digiwin.athena.km_deployer_service.config.neo4j.Neo4jManager;
import org.neo4j.driver.Driver;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author liyuetao
 * @title: CqlMapper
 * @projectName athena_deployer
 * @description: TODO
 * @date 2022/7/2816:22
 */
public class CqlMapper {

    /**
     * @description: 更新租户版本
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql updateTenantVersion(List<String> tenantIdList, String version) {
        String cql = "match (tenant:TenantEntity) where tenant.tenantId in $tenantId and tenant.version<>$version set tenant.version = $version";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantIdList);
        param.put("version", version);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: 删除租户旧版本relation
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql deleteTenantOldRelation(List<String> tenantIdList, String application) {
        String cql = "match (tenant:TenantEntity)-[oldRelation]->(node) where tenant.tenantId in $tenantId and (node.nameSpace = $application or node.athena_namespace = $application) and not any(label in labels(node) WHERE label in ['AppEntity']) delete oldRelation";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantIdList);
        param.put("application", application);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: merge租户与应用AppEntity的relation
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAppEntityRelation(List<String> tenantIdList, String application) {
        String cql = "match(t:TenantEntity) where t.tenantId in $tenantId match(app:AppEntity{code:$application}) merge (t)-[:USE]->(app)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantIdList);
        param.put("application", application);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: 查询租户下的应用
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql queryTenantAppList(String tenantId) {
        String cql = "match (tenant:TenantEntity)-[:USE]->(app:AppEntity) where tenant.tenantId=$tenantId return tenant.tenantId as tenantId,tenant.version as version,app.code as appCode,app.namespace as appNamespace";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: merge租户与common类型的Action的relation
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static List<Cql> mergeTenantAndCommonActionRelation(String tenantId) {
        String cqlStr1 = "match (te:TenantEntity{tenantId:$tenantId}) match (commonAction:Action) where commonAction.nameSpace in['espCommon','common'] and commonAction.version=te.version and commonAction.tenantId is null merge (te)-[:ACTION]->(commonAction)";
        String cqlStr2 = "match (te:TenantEntity{tenantId:$tenantId}) match (commonAction:Action) where commonAction.nameSpace in['espCommon','common'] and commonAction.version=te.version and commonAction.tenantId=te.tenantId merge (te)-[:ACTION]->(commonAction)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        Cql cql1 = new Cql().setCql(cqlStr1)
                .setParams(param);
        Cql cql2 = new Cql().setCql(cqlStr2)
                .setParams(param);
        return ListUtil.toList(cql1, cql2);
    }

    /**
     * @description: merge租户与common类型的Task的relation
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAndCommonTaskRelation(String tenantId) {
        String cql = "match (te:TenantEntity{tenantId:$tenantId}) match (commonTask:Task) where commonTask.nameSpace ='common' and commonTask.version=te.version merge (te)-[:TASK]->(commonTask)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        return new Cql().setCql(cql)
                .setParams(param);
    }
    /**
     * @description: merge租户与common类型的Activity的relation
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAndCommonActivityRelation(String tenantId) {
        String cql = "match (te:TenantEntity{tenantId:$tenantId}) match (commonActivity:Activity) where commonActivity.nameSpace ='common' and commonActivity.version=te.version merge (te)-[:ACTIVITY]->(commonActivity)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: merge租户与Task的relation（排除租户级Task）
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAndTaskExcludeTenantTask(String application, String tenantId, Driver driver) {
        List<String> tenantTaskCodeList = getTenantTaskCodeList(application, tenantId, driver);
        String cql = "match (te:TenantEntity{tenantId:$tenantId}) match(task:Task) where (task.nameSpace = $nameSpace or task.athena_namespace = $nameSpace) and task.version=te.version and not task.code in $tenantTaskCodeList merge (te)-[:TASK]->(task)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantTaskCodeList", tenantTaskCodeList);
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: 查询租户定制的Task,现在没有neo4j的task节点，方法废弃
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static List<String> getTenantTaskCodeList(String application, String tenantId, Driver driver) {
        String cql = "match (n:Task) where n.tenantId = $tenantId and n.nameSpace = $application return n.code as code";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("application", application);
        List<Map<String, Object>> maps = new Neo4jManager(driver).ExecuteQuery(cql, param);
        List<String> taskCodeList = new ArrayList<>();
        for (Map<String, Object> map : maps) {
            taskCodeList.add(String.valueOf(map.get("code")));
        }
        return taskCodeList;
    }

    /**
     * @description: merge租户与Activity的relation（排除租户级Activity）
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static List<Cql> mergeTenantAndActivityExcludeTenantActivity(String application, String tenantId, Driver driver) {
        List<String> tenantActivityCodeList = getTenantActivityCodeList(application, tenantId, driver);
        String cqlStr1 = "match (te:TenantEntity{tenantId:$tenantId}) match(activity:Activity) where (activity.nameSpace = $nameSpace or activity.athena_namespace = $nameSpace) and activity.version=te.version and activity.inclusionTenant is null and not activity.code in $tenantActivityCodeList merge (te)-[:ACTIVITY]->(activity)";
        String cqlStr2 = "match (te:TenantEntity{tenantId:$tenantId}) match(activity:Activity) where (activity.nameSpace = $nameSpace or activity.athena_namespace = $nameSpace) and activity.version=te.version and activity.inclusionTenant is not null and te.tenantId in activity.inclusionTenant and not activity.code in $tenantActivityCodeList merge (te)-[:ACTIVITY]->(activity)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantActivityCodeList", tenantActivityCodeList);
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        Cql cql1 = new Cql().setCql(cqlStr1)
                .setParams(param);
        Cql cql2 = new Cql().setCql(cqlStr2)
                .setParams(param);
        return ListUtil.toList(cql1, cql2);
    }

    /**
     * @description: 查询租户定制的Activity
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    private static List<String> getTenantActivityCodeList(String application, String tenantId, Driver driver) {
        String cql = "match (n:Activity) where n.tenantId = $tenantId and n.nameSpace = $nameSpace return n.code as code";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        Iterable<Map<String, Object>> maps = new Neo4jManager(driver).ExecuteQuery(cql, param);
        List<String> activityCodeList = new ArrayList<>();
        for (Map<String, Object> map : maps) {
            activityCodeList.add(String.valueOf(map.get("code")));
        }
        return activityCodeList;
    }

    /**
     * @description: merge租户与MonitorRule的relation（排除租户级MonitorRule）
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static List<Cql> mergeTenantAndActivityExcludeTenantMonitorRule(String application, String tenantId, Driver driver) {
        List<String> tenantMonitorRuleCodeList = getTenantMonitorRuleCodeList(application, tenantId, driver);
        String cqlStr1 = "match(t:TenantEntity) where t.tenantId=$tenantId match(monitorRule:MonitorRule) where monitorRule.athena_namespace=$nameSpace and monitorRule.version=t.version and monitorRule.inclusionTenant is null and not monitorRule.code in $tenantMonitorRuleCodeList merge (t)-[:USE]->(monitorRule)";
        String cqlStr2 = "match(t:TenantEntity) where t.tenantId=$tenantId match(monitorRule:MonitorRule) where monitorRule.athena_namespace=$nameSpace and monitorRule.version=t.version and monitorRule.inclusionTenant is not null and t.tenantId in monitorRule.inclusionTenant and not monitorRule.code in $tenantMonitorRuleCodeList merge (t)-[:USE]->(monitorRule)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantMonitorRuleCodeList", tenantMonitorRuleCodeList);
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        Cql cql1 = new Cql().setCql(cqlStr1)
                .setParams(param);
        Cql cql2 = new Cql().setCql(cqlStr2)
                .setParams(param);
        return ListUtil.toList(cql1, cql2);
    }

    /**
     * @description: 查询租户定制的MonitorRule
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    private static List<String> getTenantMonitorRuleCodeList(String application, String tenantId, Driver driver) {
        String cql = "match (n:MonitorRule) where n.tenantId = $tenantId and n.nameSpace = $nameSpace return n.code as code";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        Iterable<Map<String, Object>> maps = new Neo4jManager(driver).ExecuteQuery(cql, param);
        List<String> activityCodeList = new ArrayList<>();
        for (Map<String, Object> map : maps) {
            activityCodeList.add(String.valueOf(map.get("code")));
        }
        return activityCodeList;
    }

    /**
     * @description: merge租户与Mechanism的relation（排除租户级Mechanism）
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAndMechanismExcludeTenantMechanism(String application, String tenantId, Driver driver) {
        List<String> tenantMechanismKeyList = getTenantMechanismKeyList(application, tenantId, driver);
        String cql = "match (te:TenantEntity{tenantId:$tenantId}) match(me:Mechanism) where (me.nameSpace = $nameSpace or me.athena_namespace = $nameSpace) and me.version=te.version and not me.key in $tenantMechanismKeyList merge (te)-[:MECHANISM]->(me)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantMechanismKeyList", tenantMechanismKeyList);
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: 获取租户定制的Mechanism key
     * @author liyuetao
     * @date 2022/7/23 10:25
     */
    public static List<String> getTenantMechanismKeyList(String application, String tenantId, Driver driver) {
        String cql = "match (n:Mechanism) where n.tenantId = $tenantId and n.nameSpace = $application return n.key as code";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("application", application);
        Iterable<Map<String, Object>> maps = new Neo4jManager(driver).ExecuteQuery(cql, param);
        List<String> mechanismCodeList = new ArrayList<>();
        for (Map<String, Object> map : maps) {
            mechanismCodeList.add(String.valueOf(map.get("code")));
        }
        return mechanismCodeList;
    }

    /**
     * @description: merge租户与Action的relation（排除租户级Action）
     * @author liyuetao
     * @date 2022/7/28 16:25
     */
    public static Cql mergeTenantAndActionExcludeTenantAction(String application, String tenantId, Driver driver) {
        List<String> tenantActionIdList = getTenantActionIdList(application, tenantId, driver);
        String cql = "match (te:TenantEntity{tenantId:$tenantId}) match(action:Action) where (action.nameSpace = $nameSpace or action.athena_namespace = $nameSpace) and action.version=te.version and not action.actionId in $tenantActionIdList merge (te)-[:ACTION]->(action)";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantActionIdList", tenantActionIdList);
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        return new Cql().setCql(cql)
                .setParams(param);
    }

    /**
     * @description: 获取租户定制的Action
     * @author liyuetao
     * @date 2022/7/23 10:25
     */
    public static List<String> getTenantActionIdList(String application, String tenantId, Driver driver) {
        String cql = "match (n:Action) where n.tenantId = $tenantId and n.nameSpace = $nameSpace return n.actionId as code";
        HashMap<String, Object> param = new HashMap<>();
        param.put("tenantId", tenantId);
        param.put("nameSpace", application);
        Iterable<Map<String, Object>> maps = new Neo4jManager(driver).ExecuteQuery(cql, param);
        List<String> actionIdList = new ArrayList<>();
        for (Map<String, Object> map : maps) {
            actionIdList.add(String.valueOf(map.get("code")));
        }
        return actionIdList;
    }
}
